#' Calculates raster points for plotting raster in hazard chart with counts or percentages of avalanche problems.
#'
#' Calculates raster points for plotting raster in hazard chart with counts or percentages.
#' @param AvProblems DF with avalanche problems
#' @param Shape For specifying the shape of the blob. Values can be 'ellipse' or 'square' (default).
#' @param NumPoints Number of point per quater section of ellipse. Default value is 50.
#' @param OutputRaster Switch for specifying whether output should be dataframe or raster. Default value is F.
#' @param Percent Switch whether raster values should be percentages instead of counts. Default value is F.
#' @param SuppressWarnings Switch for suppressing warning message about application of is.na to S4 object. Default is TRUE.
#'
#' @examples
#' require(SarpBulletinToolsPrivate)
#' require(SarpBulletinTools)
#'
#' Bulletins_BYK <- getBulletins(Agency="PkCan", Seasons=2015, Regions="Banff, Yoho and Kootenay")
#' AlpAvProb_BYK <- Bulletins_BYK$AvProblems$Alp
#'
#' RasterPoints <- calculateHzdChartCountRasterPoints(AlpAvProb_BYK[AlpAvProb_BYK$CHARACTER=="Wind Slabs",], Shape = "Square")
#' RasterCount <- calculateHzdChartCountRasterPoints(AlpAvProb_BYK[AlpAvProb_BYK$CHARACTER=="Wind Slabs",], Shape = "Square", OutputRaster=T)
#' RasterPerc <- calculateHzdChartCountRasterPoints(AlpAvProb_BYK[AlpAvProb_BYK$CHARACTER=="Wind Slabs",], Shape = "Square", OutputRaster=T, Percent=T)
#'
#' @export

calculateHzdChartCountRasterPoints <- function(AvProblems, Shape='square', NumPoints=50, OutputRaster=F, Percent=F, IDCol="PROBLEM_ID", SuppressWarnings=T) {

  Shape <- toupper(Shape)

  ## Check for necessary packages
  if (!isPackageInstalled("sp")) {stop("This function requires package 'sp' to be installed!")}
  if (!isPackageInstalled("raster")) {stop("This function requires package 'raster' to be installed!")}

  ## Convert avalanche problems to SPDF
  AvProbSPDF <- convertAvProbToSPDF(AvProblems=AvProblems, Shape=Shape, NumPoints=NumPoints, IDCol=IDCol)

  if (SuppressWarnings) {options(warn=-1)}

  ## Create raster
  if (Shape=="SQUARE") {
    Raster <- raster::raster(ncol=8, nrow=8, xmn=1, xmx=9, ymn=1, ymx=9)
    RasterCoord <- raster::xyFromCell(Raster, cell=c(1:64))
  } else if (Shape=="ELLIPSE") {
    Raster <- raster::raster(ncol=40, nrow=40, xmn=0, xmx=8, ymn=0, ymx=8)
    RasterCoord <- raster::xyFromCell(Raster, cell=c(1:1600))
  }

  RasterPoints <- data.frame(x=RasterCoord[,1],
                             y=RasterCoord[,2],
                             count=0,
                             percent=0)

  ## Counting of blobs in cell
  if(!is.na(AvProbSPDF)) {

    for (Index_AvProb in 1:nrow(AvProbSPDF)) {
      for (Index_Points in 1:nrow(RasterPoints)) {
        RasterPoints$count[Index_Points] <- ifelse(sp::point.in.polygon(RasterPoints$x[Index_Points],
                                                                        RasterPoints$y[Index_Points],
                                                                        AvProbSPDF@polygons[[Index_AvProb]]@Polygons[[1]]@coords[,1],
                                                                        AvProbSPDF@polygons[[Index_AvProb]]@Polygons[[1]]@coords[,2])==1,
                                                   RasterPoints$count[Index_Points]+1, RasterPoints$count[Index_Points])
      }
    }

    ## Calculating percentage
    RasterPoints$percent <- RasterPoints$count/nrow(AvProbSPDF)*100

    ## Returning RasterPoints dataframe
    if (OutputRaster==F) {
      return(list(NumAvProb=nrow(AvProbSPDF),
                  RasterPoints=RasterPoints,
                  Raster=NULL))

    ## Returning raster
    } else {
      if (Percent==F) {
        raster::values(Raster) <- RasterPoints$count
      } else {
        raster::values(Raster) <- RasterPoints$percent
      }
      return(list(NumAvProb=nrow(AvProbSPDF),
                  RasterPoints=NULL,
                  Raster=Raster))
    }

  ## Returning null in case no avalanche problems SPDF
  } else {
    return(NULL)
  }

  if (SuppressWarnings) {options(warn=0)}

}
