#' Creates season plot of avalanche bulletins
#'
#' Creates season plot of avalanche bulletins
#' @param Agency Name of agency (PkCan or AvCan)
#' @param Season Season to be plotted.
#' @param Region Name of bulletin region
#' @param OutputFolder Folder path for output PNG files
#' @param BestDBSourceInt Dataframe with information about best data source DB to be used for query. Default is the BestDBSource dataframe that is included in the package.
#' @param User Name of user accessing the database. This name is used to record database interactions and to potentially access local resources (i.e., local folder JDBC driver)
#' @param Verbose Switch for printing SQL queries. Default value is FALSE.
#' @param SuppressDBWarnings Switch for turning warnings off. Default value is TRUE.
#'
#' @export

createBulletinSeasonPlot <- function (Agency, Season, Region, OutputFolder, BestDBSourceInt=BestDBSource, User="", Verbose=F, SuppressDBWarnings=T) {

  ## DeBugging
  # Agency <- 'AvCan'
  # Season <- 2010
  # Region <- 'North Columbia'
  # OutputFolder <- '/Users/brets/Desktop/'
  
  ## User feedback
  print(paste (Sys.time(), "- Processing -", Region, "-", Season))
  # start time
  TimeStart <- Sys.time()

  # FUNCTIONS
  # *********

  # Plotting of avalanche problems
  plotAvProblem <- function(DateArray, ProbArray, YAxis, Color, PercXAxis, ElvSum) {

    # Plot points
    points(DateArray[ProbArray==1], ProbArray[ProbArray==1]*YAxis, pch=16, cex=1.25, col=Color)

    # Write percentage
    rect(PercXAxis-4, YAxis-0.4, PercXAxis+4, YAxis+0.4, col="white", border=NA)
    text(PercXAxis, YAxis, paste(100*round(sum(ProbArray[ProbArray==1], na.rm=T)/ElvSum,2), "%"), cex=0.75)

  }

  # Print Num of Problems percentages
  printNumProbPercentages <- function(NumProbArray, PercXAxis, ElvSum) {

    for (Num in 0:4) {
      rect(PercXAxis-4, Num-0.45, PercXAxis+4, Num+0.45, col="white", border=NA)
      text(PercXAxis, Num, paste(100*round(sum(NumProbArray==Num, na.rm=T)/ElvSum,2), "%"), cex=0.75)
    }
  }

  # Write number and percentage of problems to Tbl_Output
  writeAvProblemToTblOutput <- function(Tbl_Output, Index, ProbArray, Problem, ElvSum) {

    # Index <- OutputIndexMax-2
    # ProbArray <- Tbl_AvProblems$AlpStorm
    # Problem <- "Storm"

    Tbl_Output[Index, paste(Problem, "C", sep="")] <- sum(ProbArray[ProbArray==1], na.rm=T)
    Tbl_Output[Index, paste(Problem, "P", sep="")] <- round(sum(ProbArray[ProbArray==1], na.rm=T)/ElvSum,2)
    return(Tbl_Output)

  }

  # Write number and percentages of number of problems to Tbl_Output
  writeNumProbToTblOutput <- function(Tbl_Output, Index, NumProbArray, ElvSum) {

    for (Num in 0:4) {
      Tbl_Output[Index, paste("Num", Num, "C", sep="")] <- sum(NumProbArray==Num, na.rm=T)
      Tbl_Output[Index, paste("Num", Num, "P", sep="")] <- round(sum(NumProbArray==Num, na.rm=T)/ElvSum,2)
    }
    return(Tbl_Output)

  }

  # Plotting
  # ********

  ## Get data
  Bulletins <- getBulletins(Agency, Seasons=Season, Regions=Region)

  # Query for last bulletin of multiple have been publisehd for same date issued
  Tbl_AvProblems <- createBulletinSeasonPlotData(Bulletins)
  Tbl_AvProblems$Date <- as.Date(Tbl_AvProblems$Date, format="%Y-%m-%d")
  #Tbl_AvProblems$DateIssued <- as.Date(Tbl_AvProblems$DateIssued, format="%Y-%m-%d")
  # create vector that has number of bulletins for Alp, Tl, Btl
  NumBul <- c(length(unique(Bulletins$AvProblems$Alp$BULLETIN_ID)), 
              length(unique(Bulletins$AvProblems$Tl$BULLETIN_ID)), 
              length(unique(Bulletins$AvProblems$Btl$BULLETIN_ID)))

  # Create output file
  if (exists("Tbl_Output")) {
    Tbl_Output <- rbind(Tbl_Output, data.frame(Season=Season,
                                               Region=Region,
                                               Elev=c("Alp", "Tln", "Btl"),
                                               NumBull=NumBul,
                                               PersPlC=NA, DPersC=NA, PersC=NA, StormC=NA, WindC=NA, CornC=NA, WetC=NA, LWetC=NA, LDryC=NA,
                                               PersPlP=NA, DPersP=NA, PersP=NA, StormP=NA, WindP=NA, CornP=NA, WetP=NA, LWetP=NA, LDryP=NA,
                                               Num0C=NA, Num1C=NA, Num2C=NA, Num3C=NA, Num4C=NA,
                                               Num0P=NA, Num1P=NA, Num2P=NA, Num3P=NA, Num4P=NA,
                                               LowC=NA, ModC=NA, ConsC=NA, HighC=NA, ExtC=NA, NoRC=NA,
                                               LowP=NA, ModP=NA, ConsP=NA, HighP=NA, ExtP=NA, NoRP=NA))
  } else {
    Tbl_Output <- data.frame(Season=Season,
                             Region=Region,
                             Elev=c("Alp", "Tln", "Btl"),
                             NumBull=NumBul,
                             PersPlC=NA, DPersC=NA, PersC=NA, StormC=NA, WindC=NA, CornC=NA, WetC=NA, LWetC=NA, LDryC=NA,
                             PersPlP=NA, DPersP=NA, PersP=NA, StormP=NA, WindP=NA, CornP=NA, WetP=NA, LWetP=NA, LDryP=NA,
                             Num0C=NA, Num1C=NA, Num2C=NA, Num3C=NA, Num4C=NA,
                             Num0P=NA, Num1P=NA, Num2P=NA, Num3P=NA, Num4P=NA,
                             LowC=NA, ModC=NA, ConsC=NA, HighC=NA, ExtC=NA, NoRC=NA,
                             LowP=NA, ModP=NA, ConsP=NA, HighP=NA, ExtP=NA, NoRP=NA)
  }
  OutputIndexMax <- nrow(Tbl_Output)

  # Output device
  ChartWidth <- 16
  ChartHeight <- 16
  FigName <- paste0(OutputFolder, Region, "-", Season, "-AvProblems")
  png(paste0(FigName, ".png"), width=ChartWidth, height=ChartHeight, units="cm", pointsize=8, bg="white", res=600)

  # par defaults and chart size and margins
  def.par <- par(no.readonly = TRUE) # save default, for resetting...
  nf <- layout(matrix(c(1:6), 6, 1, byrow=T), width=c(lcm(15),lcm(15),lcm(15),lcm(15),lcm(15),lcm(15)), height=c(lcm(3.5),lcm(1.2),lcm(3),lcm(1.2),lcm(3),lcm(1.7)))

  MarginsAlp  <- c(0,   0.5,0.25,0.25)
  MarginsAlp2 <- c(0.02,0.5,0,   0.25)
  MarginsTln  <- c(0,   0.5,0,   0.25)
  MarginsTln2 <- c(0.02,0.5,0,   0.25)
  MarginsBtl  <- c(0,   0.5,0,   0.25)
  MarginsBtl2 <- c(0.25,0.5,0,   0.25)

  # Plot for Alpine
  # ***************
  par(mai=MarginsAlp)

  PlotTitle <- paste(Region, " - Winter Season ", Season, sep="")
  plot (Tbl_AvProblems$Date, Tbl_AvProblems$AlpStorm, type="n",
        ylim=c(0.5,10.5), yaxt="n", ylab="",
        xlim=c(min(Tbl_AvProblems$Date), max(Tbl_AvProblems$Date)), xlab="", xaxt="n",
        main=PlotTitle)

  AvProbYAxis <- c("Pers+S", "DPersS", "PersS", "StormS","WindS", "Corn", "WetS", "LWet", "LDry")
  axis(2, at=c(1:9), labels=AvProbYAxis, las=1)

  mtext("Alpine",2, line=4.7, cex=.75)
  mtext(paste("(n=", Tbl_Output$NumBull[1], ")", sep=""),3,line=1, adj=1, padj=1, cex=.75)

  # Grid lines
  abline(h=c(1:9), col="grey", lty=2, lwd=0.5)
  abline(v=seq(min(Tbl_AvProblems$Date), max(Tbl_AvProblems$Date), by="weeks"), col="grey", lty=2, lwd=0.5)

  # DangerRating
  for (DateIndex in 1:nrow(Tbl_AvProblems)) {
    if(!is.na(Tbl_AvProblems$bulletinID[DateIndex]) & nchar(as.character(Tbl_AvProblems$bulletinID[DateIndex]))>0) {
      rect(Tbl_AvProblems$Date[DateIndex]-0.5, 9.5, Tbl_AvProblems$Date[DateIndex]+0.5, 10.5, col=getHzdColor(Tbl_AvProblems$AlpDR[DateIndex]), border=NA)
    }
  }

  # Avalanche Problems
  plotAvProblem(Tbl_AvProblems$Date, Tbl_AvProblems$AlpPersPl, 1, "#575757", Tbl_AvProblems$Date[1], Tbl_Output$NumBull[1])
  plotAvProblem(Tbl_AvProblems$Date, Tbl_AvProblems$AlpDPers,  2, getAvCharColor("Deep Persistent Slabs"), Tbl_AvProblems$Date[1], Tbl_Output$NumBull[1])
  plotAvProblem(Tbl_AvProblems$Date, Tbl_AvProblems$AlpPers,   3, getAvCharColor("Persistent Slabs"), Tbl_AvProblems$Date[1], Tbl_Output$NumBull[1])
  plotAvProblem(Tbl_AvProblems$Date, Tbl_AvProblems$AlpStorm,  4, getAvCharColor("Storm Slabs"), Tbl_AvProblems$Date[1], Tbl_Output$NumBull[1])
  plotAvProblem(Tbl_AvProblems$Date, Tbl_AvProblems$AlpWind,   5, getAvCharColor("Wind Slabs"), Tbl_AvProblems$Date[1], Tbl_Output$NumBull[1])
  plotAvProblem(Tbl_AvProblems$Date, Tbl_AvProblems$AlpCorn,   6, getAvCharColor("Cornices"), Tbl_AvProblems$Date[1], Tbl_Output$NumBull[1])
  plotAvProblem(Tbl_AvProblems$Date, Tbl_AvProblems$AlpWet,    7, getAvCharColor("Wet Slabs"), Tbl_AvProblems$Date[1], Tbl_Output$NumBull[1])
  plotAvProblem(Tbl_AvProblems$Date, Tbl_AvProblems$AlpLWet,   8, getAvCharColor("Loose Wet"), Tbl_AvProblems$Date[1], Tbl_Output$NumBull[1])
  plotAvProblem(Tbl_AvProblems$Date, Tbl_AvProblems$AlpLDry,   9, getAvCharColor("Loose Dry"), Tbl_AvProblems$Date[1], Tbl_Output$NumBull[1])

  # Writing number of avalanche problems to Tbl_Output
  Tbl_Output <- writeAvProblemToTblOutput(Tbl_Output, OutputIndexMax-2, Tbl_AvProblems$AlpPersPl, "PersPl", Tbl_Output$NumBull[1])
  Tbl_Output <- writeAvProblemToTblOutput(Tbl_Output, OutputIndexMax-2, Tbl_AvProblems$AlpDPers,  "DPers", Tbl_Output$NumBull[1])
  Tbl_Output <- writeAvProblemToTblOutput(Tbl_Output, OutputIndexMax-2, Tbl_AvProblems$AlpPers,   "Pers", Tbl_Output$NumBull[1])
  Tbl_Output <- writeAvProblemToTblOutput(Tbl_Output, OutputIndexMax-2, Tbl_AvProblems$AlpStorm,  "Storm", Tbl_Output$NumBull[1])
  Tbl_Output <- writeAvProblemToTblOutput(Tbl_Output, OutputIndexMax-2, Tbl_AvProblems$AlpWind,   "Wind", Tbl_Output$NumBull[1])
  Tbl_Output <- writeAvProblemToTblOutput(Tbl_Output, OutputIndexMax-2, Tbl_AvProblems$AlpCorn,   "Corn", Tbl_Output$NumBull[1])
  Tbl_Output <- writeAvProblemToTblOutput(Tbl_Output, OutputIndexMax-2, Tbl_AvProblems$AlpWet,    "Wet", Tbl_Output$NumBull[1])
  Tbl_Output <- writeAvProblemToTblOutput(Tbl_Output, OutputIndexMax-2, Tbl_AvProblems$AlpLWet,   "LWet", Tbl_Output$NumBull[1])
  Tbl_Output <- writeAvProblemToTblOutput(Tbl_Output, OutputIndexMax-2, Tbl_AvProblems$AlpLDry,   "LDry", Tbl_Output$NumBull[1])

  # Calculation of danger rating statistics
  Tbl_Output$LowC[OutputIndexMax-2]  <- sum(Tbl_AvProblems$AlpDR=="1 - LOW"          || Tbl_AvProblems$AlpDR=="Low", na.rm=T)
  Tbl_Output$ModC[OutputIndexMax-2]  <- sum(Tbl_AvProblems$AlpDR=="2 - MODERATE"     || Tbl_AvProblems$AlpDR=="Moderate", na.rm=T)
  Tbl_Output$ConsC[OutputIndexMax-2] <- sum(Tbl_AvProblems$AlpDR=="3 - CONSIDERABLE" || Tbl_AvProblems$AlpDR=="Considerable", na.rm=T)
  Tbl_Output$HighC[OutputIndexMax-2] <- sum(Tbl_AvProblems$AlpDR=="4 - HIGH"         || Tbl_AvProblems$AlpDR=="High", na.rm=T)
  Tbl_Output$ExtC[OutputIndexMax-2]  <- sum(Tbl_AvProblems$AlpDR=="5 - EXTREME"      || Tbl_AvProblems$AlpDR=="Extreme", na.rm=T)

  # Number of avalanche problems chart
  par(mai=MarginsAlp2)

  Tbl_AvProblems$AlpNumProb <- Tbl_AvProblems$AlpDPers + Tbl_AvProblems$AlpPers + Tbl_AvProblems$AlpStorm + Tbl_AvProblems$AlpWind +
                               Tbl_AvProblems$AlpCorn + Tbl_AvProblems$AlpWet + Tbl_AvProblems$AlpLWet + Tbl_AvProblems$AlpLDry

  plot (Tbl_AvProblems$Date, Tbl_AvProblems$AlpNumProb, type="n",
        ylim=c(-0.5,4.5), yaxt="n", ylab="",
        xlim=c(min(Tbl_AvProblems$Date), max(Tbl_AvProblems$Date)), xlab="", xaxt="n")

  # Grid lines
  abline(h=c(0:4), col="grey", lty=2, lwd=0.5)
  abline(v=seq(min(Tbl_AvProblems$Date), max(Tbl_AvProblems$Date), by="weeks"), col="grey", lty=2, lwd=0.5)

  points(Tbl_AvProblems$Date, Tbl_AvProblems$AlpNumProb, type="l")
  points(Tbl_AvProblems$Date, Tbl_AvProblems$AlpNumProb, pch=8, cex=0.5)

  axis(4, at=c(0:4), labels=c(0:4), las=1)
  mtext("Num. Prob.", 4, line=2.5, cex=.75)

  # Writing number of number of avalanche problems to chart and Tbl_Output
  printNumProbPercentages(Tbl_AvProblems$AlpNumProb, max(Tbl_AvProblems$Date), Tbl_Output$NumBull[1])
  Tbl_Output <- writeNumProbToTblOutput(Tbl_Output, OutputIndexMax-2, Tbl_AvProblems$AlpNumProb, Tbl_Output$NumBull[1])


  # Plot for Treeline
  # *****************
  par(mai=MarginsTln)

  plot (Tbl_AvProblems$Date, Tbl_AvProblems$TlnStorm, type="n",
        ylim=c(0.5,10.5), yaxt="n", ylab="",
        xlim=c(min(Tbl_AvProblems$Date), max(Tbl_AvProblems$Date)), xlab="", xaxt="n")

  axis(2, at=c(1:9), labels=AvProbYAxis, las=1)

  mtext("Treeline",2, line=4.7, cex=.75)

  # Grid lines
  abline(h=c(1:9), col="grey", lty=2, lwd=0.5)
  abline(v=seq(min(Tbl_AvProblems$Date), max(Tbl_AvProblems$Date), by="weeks"), col="grey", lty=2, lwd=0.5)

  # DangerRating
  for (DateIndex in 1:nrow(Tbl_AvProblems)) {
    if(!is.na(Tbl_AvProblems$bulletinID[DateIndex]) & nchar(as.character(Tbl_AvProblems$bulletinID[DateIndex]))>0) {
      rect(Tbl_AvProblems$Date[DateIndex]-0.5, 9.5, Tbl_AvProblems$Date[DateIndex]+0.5, 10.5, col=getHzdColor(Tbl_AvProblems$TlnDR[DateIndex]), border=NA)
    }
  }

  # Avalanche Problems
  plotAvProblem(Tbl_AvProblems$Date, Tbl_AvProblems$TlnPersPl, 1, "#575757", Tbl_AvProblems$Date[1], Tbl_Output$NumBull[2])
  plotAvProblem(Tbl_AvProblems$Date, Tbl_AvProblems$TlnDPers, 2, getAvCharColor("Deep Persistent Slabs"), Tbl_AvProblems$Date[1], Tbl_Output$NumBull[2])
  plotAvProblem(Tbl_AvProblems$Date, Tbl_AvProblems$TlnPers,  3, getAvCharColor("Persistent Slabs"), Tbl_AvProblems$Date[1], Tbl_Output$NumBull[2])
  plotAvProblem(Tbl_AvProblems$Date, Tbl_AvProblems$TlnStorm, 4, getAvCharColor("Storm Slabs"), Tbl_AvProblems$Date[1], Tbl_Output$NumBull[2])
  plotAvProblem(Tbl_AvProblems$Date, Tbl_AvProblems$TlnWind,  5, getAvCharColor("Wind Slabs"), Tbl_AvProblems$Date[1], Tbl_Output$NumBull[2])
  plotAvProblem(Tbl_AvProblems$Date, Tbl_AvProblems$TlnCorn,  6, getAvCharColor("Cornices"), Tbl_AvProblems$Date[1], Tbl_Output$NumBull[2])
  plotAvProblem(Tbl_AvProblems$Date, Tbl_AvProblems$TlnWet,   7, getAvCharColor("Wet Slabs"), Tbl_AvProblems$Date[1], Tbl_Output$NumBull[2])
  plotAvProblem(Tbl_AvProblems$Date, Tbl_AvProblems$TlnLWet,  8, getAvCharColor("Loose Wet"), Tbl_AvProblems$Date[1], Tbl_Output$NumBull[2])
  plotAvProblem(Tbl_AvProblems$Date, Tbl_AvProblems$TlnLDry,  9, getAvCharColor("Loose Dry"), Tbl_AvProblems$Date[1], Tbl_Output$NumBull[2])

  # Writing number of avalanche problems to Tbl_Output
  Tbl_Output <- writeAvProblemToTblOutput(Tbl_Output, OutputIndexMax-1, Tbl_AvProblems$TlnPersPl, "PersPl", Tbl_Output$NumBull[2])
  Tbl_Output <- writeAvProblemToTblOutput(Tbl_Output, OutputIndexMax-1, Tbl_AvProblems$TlnDPers,  "DPers", Tbl_Output$NumBull[2])
  Tbl_Output <- writeAvProblemToTblOutput(Tbl_Output, OutputIndexMax-1, Tbl_AvProblems$TlnPers,   "Pers", Tbl_Output$NumBull[2])
  Tbl_Output <- writeAvProblemToTblOutput(Tbl_Output, OutputIndexMax-1, Tbl_AvProblems$TlnStorm,  "Storm", Tbl_Output$NumBull[2])
  Tbl_Output <- writeAvProblemToTblOutput(Tbl_Output, OutputIndexMax-1, Tbl_AvProblems$TlnWind,   "Wind", Tbl_Output$NumBull[2])
  Tbl_Output <- writeAvProblemToTblOutput(Tbl_Output, OutputIndexMax-1, Tbl_AvProblems$TlnCorn,   "Corn", Tbl_Output$NumBull[2])
  Tbl_Output <- writeAvProblemToTblOutput(Tbl_Output, OutputIndexMax-1, Tbl_AvProblems$TlnWet,    "Wet", Tbl_Output$NumBull[2])
  Tbl_Output <- writeAvProblemToTblOutput(Tbl_Output, OutputIndexMax-1, Tbl_AvProblems$TlnLWet,   "LWet", Tbl_Output$NumBull[2])
  Tbl_Output <- writeAvProblemToTblOutput(Tbl_Output, OutputIndexMax-1, Tbl_AvProblems$TlnLDry,   "LDry", Tbl_Output$NumBull[2])

  # Calculation of danger rating statistics
  Tbl_Output$LowC[OutputIndexMax-1]  <- sum(Tbl_AvProblems$TlnDR=="1 - LOW"          || Tbl_AvProblems$TlnDR=="Low", na.rm=T)
  Tbl_Output$ModC[OutputIndexMax-1]  <- sum(Tbl_AvProblems$TlnDR=="2 - MODERATE"     || Tbl_AvProblems$TlnDR=="Moderate", na.rm=T)
  Tbl_Output$ConsC[OutputIndexMax-1] <- sum(Tbl_AvProblems$TlnDR=="3 - CONSIDERABLE" || Tbl_AvProblems$TlnDR=="Considerable", na.rm=T)
  Tbl_Output$HighC[OutputIndexMax-1] <- sum(Tbl_AvProblems$TlnDR=="4 - HIGH"         || Tbl_AvProblems$TlnDR=="High", na.rm=T)
  Tbl_Output$ExtC[OutputIndexMax-1]  <- sum(Tbl_AvProblems$TlnDR=="5 - EXTREME"      || Tbl_AvProblems$TlnDR=="Extreme", na.rm=T)

  # Number of avalanche problems chart
  par(mai=MarginsTln2)

  Tbl_AvProblems$TlnNumProb <- Tbl_AvProblems$TlnDPers + Tbl_AvProblems$TlnPers + Tbl_AvProblems$TlnStorm + Tbl_AvProblems$TlnWind +
    Tbl_AvProblems$TlnCorn + Tbl_AvProblems$TlnWet + Tbl_AvProblems$TlnLWet + Tbl_AvProblems$TlnLDry

  plot (Tbl_AvProblems$Date, Tbl_AvProblems$TlnNumProb, type="n",
        ylim=c(-0.5, 4.5), yaxt="n", ylab="",
        xlim=c(min(Tbl_AvProblems$Date), max(Tbl_AvProblems$Date)), xlab="", xaxt="n")

  # Grid lines
  abline(h=c(0:4), col="grey", lty=2, lwd=0.5)
  abline(v=seq(min(Tbl_AvProblems$Date), max(Tbl_AvProblems$Date), by="weeks"), col="grey", lty=2, lwd=0.5)

  points(Tbl_AvProblems$Date, Tbl_AvProblems$TlnNumProb, type="l")
  points(Tbl_AvProblems$Date, Tbl_AvProblems$TlnNumProb, pch=8, cex=0.5)

  axis(4, at=c(0:4), labels=c(0:4), las=1)
  mtext("Num. Prob.", 4, line=2.5, cex=.75)

  # Writing number of number of avalanche problems to chart and to Tbl_Output
  printNumProbPercentages(Tbl_AvProblems$TlnNumProb, max(Tbl_AvProblems$Date), Tbl_Output$NumBull[2])
  Tbl_Output <- writeNumProbToTblOutput(Tbl_Output, OutputIndexMax-1, Tbl_AvProblems$TlnNumProb, Tbl_Output$NumBull[2])


  # Plot for below Treeline
  # ***********************
  par(mai=MarginsBtl)

  plot (Tbl_AvProblems$Date, Tbl_AvProblems$BtlStorm, type="n",
        ylim=c(0.5,10.5), yaxt="n", ylab="",
        xlim=c(min(Tbl_AvProblems$Date), max(Tbl_AvProblems$Date)), xlab="", xaxt="n")

  axis(2, at=c(1:9), labels=AvProbYAxis, las=1)

  mtext("Below Treeline",2, line=4.7, cex=.75)

  # Grid lines
  abline(h=c(1:9), col="grey", lty=2, lwd=0.5)
  abline(v=seq(min(Tbl_AvProblems$Date), max(Tbl_AvProblems$Date), by="weeks"), col="grey", lty=2, lwd=0.5)

  # DangerRating
  for (DateIndex in 1:nrow(Tbl_AvProblems)) {
    if(!is.na(Tbl_AvProblems$bulletinID[DateIndex]) & nchar(as.character(Tbl_AvProblems$bulletinID[DateIndex]))>0) {
      rect(Tbl_AvProblems$Date[DateIndex]-0.5, 9.5, Tbl_AvProblems$Date[DateIndex]+0.5, 10.5, col=getHzdColor(Tbl_AvProblems$BtlDR[DateIndex]), border=NA)
    }
  }

  # Avalanche Problems
  plotAvProblem(Tbl_AvProblems$Date, Tbl_AvProblems$BtlPersPl, 1, "#575757", Tbl_AvProblems$Date[1], Tbl_Output$NumBull[3])
  plotAvProblem(Tbl_AvProblems$Date, Tbl_AvProblems$BtlDPers, 2, getAvCharColor("Deep Persistent Slabs"), Tbl_AvProblems$Date[1], Tbl_Output$NumBull[3])
  plotAvProblem(Tbl_AvProblems$Date, Tbl_AvProblems$BtlPers,  3, getAvCharColor("Persistent Slabs"), Tbl_AvProblems$Date[1], Tbl_Output$NumBull[3])
  plotAvProblem(Tbl_AvProblems$Date, Tbl_AvProblems$BtlStorm, 4, getAvCharColor("Storm Slabs"), Tbl_AvProblems$Date[1], Tbl_Output$NumBull[3])
  plotAvProblem(Tbl_AvProblems$Date, Tbl_AvProblems$BtlWind,  5, getAvCharColor("Wind Slabs"), Tbl_AvProblems$Date[1], Tbl_Output$NumBull[3])
  plotAvProblem(Tbl_AvProblems$Date, Tbl_AvProblems$BtlCorn,  6, getAvCharColor("Cornices"), Tbl_AvProblems$Date[1], Tbl_Output$NumBull[3])
  plotAvProblem(Tbl_AvProblems$Date, Tbl_AvProblems$BtlWet,   7, getAvCharColor("Wet Slabs"), Tbl_AvProblems$Date[1], Tbl_Output$NumBull[3])
  plotAvProblem(Tbl_AvProblems$Date, Tbl_AvProblems$BtlLWet,  8, getAvCharColor("Loose Wet"), Tbl_AvProblems$Date[1], Tbl_Output$NumBull[3])
  plotAvProblem(Tbl_AvProblems$Date, Tbl_AvProblems$BtlLDry,  9, getAvCharColor("Loose Dry"), Tbl_AvProblems$Date[1], Tbl_Output$NumBull[3])

  # Writing number of avalanche problems to Tbl_Output
  Tbl_Output <- writeAvProblemToTblOutput(Tbl_Output, OutputIndexMax, Tbl_AvProblems$BtlPersPl, "PersPl", Tbl_Output$NumBull[3])
  Tbl_Output <- writeAvProblemToTblOutput(Tbl_Output, OutputIndexMax, Tbl_AvProblems$BtlDPers,  "DPers", Tbl_Output$NumBull[3])
  Tbl_Output <- writeAvProblemToTblOutput(Tbl_Output, OutputIndexMax, Tbl_AvProblems$BtlPers,   "Pers", Tbl_Output$NumBull[3])
  Tbl_Output <- writeAvProblemToTblOutput(Tbl_Output, OutputIndexMax, Tbl_AvProblems$BtlStorm,  "Storm", Tbl_Output$NumBull[3])
  Tbl_Output <- writeAvProblemToTblOutput(Tbl_Output, OutputIndexMax, Tbl_AvProblems$BtlWind,   "Wind", Tbl_Output$NumBull[3])
  Tbl_Output <- writeAvProblemToTblOutput(Tbl_Output, OutputIndexMax, Tbl_AvProblems$BtlCorn,   "Corn", Tbl_Output$NumBull[3])
  Tbl_Output <- writeAvProblemToTblOutput(Tbl_Output, OutputIndexMax, Tbl_AvProblems$BtlWet,    "Wet", Tbl_Output$NumBull[3])
  Tbl_Output <- writeAvProblemToTblOutput(Tbl_Output, OutputIndexMax, Tbl_AvProblems$BtlLWet,   "LWet", Tbl_Output$NumBull[3])
  Tbl_Output <- writeAvProblemToTblOutput(Tbl_Output, OutputIndexMax, Tbl_AvProblems$BtlLDry,   "LDry", Tbl_Output$NumBull[3])

  # Calculation of danger rating statistics
  Tbl_Output$LowC[OutputIndexMax]  <- sum(Tbl_AvProblems$BtlDR=="1 - LOW"          || Tbl_AvProblems$BtlDR=="Low", na.rm=T)
  Tbl_Output$ModC[OutputIndexMax]  <- sum(Tbl_AvProblems$BtlDR=="2 - MODERATE"     || Tbl_AvProblems$BtlDR=="Moderate", na.rm=T)
  Tbl_Output$ConsC[OutputIndexMax] <- sum(Tbl_AvProblems$BtlDR=="3 - CONSIDERABLE" || Tbl_AvProblems$BtlDR=="Considerable", na.rm=T)
  Tbl_Output$HighC[OutputIndexMax] <- sum(Tbl_AvProblems$BtlDR=="4 - HIGH"         || Tbl_AvProblems$BtlDR=="High", na.rm=T)
  Tbl_Output$ExtC[OutputIndexMax]  <- sum(Tbl_AvProblems$BtlDR=="5 - EXTREME"      || Tbl_AvProblems$BtlDR=="Extreme", na.rm=T)


  # Number of avalanche problems chart
  par(mai=MarginsBtl2)

  Tbl_AvProblems$BtlNumProb <- Tbl_AvProblems$BtlDPers + Tbl_AvProblems$BtlPers + Tbl_AvProblems$BtlStorm + Tbl_AvProblems$BtlWind +
    Tbl_AvProblems$BtlCorn + Tbl_AvProblems$BtlWet + Tbl_AvProblems$BtlLWet + Tbl_AvProblems$BtlLDry

  plot (Tbl_AvProblems$Date, Tbl_AvProblems$BtlNumProb, type="n",
        ylim=c(-0.5, 4.5), yaxt="n", ylab="",
        xlim=c(min(Tbl_AvProblems$Date), max(Tbl_AvProblems$Date)), xlab="", xaxt="n")

  # Grid lines
  abline(h=c(0:4), col="grey", lty=2, lwd=0.5)
  abline(v=seq(min(Tbl_AvProblems$Date), max(Tbl_AvProblems$Date), by="weeks"), col="grey", lty=2, lwd=0.5)

  axis(4, at=c(0:4), labels=c(0:4), las=1)
  axis.Date(1, at=seq(min(Tbl_AvProblems$Date), max(Tbl_AvProblems$Date), by="weeks"), format="%m/%d/%y", las=2)
  mtext("Date",1, line=5.5, cex=1)
  mtext("Num. Prob.", 4, line=2.5, cex=.75)

  points(Tbl_AvProblems$Date, Tbl_AvProblems$BtlNumProb, type="l")
  points(Tbl_AvProblems$Date, Tbl_AvProblems$BtlNumProb, pch=8, cex=0.5)

  # Writing number of number of avalanche problems to chart and to Tbl_Output
  printNumProbPercentages(Tbl_AvProblems$BtlNumProb, max(Tbl_AvProblems$Date), Tbl_Output$NumBull[3])
  Tbl_Output <- writeNumProbToTblOutput(Tbl_Output, OutputIndexMax, Tbl_AvProblems$BtlNumProb, Tbl_Output$NumBull[3])

  # Close png file
  dev.off()
  par(def.par)

  # Calculation of danger rating summary stats
  Tbl_Output$NoRC  <- Tbl_Output$NumBull - Tbl_Output$LowC - Tbl_Output$ModC - Tbl_Output$ConsC- Tbl_Output$HighC - Tbl_Output$ExtC
  Tbl_Output$LowP  <- round(Tbl_Output$LowC/Tbl_Output$NumBull, 2)
  Tbl_Output$ModP  <- round(Tbl_Output$ModC/Tbl_Output$NumBull, 2)
  Tbl_Output$ConsP <- round(Tbl_Output$ConsC/Tbl_Output$NumBull, 2)
  Tbl_Output$HighP <- round(Tbl_Output$HighC/Tbl_Output$NumBull, 2)
  Tbl_Output$ExtP  <- round(Tbl_Output$ExtC/Tbl_Output$NumBull, 2)
  Tbl_Output$NoRP  <- round(Tbl_Output$NoRC/Tbl_Output$NumBull, 2)

  # Writing DFOutput output
  FigName <- gsub(' ','',FigName)
  write.csv(Tbl_Output, paste0(FigName, "-stats.csv"), row.names = F)

  #End time
  TimeEnd <- Sys.time()
  Time <- TimeEnd - TimeStart
  ## User feedback
  print(paste ("Run time:", Time, "secs ", "- Done "))

}
