#' Creates and validates conditional inference trees (ctree from the party package) for danger ratings
#'
#' Creates and validates conditional inference trees (ctree from the party package) for danger ratings.
#' This function was created for Taylor Clark's research in 2018.
#' @param TreeFormula Formula for tree as string
#' @param TrainData Data frame with training dataset
#' @param TestData Data frame with test dataset (default is training dataset)
#' @return List with tree and performance measures
#' @export
#'
#' @examples
#' require(SarpBulletinTools)
#'
#' load(url("http://data.avalancheresearch.ca/2018_Analysis_Taylor.RData"))
#' nrow(TblAnalysis)
#'
#' ## Extract assessments with strom slabs only
#' Tbl <- TblAnalysis[TblAnalysis$ PROB_COMBINATION=="STORM",c("DAY0", "STORM_LIKELIHOOD_TYP", "STORM_SIZE_TYP")]
#' names(Tbl) <- c("DR", "LH", "SZ")
#' summary(Tbl)
#'
#' ## Split into training and test dataset
#' Data <- splitDFTrainTest(Tbl)
#'
#' ## Create ctree model
#' TreeFormula <- "DR ~ LH + SZ"
#' Model <- createDRCTreeAndValidate(TreeFormula = TreeFormula, TrainData = Data$train, TestData = Data$test)
#'
#' ## Plot tree
#' plot(Model$CTree)
#'
#' ## Show paths and final rules for each of the terminal nodes
#' View(Model$PathsRules)
#'
#' ## Show perforamce measures
#' Model$ConfMatrix
#' Model$Acc
#' Model$AccPlusOne
#' Model$kappa
#' Model$mRoc.value
#'

createDRCTreeAndValidate <- function(TreeFormula, TrainData, TestData=TrainData) {

  # Create decision tree with training dataset
  CTree <- ctree(as.formula(TreeFormula), data = TrainData)

  ## Pick dependent column in TestData
  AssessCol <- gsub(" ", "", strsplit(TreeFormula, "~")[[1]][1])
  TestData$DRAssess <- TestData[,AssessCol]

  ## Apply tree to test data
  TestData$DRPred <- predict(CTree, newdata = TestData, type = "response")

  ## Create table for asssessment vs Prediction
  ConfMatrix <- table(Assess=TestData$DRAssess, Pred=TestData$DRPred)

  ## Accuracy value for DR
  Correct <- sum(diag(ConfMatrix))
  Acc <- round(Correct/nrow(TrainData),3)

  ## Accuracy value for DR + 1
  CorrectPlusOne <- Correct
  for (i in 2:4) {
    CorrectPlusOne <- CorrectPlusOne + ConfMatrix[i,i-1]
  }
  AccPlusOne <- round(CorrectPlusOne/nrow(TrainData),3)

  ## Kappa values
  cind <- grep("DR",colnames(TestData))
  kTab <- TestData[,cind]
  kappa <- kappa2(kTab)

  ## ROC values
  mRoc <- multiclass.roc(TestData$DRAssess, predictor = TestData$DRPred)
  mRoc.value <- round(mRoc[[7]],3)

  ## Extract paths and rules
  PathsRules <- extractRulesFromCTree(CTree, TrainData)

  ## Return values
  Output <- list(CTree = CTree,
                 Formula = TreeFormula,
                 TrainData = TrainData,
                 TestData = TestData,
                 ConfMatrix = ConfMatrix,
                 Acc = Acc,
                 AccPlusOne = AccPlusOne,
                 kappa = kappa,
                 mRoc.value = mRoc.value,
                 PathsRules = PathsRules)

  class(Output) <- append(class(Output),"CTreeDRModel")

  return(Output)

}

