#' Derives Mock and Birkeland (2000) snow climate classification from data frame with daily weather observations
#'
#' Derives Mock and Birkeland (2000) snow climate classification from data frame with daily weather observations
#' @param DailyWxData Data frame with daily weather observations. The data frame needs to have the following columns: \cr
#'        Date: Date of class chronological object from the chron package \cr
#'        AvgTemp: Average daily air temperature in degrees Celcius \cr
#'        Precip: Total amount of precip (snow and rain) in mm of snow water equivalent \cr
#'        Hn24: Amount of new snow in 24 hour period in cm \cr
#'        Rain: Amount of rain in 24 hous period in mm \cr
#'        HS: Height of snowpack in cm
#'
#' @return Dataframe classification results for each season
#' @export

deriveSnowClimateClass_MockBirkeland2000 <- function(DailyWxData) {
  ## Decision thresholds and classifications from Moke & Birkeland (2000)
  ## ********************************************************************

  Decision_01_Thres <- 80    ## Total Rain > 80 mm
  Decision_01_Class <- "maritime"

  Decision_02_Thres <- -3.5  ## Average air temp > -3.5 degrees C
  Decision_02_Class <- "maritime"

  Decision_03_Thres <- -10.0  ## Dec TG < -10 degrees / m
  Decision_03_Class <- "continental"

  Decision_04_Thres <- 1000  ## Total SWE > 100 cm / 1000 mm
  Decision_04_Class <- "maritime"

  Decision_05_Thres <- 560 ## Total snowfall > 560 cm
  Decision_05_Class <- "transitional"

  Decision_06_Thres <- -7  ## Average air temp < -7 degrees C
  Decision_06_Class <- "continental"

  Decision_Default  <- "transitional"


  ## Custom functions
  ## ****************

  ## Gets first day of a season
  getSeasonDateStart <- function(Season) {
    Date_Start <- dates(paste(Season-1, "12", "01", sep="-"), format="y-m-d")
    return(Date_Start)
  }

  ## Gets last day of a season
  getSeasonDateEnd <- function(Season) {
    Date_Start <- dates(paste(Season, "03", "31", sep="-"), format="y-m-d")
    return(Date_Start)
  }

  ## Gets first day of a December
  getDecDateStart <- function(Season) {
    Date_Start <- dates(paste(Season-1, "12", "01", sep="-"), format="y-m-d")
    return(Date_Start)
  }

  ## Gets last day of a December
  getDecDateEnd <- function(Season) {
    Date_Start <- dates(paste(Season-1, "12", "31", sep="-"), format="y-m-d")
    return(Date_Start)
  }


  ## Check input dataframe for correct columns
  ## *****************************************

  DailyWxData_ColNames <- names(DailyWxData)

  ## Date
  if (!("Date" %in% DailyWxData_ColNames)) {stop("Input DF need to have a 'Date' column!")}
  if (!("dates" %in% class(DailyWxData$Date))) {stop("'Date' column needs to be of class 'dates'!")}

  ## Other columns
  if (!("Date" %in% DailyWxData_ColNames)) {stop("Input DF need to have a 'Date' column!")}
  if (!("AvgTemp" %in% DailyWxData_ColNames)) {stop("Input DF need to have a 'AvgTemp' column!")}
  if (!("Precip" %in% DailyWxData_ColNames)) {stop("Input DF need to have a 'Precip' column!")}
  if (!("Hn24" %in% DailyWxData_ColNames)) {stop("Input DF need to have a 'Hn24' column!")}
  if (!("Rain" %in% DailyWxData_ColNames)) {stop("Input DF need to have a 'Rain' column!")}
  if (!("Hs" %in% DailyWxData_ColNames)) {stop("Input DF need to have a 'Hs' column!")}


  ## Determine seasons and build output dataframe
  ## ********************************************

  DailyWxData$Season <- deriveAvSeasonFromDate(DailyWxData$Date)

  Output <- data.frame(Season = unique(DailyWxData$Season)[!is.na(unique(DailyWxData$Season))], 
                       NumObs = NA,
                       AvgTempNA = NA,
                       PrecipNA = NA,
                       Hn24NA = NA,
                       RainNA = NA,
                       HsNA = NA,
                       DecAvgTempNA = NA,
                       DecHsNA = NA,
                       AvgTempAvg = NA,
                       PrecipSum = NA,
                       Hn24Sum = NA,
                       RainSum = NA,
                       DecAvgTempAvg = NA,
                       DecHSAvg = NA,
                       DecTGAvg = NA,
                       Class = NA,
                       Decision = NA,
                       MissingDays = NA)


  ## For loop for processing each season
  ## ***********************************

  for (Index_Season in 1:nrow(Output)) {

    ## Create temporary tables
    Temp_Season <- DailyWxData[DailyWxData$Date>=getSeasonDateStart(Output$Season[Index_Season]) &
                               DailyWxData$Date<=getSeasonDateEnd(Output$Season[Index_Season]),]

    Temp_Dec <- DailyWxData[DailyWxData$Date>=getDecDateStart(Output$Season[Index_Season]) &
                            DailyWxData$Date<=getDecDateEnd(Output$Season[Index_Season]),]

    ## Count number of observations 
    Output$NumObs[Index_Season] <- nrow(Temp_Season) - max(sum(is.na(Temp_Season$AvgTemp)),
                                                           sum(is.na(Temp_Season$Precip)),
                                                           sum(is.na(Temp_Season$Hn24)),
                                                           sum(is.na(Temp_Season$Rain)),
                                                           sum(is.na(Temp_Season$Hs)),
                                                           sum(is.na(Temp_Dec$AvgTemp)),
                                                           sum(is.na(Temp_Dec$Hs)))                     

    ## Determine number of missing data
    Output$AvgTempNA[Index_Season]    <- sum(is.na(Temp_Season$AvgTemp))
    Output$PrecipNA[Index_Season]     <- sum(is.na(Temp_Season$Precip))
    Output$Hn24NA[Index_Season]       <- sum(is.na(Temp_Season$Hn24))
    Output$RainNA[Index_Season]       <- sum(is.na(Temp_Season$Rain))
    Output$HsNA[Index_Season]         <- sum(is.na(Temp_Season$Hs))
    Output$DecAvgTempNA[Index_Season] <- sum(is.na(Temp_Dec$AvgTemp))
    Output$DecHsNA[Index_Season]      <- sum(is.na(Temp_Dec$Hs))

    ## Calculate sums and averages
    Output$AvgTempAvg[Index_Season]    <- round(mean(Temp_Season$AvgTemp, na.rm = T), 1)
    Output$PrecipSum[Index_Season]     <- sum(Temp_Season$Precip, na.rm = T)
    Output$Hn24Sum[Index_Season]       <- sum(Temp_Season$Hn24, na.rm = T)
    Output$RainSum[Index_Season]       <- sum(Temp_Season$Rain, na.rm = T)
    Output$DecAvgTempAvg[Index_Season] <- round(mean(Temp_Dec$AvgTemp, na.rm = T), 1)
    Output$DecHSAvg[Index_Season]      <- round(mean(Temp_Dec$Hs, na.rm = T), 1)

  } ## end of for loop for Index_Season


  ## Calculate December Temperature gradient
  ## ***************************************
  Output$DecTGAvg <- round(Output$DecAvgTempAvg/(Output$DecHSAvg/100), 1)


  ## Classification
  ## **************
  Output$Class <- ifelse(Output$RainSum > Decision_01_Thres, Decision_01_Class,
                  ifelse(Output$AvgTempAvg > Decision_02_Thres, Decision_02_Class,
                  ifelse(Output$DecTGAvg < Decision_03_Thres, Decision_03_Class,
                  ifelse(Output$PrecipSum > Decision_04_Thres, Decision_04_Class,
                  ifelse(Output$Hn24Sum > Decision_05_Thres, Decision_05_Class,
                  ifelse(Output$AvgTempAvg < Decision_06_Thres, Decision_06_Class, Decision_Default))))))

  Output$Decision <- ifelse(Output$RainSum > Decision_01_Thres, 1,
                     ifelse(Output$AvgTempAvg > Decision_02_Thres, 2,
                     ifelse(Output$DecTGAvg < Decision_03_Thres, 3,
                     ifelse(Output$PrecipSum > Decision_04_Thres, 4,
                     ifelse(Output$Hn24Sum > Decision_05_Thres, 5,
                     ifelse(Output$AvgTempAvg < Decision_06_Thres, 6, 7))))))
  
  ## TEST FOR 10 CONSECUTIVE DAYS OF NA VALUES
  ## *****************************************
  years <- unique(Output$Season)
  clim.var <- c("AvgTemp","Precip","Hn24","Rain","Hs")
  
  for(year.test in min(years):max(years)){
    StartDate <- dates(paste(year.test-1, "12", "1", sep="-"), format="y-m-d")
    EndDate   <- dates(paste(year.test, "03", "31", sep="-"), format="y-m-d")
    ### subset for year & winter season
    wxobs.yr <- data_station[data_station$Date >= StartDate & data_station$Date <= EndDate,]
    ### count NA values for 10 day periods
    season.days <- wxobs.yr$Date
    ten.NA <- FALSE
    
    ### check every ten day period
    for(day10 in 1:(length(season.days)-10)){
      if(!ten.NA)
        wxobs.10 <- wxobs.yr[wxobs.yr$Date >= season.days[day10] & wxobs.yr$Date <= season.days[day10+10],]
      ### check each climate variable
      for(v in 1:length(clim.var)){
        if(!ten.NA){
          if(sum(is.na(wxobs.10[,clim.var[v]])) == 10) ten.NA <- TRUE
        }
      }### end of climate variable loop
    }### end of 10 day check loop 
    
    ### write to DF
    if(ten.NA){
      Output[Output$Season == year.test,'MissingDays'] <- 'Exceeds'
    }else{
      Output[Output$Season == year.test,'MissingDays'] <- 'Good'
    }
  }### end of year loop
  
  ## Return
  ## ******
  return (Output)

}
