#' Extracts bulletins from Bulletins object.
#'
#' Extracts bulletins from Bulletins object by index or BULLETIN_ID.
#' @param BulletinObj Object of type Bulletins (generally created by getBulletins())
#' @param ByIndex Index of bulletin(s) to be extracted in BulletinObj$Bulletins dataframe.
#' @param ByBulletinID BulletinID of bulletins to be extracted.
#' @return Object of type Bulletins
#'
#' @examples
#' require(SarpBulletinTools)
#' require(SarpBulletinToolsPrivate)
#'
#' Bulletins <- getBulletins("AvCan", Seasons=2017)
#'
#' for (Index in 1:4) {
#'  SingleBulletin <- extractFromBulletinObj(Bulletins, ByIndex = Index)
#'  summary(SingleBulletin)
#' }
#'
#' @export

extractFromBulletinObj <- function(BulletinObj, ByIndex=NA, ByBulletinID=NA) {

  ## Checks
  if(!("Bulletins" %in% class(BulletinObj))) {
    stop("BulletinObj needs to be of class 'Bulletins'")
  }

  if(is.na(ByIndex[1]) & is.na(ByBulletinID[1])) {
    stop("Either ByIndex or ByBulletinID need to be specified!")
  }

  if(!is.na(ByIndex[1]) & !is.na(ByBulletinID[1])) {
    stop("Only either ByIndex or ByBulletinID can be specified!")
  }

  if (!is.na(ByIndex[1])) {
    ByBulletinID <- BulletinObj$Bulletins$BULLETIN_ID[ByIndex]
  }

  ## Extract objects
  Output <- list(Bulletins = BulletinObj$Bulletins[BulletinObj$Bulletins$BULLETIN_ID %in% ByBulletinID,],
                 AvProblems = list (Alp = BulletinObj$AvProblems$Alp[BulletinObj$AvProblems$Alp$BULLETIN_ID %in% ByBulletinID,],
                                    Tl =  BulletinObj$AvProblems$Tl[BulletinObj$AvProblems$Tl$BULLETIN_ID %in% ByBulletinID,],
                                    Btl = BulletinObj$AvProblems$Btl[BulletinObj$AvProblems$Btl$BULLETIN_ID %in% ByBulletinID,]),
                 DngRating = list (Alp = BulletinObj$DngRating$Alp[BulletinObj$DngRating$Alp$BULLETIN_ID %in% ByBulletinID,],
                                   Tl =  BulletinObj$DngRating$Tl[BulletinObj$DngRating$Tl$BULLETIN_ID %in% ByBulletinID,],
                                   Btl = BulletinObj$DngRating$Btl[BulletinObj$DngRating$Btl$BULLETIN_ID %in% ByBulletinID,]))

  if (nrow(Output$Bulletins)==0) {
    warning("No bulletins were found for specified index or BulletinID!", immediate. = TRUE)
  }

  ## Add class label
  class(Output) <- append(class(Output), "Bulletins")

  ## Output
  return(Output)

}
