#' Extracts decision rules from CTreeModel object
#'
#' Extracts decision rules from CTreeModel object. This is an internal function that is used by createDRCTreeAbndValidate()
#' This function was created for Taylor's research in 2018.
#'
#' @param CTree CTree
#' @param TrainData Training dataframe that the tree was built with
#'
extractRulesFromCTree <- function(CTree, TrainData) {

  ## Subfunction
  ## ***********

  ## Extract rule paths
  extractRulePaths <- function(CTree, NodeID=1, ParentNodeID=NA, Branch=NA, ListOfPaths=data.frame(node=numeric(0), pnode=numeric(0), path=character(0)), Verbose=F) {

    ## Extracts rootnode from a tree
    if("constparty" %in% class(CTree)) {
      Node <- CTree[[1]]$node
    } else {
      Node <- CTree
    }

    if (is.terminal(Node)) {

      if (Verbose) {cat(paste(NodeID, "--> terminal node\n"))}
      Output <- ListOfPaths

    } else {

      ## Splitting rule of node and kid node IDs
      Slabs <- character_split(split_node(Node))
      KidNode1ID <- kids_node(Node)[[1]]$id
      KidNode2ID <- kids_node(Node)[[2]]$id

      if (Verbose) {cat(paste(NodeID, "-->", KidNode1ID, "&", KidNode2ID, "\n"))}

      ## Updating paths for very first nodes
      if(is.na(ParentNodeID)) {
        ListOfPaths <- rbind(ListOfPaths, data.frame(node=c(KidNode1ID, KidNode2ID), pnode=rep(NodeID,2), path=paste("{1}", Slabs$name, Slabs$levels)))
        ListOfPaths$path <- as.character(ListOfPaths$path)

      ## Updating path for any subsequent nodes
      } else {

        PathIndex <- which(ListOfPaths$pnode==ParentNodeID & ListOfPaths$node==NodeID)
        PrevPathTemp <- ListOfPaths$path[PathIndex] <- paste(ListOfPaths$path[PathIndex],"-->")

        ## First kid
        ListOfPaths$path[PathIndex] <- paste(PrevPathTemp, paste0("{", NodeID, "}"), Slabs$name, Slabs$levels[1])
        ListOfPaths$node[PathIndex] <- KidNode1ID
        ListOfPaths$pnode[PathIndex] <- NodeID

        ## Second kid
        ListOfPaths <- rbind(ListOfPaths, data.frame(node=KidNode2ID, pnode=NodeID, path=paste(PrevPathTemp, paste0("{", NodeID, "}"), Slabs$name, Slabs$levels[2])))

      }

      ## ListOfPaths$path <- as.character(ListOfPaths$path)

      ## Working down the tree recursively
      ParentNodeID <- NodeID
      ListOfPaths <- extractRulePaths(CTree = Node$kids[[1]], NodeID = KidNode1ID, ParentNodeID = ParentNodeID, ListOfPaths = ListOfPaths)
      ListOfPaths <- extractRulePaths(CTree = Node$kids[[2]], NodeID = KidNode2ID, ParentNodeID = ParentNodeID, ListOfPaths = ListOfPaths)

      Output <- ListOfPaths

    }

    ## Return output dataframe
    return(Output)

  }

  ## Translates original rule string into proper English
  translatePath2Rules <- function(PathString, TrainData) {

    ## Remove node numbers
    PathString <- gsub(pattern = "\\{[0-9]*\\} ", "", PathString)

    ## Get variable names
    VarNames <- names(TrainData)

    ## Extract individuals rules from string
    Tbl <- data.frame(Term = character(0), Pos=numeric(0))
    for (i in 1:length(VarNames)) {
      Temp <- as.numeric(gregexpr(VarNames[i], PathString)[[1]])
      if (Temp[1]!=-1) {
        for (l in 1:length(Temp)) {
          Tbl <- rbind(Tbl, data.frame(Term=VarNames[i], Pos=Temp[l]))
        }
      }
    }
    Tbl <- Tbl[order(Tbl$Pos),]
    Tbl$Term <- as.character(Tbl$Term)
    rm(Temp)

    Tbl$Start <- Tbl$Pos + nchar(Tbl$Term) + 1
    Tbl$End <- c((Tbl$Pos[2:nrow(Tbl)]-6), nchar(PathString))

    for (i in 1:nrow(Tbl)) {
      Tbl$ValuesOrig[i] <- substr(PathString, Tbl$Start[i], Tbl$End[i])
    }

    ## Deterime type of comparison
    Tbl$Oper <- "Group"
    Tbl$Oper[grep("<=", Tbl$ValuesOrig)] <- "SmallerEqual"
    Tbl$Oper[grep(">", Tbl$ValuesOrig)] <- "Larger"

    ## Extract values of rules
    Tbl$ValuesNum[Tbl$Oper=="Larger"] <- gsub("> ", "", Tbl$ValuesOrig[Tbl$Oper=="Larger"])
    Tbl$ValuesNum[Tbl$Oper=="SmallerEqual"] <- gsub("<= ", "", Tbl$ValuesOrig[Tbl$Oper=="SmallerEqual"])
    Tbl$ValuesNum[Tbl$Oper=="Group"] <- Tbl$ValuesOrig[Tbl$Oper=="Group"]
    Tbl$ValuesNum <- gsub(pattern = "\\(-Inf,", "", Tbl$ValuesNum)
    Tbl$ValuesNum <- gsub(pattern = "\\(.,", "", Tbl$ValuesNum)
    Tbl$ValuesNum <- gsub(pattern = "\\]", "", Tbl$ValuesNum)
    Tbl$ValuesNum <- gsub(pattern = " ", "", Tbl$ValuesNum)

    Tbl[order(Tbl$Term),]

    ## Condense rules
    TblCond <- data.frame(Term=character(), ValuesLrg=numeric(), ValuesSmlEqu=numeric(), ValuesGrp=character())
    for (col in 1:ncol(TrainData)) {
      if (names(TrainData)[col] %in% Tbl$Term) {
        Temp <- Tbl[Tbl$Term==names(TrainData)[col],]
        if ("Larger" %in% Temp$Oper) {
          ValuesLrg <- max(as.numeric(Temp$ValuesNum[Temp$Oper=="Larger"]))
        } else {
          ValuesLrg <- NA
        }
        if ("SmallerEqual" %in% Temp$Oper) {
          ValuesSmlEqu <- min(as.numeric(Temp$ValuesNum[Temp$Oper=="SmallerEqual"]))
        } else {
          ValuesSmlEqu <- NA
        }
        if ("Group" %in% Temp$Oper) {
          ValuesGrp <- Temp$ValuesNum[nchar(Temp$ValuesNum)==min(nchar(Temp$ValuesNum))]
        } else {
          ValuesGrp <- NA
        }
        TblCond <- rbind(TblCond, data.frame(Term=Temp$Term, ValuesLrg=ValuesLrg, ValuesSmlEqu=ValuesSmlEqu, ValuesGrp=ValuesGrp))
      }
    }
    TblCond <- unique(TblCond)

    ## Replace numeric values with proper value
    TblCond$TermLrg <- as.character(TblCond$ValuesLrg)
    TblCond$TermSmlEqu <- as.character(TblCond$ValuesSmlEqu)
    TblCond$TermGrp <- as.character(TblCond$ValuesGrp)

    for (i in 1:nrow(TblCond)) {
      Levels <- levels(TrainData[[as.character(TblCond$Term[i])]])
      if(!is.na(TblCond$ValuesLrg[i])) {TblCond$TermLrg[i] <- Levels[TblCond$ValuesLrg[i]]}
      if(!is.na(TblCond$ValuesSmlEqu[i])) {TblCond$TermSmlEqu[i] <- Levels[TblCond$ValuesSmlEqu[i]]}
      if(!is.na(TblCond$ValuesGrp[i])) {
        for(l in 1:length(Levels)) {
          TblCond$TermGrp[i] <- gsub(l, Levels[l], TblCond$TermGrp[i])
        }
        TblCond$TermGrp[i] <- gsub("Inf", Levels[length(Levels)], TblCond$TermGrp[i])
      }
    }

    ## Identify neighbouring values
    for (i in 1:nrow(TblCond)) {
      if(!is.na(TblCond$ValuesLrg[i]) & !is.na(TblCond$ValuesSmlEqu[i])) {
        if((TblCond$ValuesSmlEqu[i]-TblCond$ValuesLrg[i])==1) {
          TblCond$TermGrp[i] <- TblCond$TermSmlEqu[i]
          TblCond$TermSmlEqu[i] <- NA
          TblCond$TermLrg[i] <- NA
        }
      }
    }

    ## Identify smallest values
    for (i in 1:nrow(TblCond)) {
      if(!is.na(TblCond$ValuesSmlEqu[i])) {
        if(TblCond$ValuesSmlEqu[i]==1) {
          TblCond$TermGrp[i] <- TblCond$TermSmlEqu[i]
          TblCond$TermSmlEqu[i] <- NA
        }
      }
    }

    ## Identify largest values
    for (i in 1:nrow(TblCond)) {
      if(!is.na(TblCond$ValuesLrg[i])) {
        Levels <- levels(TrainData[[as.character(TblCond$Term[i])]])
        if (TblCond$ValuesLrg[i]==(length(Levels)-1)) {
          TblCond$TermGrp[i] <- Levels[length(Levels)]
          TblCond$TermLrg[i] <- NA
        }
      }
    }

    ## Formulate rules
    TblCond$Rules <- NA
    for (i in 1:nrow(TblCond)) {

      if(!is.na(TblCond$TermGrp[i])) {
        TblCond$Rules[i] <- paste(TblCond$Term[i], "=", TblCond$TermGrp[i])
      } else if (!is.na(TblCond$TermLrg[i]) & !is.na(TblCond$TermSmlEqu[i]) ) {
        TblCond$Rules[i] <- paste(TblCond$TermLrg[i], "<", TblCond$Term[i], "<=", TblCond$TermSmlEqu[i])
      } else if (!is.na(TblCond$TermLrg[i])) {
        TblCond$Rules[i] <- paste(TblCond$Term[i], ">", TblCond$TermLrg[i])
      } else if (!is.na(TblCond$TermSmlEqu[i])) {
        TblCond$Rules[i] <- paste(TblCond$Term[i], "<=", TblCond$TermSmlEqu[i])
      }

    }

    ## Replace known issues that are unclear
    TblCond$Rules[TblCond$Rules=="ELEV > Alp"] <- "ELEV = Tl|Btl"

    ## Concatenate rules into single string
    TblCond <- TblCond[order(TblCond$Term),]
    Rule <- paste(TblCond$Rules, collapse = " & ")

    ## Return rule
    return(Rule)

  }


  ## Main function
  ## *************

  ## Get variable names
  VarNames <- attr(CTree[[1]]$terms, "term.labels")

  ## Extract rule paths
  OutputDF <- extractRulePaths(CTree)

  ## Eliminate parent node
  OutputDF <- deleteDFColumnsByName(OutputDF, "pnode")

  ## Include proper variable names
  for (VarIndex in 1:length(VarNames)) {
    OutputDF$path <- gsub(pattern = paste0("V", VarIndex+1), VarNames[VarIndex], OutputDF$path)
  }

  ## Translate paths into rules
  OutputDF$rules <- NA
  for (i in 1:nrow(OutputDF)) {
   OutputDF$rules[i] <- translatePath2Rules(OutputDF$path[i], TrainData)
  }

  ## Sorting
  OutputDF[order(OutputDF$node),]

  ## Return output dataframe
  return(OutputDF)

}
