#' Formats bulletin object for hazard analysis.
#'
#' Formats bulletin object for hazard analysis.
#' @param BulletinObj Bulletin object created by getBulletins() function
#' @param Format Format of output dataframe. Either 'wide' (default) with one record for each elevation band and avalanche problems in different columns or 'long' with avalanche problems for each elevant band stacked.
#' @param ElevBands What elevation band to be included. Default is c("Alp", "Tl", "Btl")
#' @param AvProbColIncluded What avalanche problme columns to be included. NA (default) includes all.
#' @param MissingAvProblemInWideFormat Specifies how missing avalanche problem values are handled. NA (default) does nothing and 0 fills NA with 0.
#' @param EliminateEmptyRecords Switch for eliminating completely empty records (no danger rating, no av problems) in wide format.
#' @param AddDngRatingPrevDay Switch for adding danger rating DAY0 of previous day. Default=FALSE
#' @param AddDngRatingOtherElev Switch for adding danger rating DAY0 of other elevation bands. Default=FALSE
#' @param Verbose Switch for turning user feedback on/off.
#' @return Dataframe with reformatted avalanche problem information
#'
#' @examples
#'
#' require(SarpBulletinTools)
#' require(SarpBulletinToolsPrivate)
#'
#' ## SINGLE BULLETIN
#' ## ***************
#'
#' Agency <- "AvCan"
#' Region <- "Kootenay Boundary"
#' StartDate <- dates("2014-12-18", format="y-m-d")
#' BulletinObj <- getBulletins(Agency, StartDate=StartDate, Regions=Region)
#' summary(BulletinObj)
#'
#' BulletinAnalysis1 <- formatBulletinObjForHzdAnalysis(BulletinObj)
#' BulletinAnalysis2 <- formatBulletinObjForHzdAnalysis(BulletinObj, MissingAvProblemInWideFormat = 0)
#' BulletinAnalysis3 <- formatBulletinObjForHzdAnalysis(BulletinObj, AvProbColIncluded = c("LIKELIHOOD_MAX", "SIZE_MAX"))
#' BulletinAnalysis4 <- formatBulletinObjForHzdAnalysis(BulletinObj, AvProbColIncluded = c("LIKELIHOOD_MAX", "SIZE_MAX"), MissingAvProblemInWideFormat = 0)
#'
#' ## MULTIPLE BULLETINS
#' ## ******************
#'
#' Agency <- "AvCan"
#' Season <- 2010
#'
#' ## Get bulletin data for a season
#' Bulletins <- getBulletins(Agency=Agency, Seasons=Season)
#' summary(Bulletins)
#'
#' ## Format bulletin data for analysis
#' ForAnalysis <- formatBulletinObjForHzdAnalysis(Bulletins)
#'
#' ## TESTING FOR CONSISTENCY
#' ## ***********************
#' ElevBand <- "Alp"
#'
#' ## Number of bulletin records
#' nrow(Bulletins$DngRating[[ElevBand]])
#' nrow(ForAnalysis[ForAnalysis$ELEV==ElevBand,])
#' # -> it is possible that the analysis DF has more. The difference should be made up by the number of NA danger ratings (see next comparison)
#'
#' ## Number of NA danger ratings
#' sum(is.na(Bulletins$DngRating[[ElevBand]]$DAY0))
#' sum(is.na(ForAnalysis[ForAnalysis$ELEV==ElevBand,]$DAY0))
#' # -> it is possible that the analysis DF has more than the original bulletin object (e.g., av problems that were characterized for a elevation band where there is not danger rating.)
#'
#' ## Distribution of danger ratings
#' table(Bulletins$DngRating[[ElevBand]]$DAY0)
#' table(ForAnalysis[ForAnalysis$ELEV==ElevBand,]$DAY0)
#'
#' ## Distribution of avalanche problems
#' table(Bulletins$AvProblems[[ElevBand]]$CHARACTER)
#' sum(!is.na(ForAnalysis$DPERS_PROBLEM_ID[ForAnalysis$ELEV==ElevBand]))
#' sum(!is.na(ForAnalysis$PERS_PROBLEM_ID[ForAnalysis$ELEV==ElevBand]))
#' sum(!is.na(ForAnalysis$STORM_PROBLEM_ID[ForAnalysis$ELEV==ElevBand]))
#' sum(!is.na(ForAnalysis$WIND_PROBLEM_ID[ForAnalysis$ELEV==ElevBand]))
#' sum(!is.na(ForAnalysis$CORN_PROBLEM_ID[ForAnalysis$ELEV==ElevBand]))
#' sum(!is.na(ForAnalysis$LDRY_PROBLEM_ID[ForAnalysis$ELEV==ElevBand]))
#' sum(!is.na(ForAnalysis$WET_PROBLEM_ID[ForAnalysis$ELEV==ElevBand]))
#' sum(!is.na(ForAnalysis$LWET_PROBLEM_ID[ForAnalysis$ELEV==ElevBand]))
#' # -> it is possible to have fewer avalanche problems in analaysis DF (e.g., elimination of secondary problems of same problem type during formatting). Messages during formatting and comparison should be consistent.
#'
#' @export

formatBulletinObjForHzdAnalysis <- function(BulletinObj, Format="wide", ElevBands=c("Alp", "Tl", "Btl"), AvProbColIncluded=NA, MissingAvProblemInWideFormat=NA, EliminateEmptyRecords=TRUE, AddDngRatingPrevDay=FALSE, AddDngRatingOtherElev=FALSE, Verbose=TRUE) {

  ## SUB FUNCTIONS
  ## *************

  ## Eliminating secondary avalanche problems of the same time (only needed for HzdWebsite data)
  eliminateMultipleAvProblems <- function(AvProbDF, AvProbAbbrev="", ElevBand="", Verbose=Verbose) {

    DuplicBulletinID <- AvProbDF$BULLETIN_ID[duplicated(AvProbDF$BULLETIN_ID)]

    if(length(DuplicBulletinID)>0) {

      AvProbDFSingle <- AvProbDF[!(AvProbDF$BULLETIN_ID %in% DuplicBulletinID),]

      CounterContri <- 0
      CounterProduct <- 0
      CounterOrder <- 0
      CounterFirst <- 0

      for (IndexDuplic in 1:length(DuplicBulletinID)) {

        AvProbDFMultiple <- AvProbDF[(AvProbDF$BULLETIN_ID == DuplicBulletinID[IndexDuplic]),]

        ProductMax <- max(AvProbDFMultiple$LIKELIHOOD_MAX*AvProbDFMultiple$SIZE_MAX, na.rm = T)
        ProductMin <- min(AvProbDFMultiple$LIKELIHOOD_MAX*AvProbDFMultiple$SIZE_MAX, na.rm = T)

        if (!is.na(AvProbDFMultiple$CONTRI[1])) {
          ContriMax <- max(AvProbDFMultiple$CONTRI, na.rm = T)
          ContriMin <- min(AvProbDFMultiple$CONTRI, na.rm = T)
        } else {
          ContriMax <- 0
          ContriMin <- 0
        }

        if (!is.na(AvProbDFMultiple$SORT_ORDER[1])) {
          OrderMax <- max(AvProbDFMultiple$SORT_ORDER, na.rm = T)
          OrderMin <- min(AvProbDFMultiple$SORT_ORDER, na.rm = T)
        } else {
          OrderMax <- 0
          OrderMin <- 0
        }

        if (ProductMax > ProductMin) {
          AvProbDFSingle <- rbind(AvProbDFSingle, AvProbDFMultiple[(AvProbDFMultiple$LIKELIHOOD_MAX*AvProbDFMultiple$SIZE_MAX)==ProductMax,][1,])
          CounterProduct <- CounterProduct + 1
        } else if(ContriMax > ContriMin) {
          AvProbDFSingle <- rbind(AvProbDFSingle, AvProbDFMultiple[AvProbDFMultiple$CONTRI==ContriMax,][1,])
          CounterContri <- CounterContri + 1
        } else if (OrderMin < OrderMax) {
          AvProbDFSingle <- rbind(AvProbDFSingle, AvProbDFMultiple[AvProbDFMultiple$SORT_ORDER==OrderMin,][1,])
          CounterOrder <- CounterOrder + 1
        } else {
          AvProbDFSingle <- rbind(AvProbDFSingle, AvProbDFMultiple[1,])
          CounterFirst <- CounterFirst + 1
        }

      }

      if(Verbose) {print(paste0(ElevBand, ": Eliminated ", CounterContri+CounterProduct+CounterOrder+CounterFirst, " secondary ", AvProbAbbrev, " records: ", CounterProduct, " (by smaller likelihood x size), ", CounterContri, " (by smaller contribution), ", CounterOrder, " (by lower sort order), ", CounterFirst, " (by first)"))}

    } else {

      AvProbDFSingle <- AvProbDF

    }

    ## Eliminating straight duplicates <<- NOT SURE WHERE THEY COME FROM!!
    if (sum(duplicated(AvProbDFSingle)>0)) {

      ##if(Verbose) {print(paste0("Eliminated ", sum(duplicated(AvProbDFSingle)), " additional straight duplicate ", AvProbAbbrev, " records in ", ElevBand, "."))}
      AvProbDFSingle <- AvProbDFSingle[!duplicated(AvProbDFSingle),]

    }

    return(AvProbDFSingle)

  }


  ## MAIN FUNCTION
  ## *************

  ## Initialization
  Output <- NULL

  ## WIDE FORMAT
  if (tolower(Format)=="wide") {

    for(IndexElevBand in 1:length(ElevBands)) {

      ## Start output df with bulletin and add elevation band
      OutputElev <- BulletinObj$Bulletins
      OutputElev$ELEV <- ElevBands[IndexElevBand]

      ## Add danger rating information
      Dng <- BulletinObj$DngRating[[ElevBands[IndexElevBand]]]
      OutputElev <- merge(OutputElev, Dng, by.x = "BULLETIN_ID", by.y="BULLETIN_ID", all.x=TRUE)

      ## Extract requested columns of avalanche problme table
      if (is.na(AvProbColIncluded[1])) {
        AvProb <- BulletinObj$AvProblems[[ElevBands[IndexElevBand]]]
      } else {
        if (IndexElevBand==1) {
          AvProbColIncluded <- c('BULLETIN_ID', 'PROBLEM_ID', 'CHARACTER', AvProbColIncluded)
        }
        if (sum(!(AvProbColIncluded %in% names(BulletinObj$AvProblems[[ElevBands[IndexElevBand]]])))>0) {
          stop("Specified avalanche problem columns do not exist!")
        } else {
          AvProb <- BulletinObj$AvProblems[[ElevBands[IndexElevBand]]][,AvProbColIncluded]
        }
      }

      ## Merge avalanche problem information
      for (IndexType in 1:length(ListAvProblemTypes)) {

        TempAvProb <- AvProb[AvProb$CHARACTER==ListAvProblemTypes[IndexType],]
        TempAvProb <- eliminateMultipleAvProblems(TempAvProb, AvProbAbbrev=ListAvProblemTypes[IndexType], ElevBands[IndexElevBand], Verbose=Verbose)

        names(TempAvProb)[2:length(names(TempAvProb))] <- paste(ListAvProblemTypesAbbrev[IndexType], names(TempAvProb)[2:length(names(TempAvProb))], sep="_")
        OutputElev <- merge(OutputElev, TempAvProb, by.x = "BULLETIN_ID", by.y="BULLETIN_ID", all.x=TRUE)

      }

      ## Stack elevation band tables together to an overall table
      if(is.null(Output)) {
        Output <- OutputElev
      } else {
        Output <- rbind(Output, OutputElev)
      }

    }

    ## Eliminate totally empty records
    if(EliminateEmptyRecords) {

      EmptyRecords <- (is.na(Output$DAY0) & is.na(Output$DPERS_PROBLEM_ID) &  is.na(Output$PERS_PROBLEM_ID) &
                       is.na(Output$STORM_PROBLEM_ID) & is.na(Output$WIND_PROBLEM_ID) & is.na(Output$CORN_PROBLEM_ID) &
                       is.na(Output$LDRY_PROBLEM_ID) & is.na(Output$WET_PROBLEM_ID) & is.na(Output$LWET_PROBLEM_ID))

      if(sum(EmptyRecords)>0) {
        ##if(Verbose) {print(paste0("Eliminated ", sum(EmptyRecords), " completely empty records (no danger rating, no problems) in combined elevation bands."))}
        Output <- Output[!EmptyRecords,]
      }

    }

    ## Dealing with missing avalanche problems
    if (is.na(MissingAvProblemInWideFormat)) {

      ## do nothing

    } else if (MissingAvProblemInWideFormat==0) {

      for (IndexAvProblem in 1:length(ListAvProblemTypes)) {
        RowsWithMissingValues <- is.na(Output[paste(ListAvProblemTypesAbbrev[IndexAvProblem], "CHARACTER", sep="_")])
        Output[paste(ListAvProblemTypesAbbrev[IndexAvProblem], "CHARACTER", sep="_")][RowsWithMissingValues] <- ListAvProblemTypes[IndexAvProblem]
        if (paste(ListAvProblemTypesAbbrev[IndexAvProblem], "LIKELIHOOD_MIN", sep="_") %in% names(Output)) {
          Output[paste(ListAvProblemTypesAbbrev[IndexAvProblem], "LIKELIHOOD_MIN", sep="_")][RowsWithMissingValues] <- 0
        }
        if (paste(ListAvProblemTypesAbbrev[IndexAvProblem], "LIKELIHOOD_TYP", sep="_") %in% names(Output)) {
          Output[paste(ListAvProblemTypesAbbrev[IndexAvProblem], "LIKELIHOOD_TYP", sep="_")][RowsWithMissingValues] <- 0
        }
        if (paste(ListAvProblemTypesAbbrev[IndexAvProblem], "LIKELIHOOD_MAX", sep="_") %in% names(Output)) {
        Output[paste(ListAvProblemTypesAbbrev[IndexAvProblem], "LIKELIHOOD_MAX", sep="_")][RowsWithMissingValues] <- 0
        }
        if (paste(ListAvProblemTypesAbbrev[IndexAvProblem], "SIZE_MIN", sep="_") %in% names(Output)) {
        Output[paste(ListAvProblemTypesAbbrev[IndexAvProblem], "SIZE_MIN", sep="_")][RowsWithMissingValues] <- 0
        }
        if (paste(ListAvProblemTypesAbbrev[IndexAvProblem], "SIZE_TYP", sep="_") %in% names(Output)) {
          Output[paste(ListAvProblemTypesAbbrev[IndexAvProblem], "SIZE_TYP", sep="_")][RowsWithMissingValues] <- 0
        }
        if (paste(ListAvProblemTypesAbbrev[IndexAvProblem], "SIZE_MAX", sep="_") %in% names(Output)) {
          Output[paste(ListAvProblemTypesAbbrev[IndexAvProblem], "SIZE_MAX", sep="_")][RowsWithMissingValues] <- 0
        }
      }

    } else {

      stop("Value for MissingAvProblemInWideFormat can only be NA (default) or 0!")

    }

    ## Fix bulletin region names
    Output$REGION <- fixBulletinRegionNames(Output$REGION, Output$SEASON)

    ## Fix forecaster names
    Output$FORECASTER <- fixForecasterNames(Output$FORECASTER)

    ## Make ELEV ordinal
    Output$ELEV <- ordered(Output$ELEV, levels=c("Alp", "Tl", "Btl"))

    ## Add danger ratings of previous day and other elevations bands
    if (AddDngRatingPrevDay & AddDngRatingOtherElev) {

      print("Getting danger rating of previous day and other elevation bands.")

      Output$DAYPREV <- NA
      Output$DAY0ALP <- NA
      Output$DAY0TL  <- NA
      Output$DAY0BTL <- NA

      Output[,paste0(ListAvProblemTypesAbbrev, "_PRES_ALP")] <- NA
      Output[,paste0(ListAvProblemTypesAbbrev, "_PRES_TL")]  <- NA
      Output[,paste0(ListAvProblemTypesAbbrev, "_PRES_BTL")] <- NA

      for (Index in 1:nrow(Output)) {

        if((Index %% 500)==0) {print(paste("Processing record", Index, "of", nrow(Output), "..."))}

        Temp_Alp <- Output[(Output$REGION==Output$REGION[Index]) & (Output$PUBLISH_DATE==Output$PUBLISH_DATE[Index]) & (Output$ELEV == 'Alp'),]
        Temp_Tl  <- Output[(Output$REGION==Output$REGION[Index]) & (Output$PUBLISH_DATE==Output$PUBLISH_DATE[Index]) & (Output$ELEV == 'Tl'),]
        Temp_Btl <- Output[(Output$REGION==Output$REGION[Index]) & (Output$PUBLISH_DATE==Output$PUBLISH_DATE[Index]) & (Output$ELEV == 'Btl'),]

        if (nrow(Temp_Alp)==1) {
          for (IndexAvProblem in 1:length(ListAvProblemTypesAbbrev)) {
            Output[Index, paste0(ListAvProblemTypesAbbrev[IndexAvProblem], "_PRES_ALP")] <- ifelse(Temp_Alp[paste0(ListAvProblemTypesAbbrev[IndexAvProblem], "_LIKELIHOOD_MAX")]>0, 1, 0)
          }
        }
        if (nrow(Temp_Tl)==1) {
          for (IndexAvProblem in 1:length(ListAvProblemTypesAbbrev)) {
            Output[Index, paste0(ListAvProblemTypesAbbrev[IndexAvProblem], "_PRES_TL")]  <- ifelse(Temp_Tl[paste0(ListAvProblemTypesAbbrev[IndexAvProblem], "_LIKELIHOOD_MAX")]>0, 1, 0)
          }
        }
        if (nrow(Temp_Btl)==1) {
          for (IndexAvProblem in 1:length(ListAvProblemTypesAbbrev)) {
            Output[Index, paste0(ListAvProblemTypesAbbrev[IndexAvProblem], "_PRES_BTL")] <- ifelse(Temp_Btl[paste0(ListAvProblemTypesAbbrev[IndexAvProblem], "_LIKELIHOOD_MAX")]>0, 1, 0)
          }
        }

        DAYPREV <- as.character(Output$DAY0[(Output$REGION==Output$REGION[Index]) & (Output$PUBLISH_DATE==(Output$PUBLISH_DATE[Index]-1)) & (Output$ELEV == Output$ELEV[Index])])
        DAY0ALP <- as.character(Output$DAY0[(Output$REGION==Output$REGION[Index]) & (Output$PUBLISH_DATE==Output$PUBLISH_DATE[Index]) & (Output$ELEV == 'Alp')])
        DAY0TL  <- as.character(Output$DAY0[(Output$REGION==Output$REGION[Index]) & (Output$PUBLISH_DATE==Output$PUBLISH_DATE[Index]) & (Output$ELEV == 'Tl')])
        DAY0BTL <- as.character(Output$DAY0[(Output$REGION==Output$REGION[Index]) & (Output$PUBLISH_DATE==Output$PUBLISH_DATE[Index]) & (Output$ELEV == 'Btl')])

        if (length(DAYPREV)==0) { Output$DAYPREV[Index] <- NA } else { Output$DAYPREV[Index] <- DAYPREV }
        if (length(DAY0ALP)==0) { Output$DAY0ALP[Index] <- NA } else { Output$DAY0ALP[Index] <- DAY0ALP }
        if (length(DAY0TL)==0)  { Output$DAY0TL[Index]  <- NA } else { Output$DAY0TL[Index]  <- DAY0TL }
        if (length(DAY0BTL)==0) { Output$DAY0BTL[Index] <- NA } else { Output$DAY0BTL[Index] <- DAY0BTL }

      }

      Output$DAYPREV <- ordered(Output$DAYPREV, levels=levels(Output$DAY0))
      Output$DAY0ALP <- ordered(Output$DAY0ALP, levels=levels(Output$DAY0))
      Output$DAY0TL  <- ordered(Output$DAY0TL,  levels=levels(Output$DAY0))
      Output$DAY0BTL <- ordered(Output$DAY0BTL, levels=levels(Output$DAY0))

      for (IndexAvProblem in 1:length(ListAvProblemTypesAbbrev)) {
        Output[paste0(ListAvProblemTypesAbbrev[IndexAvProblem], "_PRES_ALP")][is.na(Output[paste0(ListAvProblemTypesAbbrev[IndexAvProblem], "_PRES_ALP")])] <- 0
        Output[paste0(ListAvProblemTypesAbbrev[IndexAvProblem], "_PRES_ALP")][Output[paste0(ListAvProblemTypesAbbrev[IndexAvProblem], "_PRES_ALP")] > 0] <- 1
        Output[paste0(ListAvProblemTypesAbbrev[IndexAvProblem], "_PRES_TL")][is.na(Output[paste0(ListAvProblemTypesAbbrev[IndexAvProblem], "_PRES_TL")])] <- 0
        Output[paste0(ListAvProblemTypesAbbrev[IndexAvProblem], "_PRES_TL")][Output[paste0(ListAvProblemTypesAbbrev[IndexAvProblem], "_PRES_TL")] > 0] <- 1
        Output[paste0(ListAvProblemTypesAbbrev[IndexAvProblem], "_PRES_BTL")][is.na(Output[paste0(ListAvProblemTypesAbbrev[IndexAvProblem], "_PRES_BTL")])] <- 0
        Output[paste0(ListAvProblemTypesAbbrev[IndexAvProblem], "_PRES_BTL")][Output[paste0(ListAvProblemTypesAbbrev[IndexAvProblem], "_PRES_BTL")] > 0] <- 1
      }


    } else if(AddDngRatingPrevDay) {
      print("Getting danger rating of previous day.")

      Output$DAYPREV <- NA

      for (Index in 1:nrow(Output)) {
        if((Index %% 500)==0) {print(paste("Processing record", Index, "of", nrow(Output), "..."))}

        DAYPREV <- as.character(Output$DAY0[(Output$REGION==Output$REGION[Index]) & (Output$PUBLISH_DATE==(Output$PUBLISH_DATE[Index]-1)) & (Output$ELEV == Output$ELEV[Index])])

        if (length(DAYPREV)==0) { Output$DAYPREV[Index] <- NA } else { Output$DAYPREV[Index] <- DAYPREV }
      }

      Output$DAYPREV <- ordered(Output$DAYPREV, levels=levels(Output$DAY0))

    } else if (AddDngRatingOtherElev) {

      print("Getting danger rating of other elevation bands.")

      Output$DAY0ALP <- NA
      Output$DAY0TL  <- NA
      Output$DAY0BTL <- NA

      for (Index in 1:nrow(Output)) {
        if((Index %% 500)==0) {print(paste("Processing record", Index, "of", nrow(Output), "..."))}

        DAYPREV <- as.character(Output$DAY0[(Output$REGION==Output$REGION[Index]) & (Output$PUBLISH_DATE==(Output$PUBLISH_DATE[Index]-1)) & (Output$ELEV == Output$ELEV[Index])])
        DAY0ALP <- as.character(Output$DAY0[(Output$REGION==Output$REGION[Index]) & (Output$PUBLISH_DATE==Output$PUBLISH_DATE[Index]) & (Output$ELEV == 'Alp')])
        DAY0TL  <- as.character(Output$DAY0[(Output$REGION==Output$REGION[Index]) & (Output$PUBLISH_DATE==Output$PUBLISH_DATE[Index]) & (Output$ELEV == 'Tl')])
        DAY0BTL <- as.character(Output$DAY0[(Output$REGION==Output$REGION[Index]) & (Output$PUBLISH_DATE==Output$PUBLISH_DATE[Index]) & (Output$ELEV == 'Btl')])

        if (length(DAYPREV)==0) { Output$DAYPREV[Index] <- NA } else { Output$DAYPREV[Index] <- DAYPREV }
        if (length(DAY0ALP)==0) { Output$DAY0ALP[Index] <- NA } else { Output$DAY0ALP[Index] <- DAY0ALP }
        if (length(DAY0TL)==0)  { Output$DAY0TL[Index]  <- NA } else { Output$DAY0TL[Index]  <- DAY0TL }
        if (length(DAY0BTL)==0) { Output$DAY0BTL[Index] <- NA } else { Output$DAY0BTL[Index] <- DAY0BTL }

      }

      Output$DAY0ALP <- ordered(Output$DAY0ALP, levels=levels(Output$DAY0))
      Output$DAY0TL  <- ordered(Output$DAY0TL,  levels=levels(Output$DAY0))
      Output$DAY0BTL <- ordered(Output$DAY0BTL, levels=levels(Output$DAY0))

    }

  ## LONG FORMAT
  ## ***********
  } else if (tolower(Format)=="long") {

    stop("Long format not implemented!")

  } else {

    stop("Format parameter can only be 'wide' or 'long'!")

  }

  ## Output
  return(Output)

}
