#' Get size of bulletin database
#'
#' Get size of bulletin database.
#' @param Agency Name of agency
#' @param DBFormat Name of database type. Needs to be either 'AvalX' (default), 'Bulletin' or 'HzdWebsite'
#' @param User Name of user
#' @param Verbose Switch for printing of query. Default value is FALSE.
#' @param SuppressDBWarnings Switch for turning warnings off. Default value is TRUE.
#' @return Size in MB
#'
#' @examples
#' require(SarpBulletinToolsPrivate)
#' require(SarpBulletinTools)
#'
#' DB_Size <- getBulletinDBSize("PkCan", "AvalX")
#' DB_Size <- getBulletinDBSize("AvCan", "HzdWebsite")
#'
#' @export

getBulletinDBSize <- function(Agency, DBFormat="AvalX", User="", Verbose=F, SuppressDBWarnings=T) {

  ## Preparation
  DBInfo <- getBulletinDBInfo(Agency, DBFormat)

  if (DBInfo$DBType=="MSSQL") {

    Query <- paste0("SELECT ",
                    "database_name = DB_NAME(database_id), ",
                    "log_size_mb = CAST(SUM(CASE WHEN type_desc = 'LOG' THEN size END) * 8. / 1024 AS DECIMAL(8,2)), ",
                    "row_size_mb = CAST(SUM(CASE WHEN type_desc = 'ROWS' THEN size END) * 8. / 1024 AS DECIMAL(8,2)), ",
                    "total_size_mb = CAST(SUM(size) * 8. / 1024 AS DECIMAL(8,2)) ",
                    "FROM sys.master_files WITH(NOWAIT) ",
                    "WHERE database_id = DB_ID() ",
                    "GROUP BY database_id")

  } else if (DBInfo$DBType=="MySQL") {

    Query <- paste0("SELECT ",
                    "table_schema AS 'database_name', ",
                    "ROUND(SUM(data_length + index_length) / 1024 / 1024,2) AS 'total_size_mb' ",
                    "FROM information_schema.TABLES GROUP BY table_schema;")

  } else {

    stop("Incorrect value for DBType. Currently only sypports 'MSSQL' and 'MySQL'!")

  }

  Result <- getRecordsFromQuery(Agency, Query, DBFormat, User, ReadOnly=T, Verbose, SuppressDBWarnings)
  DBSize <- paste(Result$total_size_mb[Result$database_name==DBInfo$DBName], "MB")

  return(DBSize)

}
