#' Gets list of bulletin ID with source database information.
#'
#' Gets list of bulletin ID with source database information.
#' @param Agency Name of agency (PkCan or AvCan)
#' @param Seasons List of seasons to be included in query.
#' @param StartDate Either publishing date of requested bulletin id or start date for time period if EndDate is provided as well.
#' @param EndDate End date for time period of requested bulletin ids.
#' @param Regions Name of bulletin region
#' @param BestDBSourceInt Dataframe with information about best data source DB to be used for query. Default is the BestDBSource dataframe that is included in the package.
#' @param User Name of user accessing the database. This name is used to record database interactions and to potentially access local resources (i.e., local folder JDBC driver)
#' @param SeasonType Parameter passed on to deriveAvSeasonInfo() from SARPGeneral package to specify start and end date of a season. Values are 'Operation' (Dec. 15 to Apr. 15) (default) or 'AvYear' (Oct. 1 to Sept. 30).
#' @param Verbose Switch for printing SQL query. Default value is FALSE.
#' @param SuppressDBWarnings Switch for turning warnings off. Default value is TRUE.
#' @return Dataframe with result
#'
#' @export

getBulletinIds <- function(Agency, Seasons=NA, StartDate=NA, EndDate=StartDate, Regions="All", BestDBSourceInt=BestDBSource, User="", SeasonType="Operation", Verbose=F, SuppressDBWarnings=T) {

  ## Check to ensure that either a Season or a StartDate are given but not both
  if (is.na(Seasons[1]) & is.na(StartDate)) {stop("A Season or StartDate value must be provided.")}
  if (!is.na(Seasons[1]) & !is.na(StartDate)) {stop("Only either a Season or a StartDate value can be provided.")}

  ## Convert dates to proper class if necessary
  if (!is.na(StartDate) & !("dates" %in% class(StartDate))) {StartDate <- dates(StartDate, format="y-m-d")}
  if (!is.na(StartDate) & !("dates" %in% class(EndDate)))   {EndDate   <- dates(EndDate, format="y-m-d")}

  ## Determine StartDate and EndDate if Seasons is provided
  if(!is.na(Seasons[1])) {
    StartSeason <- min(Seasons)
    EndSeason <- max(Seasons)
    StartDate <- deriveAvSeasonInfo(StartSeason, Type=SeasonType)$StartDate
    EndDate <- deriveAvSeasonInfo(EndSeason, Type=SeasonType)$EndDate
  }

  ## Determine Seasons to be queried if StartDate is provided
  if(!is.na(StartDate)) {
    StartSeason <- deriveAvSeasonFromDate(StartDate)
    EndSeason <- deriveAvSeasonFromDate(EndDate)
    Seasons <- c(StartSeason:EndSeason)
  }

  ## Check whether dates are within available seasons
  if (!(StartSeason %in% BestDBSourceInt$Season[as.character(BestDBSourceInt$Agency)==Agency])) {stop("StartDate not within available seasons.")}
  if (!(EndSeason %in% BestDBSourceInt$Season[as.character(BestDBSourceInt$Agency)==Agency])) {stop("EndDate not within available seasons.")}

  ## Determine best DB Source
  DBSource <- as.character(BestDBSourceInt$DB[(BestDBSourceInt$Season %in% Seasons) & (BestDBSourceInt$Agency==Agency)])

  ## Initialize Output
  Output <- NULL

  ## Loop for seasons
  for (Index_Season in 1:length(Seasons)) {

    ## Determine start date for query
    if (Seasons[Index_Season]==min(Seasons[Index_Season])) {
      StartDateQuery <- StartDate
    } else {
      StartDateQuery <- deriveAvSeasonInfo(Seasons[Index_Season], Type=SeasonType)$StartDate
    }

    ## Determine end date for query
    if (Seasons[Index_Season]==max(Seasons[Index_Season])) {
      EndDateQuery <- EndDate
    } else {
      EndDateQuery <- deriveAvSeasonInfo(Seasons[Index_Season], Type=SeasonType)$EndDate
    }


    ## Database query - AvalX
    if (DBSource[Index_Season]=="AvalX") {

      if(Agency=="PkCan") {
        TimeDiff <- -7
      } else {
        TimeDiff <- -8
      }

      Query <- paste0("SELECT ABS_BULLETIN.BULLETIN_GUID AS BULLETIN_ID, 'AvalX' AS SOURCE_DB ",
                      "FROM ABS_BULLETIN INNER JOIN ABS_REGION ON ABS_BULLETIN.REGION_ID = ABS_REGION.REGION_ID ",
                      "WHERE ABS_BULLETIN.STATUS_ID=1 and ",
                      "convert(date, DATEADD(hh, ", TimeDiff, ", ABS_BULLETIN.PUBLISH_DATE))>=", convertDateToSQL(StartDateQuery), " and ",
                      "convert(date, DATEADD(hh, ", TimeDiff, ", ABS_BULLETIN.PUBLISH_DATE))<=", convertDateToSQL(EndDateQuery))

      if (Regions[1]!="All") {
        Query <- addIDsToSQLQuery(Query, Regions, "ABS_REGION.DESC_EN", Connector = "and", InBrackets=T, WithQuotes=T)
      }

    } else if (DBSource[Index_Season]=="Bulletin") {

      Query <- paste0("SELECT max(BulletinMaster.BulletinID) AS BULLETIN_ID, 'Bulletin' AS SOURCE_DB  ",
                      "FROM BulletinMaster INNER JOIN BulletinRegion ON BulletinMaster.RegionID = BulletinRegion.RegionID ",
                      "WHERE BulletinMaster.Archived=1 and ",
                      "convert(date, BulletinMaster.DateIssued)>=", convertDateToSQL(StartDateQuery), " and ",
                      "convert(date, BulletinMaster.DateIssued)<=", convertDateToSQL(EndDateQuery))

      if (Regions[1]!="All") {
        Query <- addIDsToSQLQuery(Query, Regions, "BulletinRegion.Name", Connector = "and", InBrackets=T, WithQuotes=T)
      }

      Query <- paste0(Query, " ",
                      "Group by BulletinMaster.RegionID, convert(date, BulletinMaster.DateIssued) ",
                      "Order by count(BulletinMaster.BulletinID) desc")

    } else if (DBSource[Index_Season]=="HzdWebsite") {

      Query <- paste0("SELECT tbl_assessments.ID AS BULLETIN_ID, 'HzdWebsite' AS SOURCE_DB ",
                      "FROM tbl_assessments INNER JOIN tbl_regions ON tbl_regions.ID = tbl_assessments.RegionID ",
                      "WHERE tbl_assessments.`Show` = 1 and ",
                      "tbl_assessments.AssessDate>=", convertDateToSQL(StartDateQuery), " and ",
                      "tbl_assessments.AssessDate<=", convertDateToSQL(EndDateQuery))

      if (Regions[1]!="All") {
        Query <- addIDsToSQLQuery(Query, Regions, "tbl_regions.`Name`", Connector = "and", InBrackets=T, WithQuotes=T)
      }

    } else {

      stop(paste0("Database type '", DBSource[Index_Season], "' not supported!"))

    }

    ## Run query for season
    QueryResult <- getRecordsFromQuery(Agency, Query, DBSource[Index_Season], User, ReadOnly=T, Verbose, SuppressDBWarnings)

    ## Combine seasonal data frames
    if (!is.null(Output)) {
      Output <- rbind(Output, QueryResult)
    } else {
      Output <- QueryResult
    }

  }

  ## Finish output dataframe
  Output$SOURCE_DB <- factor(Output$SOURCE_DB)

  ## Return result
  return(Output)

}
