#' Gets bulletin region geometry as spatial dataframe
#'
#' Gets bulletin region geometry as spatial dataframe
#' @param RegionNames Single or array of bulletin region names. Default is 'All'
#' @param Agency Instead of RegionsNames, it is possible to provide the Agency name. Either AvCan or PkCan.
#' @param Season Single or array of season for bulletin regions.
#' @param AsCentroid Switch for whether returned geometry should be the centroid. Default value is F.
#' @param CoreRegionsOnly Switch for whether core regions should be returned only. Default value is T.
#' @param Verbose Switch for printing of query
#' @param SuppressDBWarnings Switch for supressing warning messages from postgresqlExecStatement. Turn on for debugging!
#'
#' @return Appropriate sp spatial objects (SpatialPoints, SpatialLines, SpatialPolygons)
#'
#' @export
#'
#' @examples
#' require(SarpBulletinTools)
#' require(SarpBulletinToolsPrivate)
#'
#' BulletinRegions <- c("Purcells", "Sea-to-Sky")
#' Season <- "2016"
#'
#' BulletinRegionSPDF <- getBulletinRegionGeomSPDF(RegionNames=BulletinRegions, Season=Season)
#' plot(BulletinRegionSPDF)
#'
#' BulletinRegionSPDF_AsCentroid <- getBulletinRegionGeomSPDF(RegionNames=BulletinRegions, Season=Season, AsCentroid = T)
#' plot(BulletinRegionSPDF_AsCentroid, pch=21, cex=5, bg="red", add=T)
#'
#' BulletinRegionSPDF <- getBulletinRegionGeomSPDF(Agency="PkCan", Season=Season)
#' plot(BulletinRegionSPDF)


getBulletinRegionGeomSPDF <- function(RegionNames="All", Agency=NA, Season, AsCentroid=F, CoreRegionsOnly=T, Verbose=F, SuppressDBWarnings=T) {

  require(sp)

  ## Initial check
  if(is.na(RegionNames[1]) & is.na(Agency)) { stop("values for either RegionNames or Agency must be provided!")}

  # ## Get RegionNames is necessary
  # if(!is.na(Agency)) {
  #   RegionNames <- getListForecastRegions(Agency)$REGION_NAME
  # }

  ## Supporting functions
  getBulletinRegionGeom <- function(RegionNames, Agency=NA, Season, CoreRegionsOnly=T, AsCentroid=F, Verbose=F, SuppressDBWarnings=T) {

    ## Build query - Base
    if (AsCentroid) {
      Query <- paste0("SELECT uuid, ST_AsText(ST_Centroid(geom)) AS wkt, ST_SRID(geom) AS srid FROM bulletin.regions")
    } else {
      Query <- paste0("SELECT uuid, ST_AsText(geom) AS wkt, ST_SRID(geom) AS srid FROM bulletin.regions")
    }

    ## Add beginning of where clause
    Query <- paste0(Query, " WHERE ")

    ## Build query - Agency
    if (!is.na(Agency)) {

      Query <- paste0(Query, "(agency='", Agency, "') AND ")

    ## Build query - Region Names
    } else {

      ## Build query - Region names selection
      if(RegionNames[1] != 'All') {
        if(length(RegionNames)==1) {
            Query <- paste0(Query, "(name=", convertStringToSQL(RegionNames), ")")
        } else {
          Query <- paste0(Query, "(")
          for (Index_RegionNames in 1:length(RegionNames)) {
            Query <- paste0(Query, "name=", convertStringToSQL(RegionNames[Index_RegionNames]), " OR ")
          }
          Query <- paste0(substr(Query, 1, nchar(Query)-4), ") ")
        }

        Query <- paste0(Query, "AND ")

      }

    }

    ## Start Season
    Query <- paste0(Query, "(season_start <=", Season, ") ")

    ## End season
    Query <- paste0(Query, "AND (season_end >=", Season, " OR season_end IS NULL)")

    ## Core regions only
    if (CoreRegionsOnly) {
      Query <- paste0(Query, " AND (coreregion IS TRUE)")
    }


    ## Send query to database
    ResultDF <- getRecordsFromQueryBulletinRegionsDB(Query=Query, Verbose=Verbose, SuppressDBWarnings=SuppressDBWarnings)

    if(nrow(ResultDF)>0) {

      ## Create spatial objects
      for (Index_Rows in 1:nrow(ResultDF)) {

        ## Convert WKT to spatial object (requires rgeos package)
        SpatialObject <- rgeos::readWKT(ResultDF$wkt[Index_Rows], id=ResultDF$uuid[Index_Rows])

        ## Add projection information (SRID=4617)
        proj4string(SpatialObject) <- "+proj=longlat +ellps=GRS80 +towgs84=0,0,0,0,0,0,0 +no_defs"

        ## Combining spatial objects (requires maptools package)
        if (Index_Rows==1) {
          SpatialObjects <- SpatialObject
        } else {
          SpatialObjects <- maptools::spRbind(SpatialObjects, SpatialObject)
        }
      }
      rm(Index_Rows)

    } else {

      SpatialObjects <- NULL

    }

    ## Return final result
    return (SpatialObjects)

  }


  ## Retrieve spatial objects
  SpatialObjects <- getBulletinRegionGeom(RegionNames=RegionNames, Season=Season, Agency=Agency, CoreRegionsOnly=CoreRegionsOnly, AsCentroid=AsCentroid, Verbose=Verbose, SuppressDBWarnings=SuppressDBWarnings)

  if (!is.null(SpatialObjects)) {

    ## Retrieve attributes
    Query <- "SELECT uuid, name, description, season_start, season_end, agency, mtnrange, snowclimate, comment, sort_order, color, flag_delete, dtcre, dtmod FROM bulletin.regions where "


    if (class(SpatialObjects)=='SpatialPolygons') {
      NumRegions <- length(SpatialObjects@polygons)
      for (Region_Index in 1:NumRegions) {
        Query <- paste0(Query, "(uuid='", SpatialObjects@polygons[[Region_Index]]@ID, "')")
        if (Region_Index < NumRegions) {
          Query <- paste0(Query, " OR ")
        }
      }
    } else if (class(SpatialObjects)=='SpatialPoints') {
      NumRegions <- length(row.names(SpatialObjects@coords))
      for (Region_Index in 1:NumRegions) {
        Query <- paste0(Query, "(uuid='", row.names(SpatialObjects@coords)[Region_Index], "')")
        if (Region_Index < NumRegions) {
          Query <- paste0(Query, " OR ")
        }
      }
    } else {
      stop(paste0("Class '", class(SpatialObjects), "' not supported!"), immediate. = T)
    }

    AttributesDF <- getRecordsFromQueryBulletinRegionsDB(Query, Verbose=Verbose, SuppressDBWarnings=SuppressDBWarnings)
    row.names(AttributesDF) <- AttributesDF$uuid

    ## Order
    ## AttributesDF <- AttributesDF[order(AttributesDF$sort_order),]

    ## Create spatial objects dataframe
    if (class(SpatialObjects)=="SpatialPoints") {
      SPDF <- SpatialPointsDataFrame(SpatialObjects, AttributesDF)
    } else if (class(SpatialObjects)=="SpatialPolygons") {
      SPDF <- SpatialPolygonsDataFrame(SpatialObjects, AttributesDF)
    } else {
      SPDF <- NULL
      warning(paste0("Class '", class(SpatialObjects), "' not supported."), immediate. = T)
    }

  } else {

    SPDF <- NULL

  }

  ## Return SPDF
  return(SPDF)
}
