#' Gets bulletin information in a list of dataframes.
#'
#' Gets bulletin information in a list of dataframes.
#' @param Agency Name of agency (PkCan or AvCan)
#' @param Seasons List of seasons to be included in query.
#' @param StartDate Either publishing date of requested bulletin id or start date for time period if EndDate is provided as well.
#' @param EndDate End date for time period of requested bulletin ids.
#' @param Regions Name of bulletin region
#' @param FromRDataFile Switch whether data should be retrieved from RData file posted on http://data.avalancheresearch.ca instead of the database. Default is False.
#' @param BestDBSourceInt Dataframe with information about best data source DB to be used for query. Default is the BestDBSource dataframe that is included in the package.
#' @param User Name of user accessing the database. This name is used to record database interactions and to potentially access local resources (i.e., local folder JDBC driver)
#' @param SeasonType Parameter passed on to deriveAvSeasonInfo() from SARPGeneral package to specify start and end date of a season. Values are 'Operation' (Dec. 15 to Apr. 15) (default) or 'AvYear' (Oct. 1 to Sept. 30).
#' @param GetOnlyFinalBulletins Switch for specifying whether to only include final bulletins in the dataframe. When a bulletin is published twice, only the last one is included. Default is TRUE.
#' @param Verbose Switch for printing SQL queries. Default value is FALSE.
#' @param SuppressDBWarnings Switch for turning warnings off. Default value is TRUE.
#' @return Object of type Bulletins
#'
#' @examples
#'
#' require(SarpBulletinTools)
#' require(SarpBulletinToolsPrivate)
#'
#' ## Get single bulletin from single region
#' Agency <- "AvCan"
#' Date <- "2015-02-05"
#' Region <- "Kootenay Boundary"
#' Bulletin <- getBulletins(Agency, StartDate=Date, Regions=Region)
#' summary(Bulletin)
#'
#' ## Get bulletins from single region for date range
#' Agency <- "AvCan"
#' StartDate <- "2015-02-05"
#' EndDate <- "2015-02-12"
#' Region <- "Kootenay Boundary"
#' Bulletin <- getBulletins(Agency, StartDate=StartDate, EndDate=EndDate, Regions=Region)
#' summary(Bulletin)
#'
#' ## Get bulletins from single region for entire season
#' Agency <- "AvCan"
#' Season <- 2015
#' Region <- "Kootenay Boundary"
#' Bulletin <- getBulletins(Agency, Seasons=Season, Regions=Region)
#' summary(Bulletin)
#'
#' ## Get bulletins from single region for multiple seasons
#' Agency <- "AvCan"
#' Seasons <- c(2013:2015)
#' Region <- "Kootenay Boundary"
#' Bulletin <- getBulletins(Agency, Seasons=Seasons, Regions=Region)
#' summary(Bulletin)
#'
#' ## Get bulletins from multiple regions for single date
#' Agency <- "AvCan"
#' Date <- "2015-02-05"
#' Regions <- c("South Coast - Inland", "Sea-to-Sky")
#' Bulletin <- getBulletins(Agency, StartDate=Date, Regions=Regions)
#' summary(Bulletin)
#'
#' ## Get bulletins from all regions for single date
#' Agency <- "AvCan"
#' Date <- "2015-02-05"
#' Bulletin <- getBulletins(Agency, StartDate=Date)
#' summary(Bulletin)
#'
#' ## Get Glacier bulletins from database
#' Agency <- "PkCan"
#' Season <- 2017
#' Region <- "Glacier"
#' Bulletin <- getBulletins(Agency, Seasons=Season, Regions=Region)
#' summary(Bulletin)
#'
#' ## Get Glacier bulletins from RData file instead
#' Bulletin <- getBulletins(Agency, Seasons=Season, Regions=Region, FromRDataFile = T)
#' summary(Bulletin)
#'
#' @export

getBulletins <- function(Agency, Seasons=NA, StartDate=NA, EndDate=StartDate, Regions="All", FromRDataFile = F, BestDBSourceInt=BestDBSource, User="", SeasonType="Operation", GetOnlyFinalBulletins=TRUE, Verbose=F, SuppressDBWarnings=T) {

  ## ========================= ##
  ## GENERAL SUPPORT FUNCTIONS ##
  ## ========================= ##

  ## Add ID prefix to TblAvProblems
  addIDPrefixToTblAvProblem <- function (TblAvProblem, Prefix) {

    if(nrow(TblAvProblem)>0) {
      TblAvProblem$BULLETIN_ID <- paste0(Prefix, "_", TblAvProblem$BULLETIN_ID)
      TblAvProblem$PROBLEM_ID  <- paste0(Prefix, "_", TblAvProblem$PROBLEM_ID)
    }
    return(TblAvProblem)

  }

  ## Add ID prefix to TblBulletin
  addIDPrefixToTblBulletin <- function (TblBulletin, Prefix) {

    if(nrow(TblBulletin)>0) {
      TblBulletin$BULLETIN_ID <- paste0(Prefix, "_", TblBulletin$BULLETIN_ID)
    }
    return(TblBulletin)

  }

  ## Add ID prefix to TblDngRating
  addIDPrefixToTblDngRating <- function (TblDngRating, Prefix) {

    if(nrow(TblDngRating)>0) {
      TblDngRating$BULLETIN_ID <- paste0(Prefix, "_", TblDngRating$BULLETIN_ID)
    }
    return(TblDngRating)

  }

  ## Merges tables from three databases
  mergeTables <- function (TblName) {

    TblName_AvalX <- paste0(TblName, "_AvalX")
    TblName_Bull  <- paste0(TblName, "_Bulletin")
    TblName_HzdW  <- paste0(TblName, "_HzdWeb")

    if(exists(TblName_AvalX)) {TblOutput <- get(TblName_AvalX)}

    if(exists(TblName_Bull)) {
      if(exists("TblOutput")) {
        TblOutput <- rbind(TblOutput, get(TblName_Bull))
      } else {
        TblOutput <- get(TblName_Bull)
      }
    }

    if(exists(TblName_HzdW)) {
      if(exists("TblOutput")) {
        TblOutput <- rbind(TblOutput, get(TblName_HzdW))
      } else {
        TblOutput <- get(TblName_HzdW)
      }
    }

    if(exists("TblOutput")) {
      return(TblOutput)
    } else {
      return(NULL)
    }

  }

  ## Format TblBulletin
  formatTblBulletin <- function(TblBulletin) {
    if(class(TblBulletin)=="data.frame") {
      if (nrow(TblBulletin)>0) {
        TblBulletin$SOURCE_DB <- factor(as.character(TblBulletin$SOURCE_DB))
        TblBulletin$PUBLISH_DATE <- dates(TblBulletin$PUBLISH_DATE, format="y-m-d")
        TblBulletin$SEASON <- deriveAvSeasonFromDate(TblBulletin$PUBLISH_DATE)
        TblBulletin$PUBLISH_TIME <- times(substr(TblBulletin$PUBLISH_TIME, 12, 19))
        TblBulletin$VALID_UNTIL[TblBulletin$VALID_UNTIL=="0000-00-00"] <- NA
        TblBulletin$VALID_UNTIL <- dates(TblBulletin$VALID_UNTIL, format="y-m-d")
        TblBulletin$REGION <- factor(as.character(TblBulletin$REGION))
        TblBulletin$FORECASTER <- tolower(TblBulletin$FORECASTER)
        TblBulletin <- TblBulletin[order(TblBulletin$PUBLISH_DATE),]
      }
    }
    return(TblBulletin)
  }

  ## Format TblAvProblem
  formatTblAvProblem <- function(TblAvProblem) {
    if(class(TblAvProblem)=="data.frame") {
      if (nrow(TblAvProblem)) {
        TblAvProblem$CHARACTER <- factor(TblAvProblem$CHARACTER, levels=ListAvProblemTypes)
        TblAvProblem$DISTRIBUTION <- ordered(TblAvProblem$DISTRIBUTION, levels=c("Isolated", "Specific", "Widespread"))
        TblAvProblem$SENSITIVITY <- ordered(TblAvProblem$SENSITIVITY, levels=c("Unreactive", "Stubborn", "Touchy", "Very Touchy"))
      }
    }
    return(TblAvProblem)
  }

  ## Format TblDngRating
  formatTblDngRating <- function(TblDngRating) {
    if(class(TblDngRating)=="data.frame") {
      if (nrow(TblDngRating)) {

        TblDngRating$DAY0 <- as.character(TblDngRating$DAY0)
        TblDngRating$DAY0[TblDngRating$DAY0=="'Spring'"] <- "Spring"
        TblDngRating$DAY0[TblDngRating$DAY0=="n/a"] <- "No Rating"
        TblDngRating$DAY0[is.na(TblDngRating$DAY0)] <- "No Rating"
        TblDngRating$DAY0 <- ordered(TblDngRating$DAY0, levels=c("No Rating", "Spring", "Low", "Moderate", "Considerable", "High", "Extreme"))

        TblDngRating$DAY1 <- as.character(TblDngRating$DAY1)
        TblDngRating$DAY1[TblDngRating$DAY1=="'Spring'"] <- "Spring"
        TblDngRating$DAY1[TblDngRating$DAY1=="n/a"] <- "No Rating"
        TblDngRating$DAY1[is.na(TblDngRating$DAY1)] <- "No Rating"
        TblDngRating$DAY1 <- ordered(TblDngRating$DAY1, levels=c("No Rating", "Spring", "Low", "Moderate", "Considerable", "High", "Extreme"))

        TblDngRating$DAY2 <- as.character(TblDngRating$DAY2)
        TblDngRating$DAY2[TblDngRating$DAY2=="'Spring'"] <- "Spring"
        TblDngRating$DAY2[TblDngRating$DAY2=="n/a"] <- "No Rating"
        TblDngRating$DAY2[is.na(TblDngRating$DAY2)] <- "No Rating"
        TblDngRating$DAY2 <- ordered(TblDngRating$DAY2, levels=c("No Rating", "Spring", "Low", "Moderate", "Considerable", "High", "Extreme"))

        TblDngRating$DAY3 <- as.character(TblDngRating$DAY3)
        TblDngRating$DAY3[TblDngRating$DAY3=="'Spring'"] <- "Spring"
        TblDngRating$DAY3[TblDngRating$DAY3=="n/a"] <- "No Rating"
        TblDngRating$DAY3[is.na(TblDngRating$DAY3)] <- "No Rating"
        TblDngRating$DAY3 <- ordered(TblDngRating$DAY3, levels=c("No Rating", "Spring", "Low", "Moderate", "Considerable", "High", "Extreme"))
      }
    }
    return(TblDngRating)
  }

  ## Remove white space from string
  trimWhiteSpace <- function (x) gsub("^\\s+|\\s+$", "", x)

  ## =================================== ##
  ## SCRIPT FOR RETRIEVING BULLETIN DATA ##
  ## =================================== ##

  ## Convert dates to proper class if necessary
  if (!is.na(StartDate) & !("dates" %in% class(StartDate))) {StartDate <- dates(StartDate, format="y-m-d")}
  if (!is.na(StartDate) & !("dates" %in% class(EndDate)))   {EndDate   <- dates(EndDate, format="y-m-d")}


  ## *********************************
  ## LOADING BULLETINS FROM RDATA FILE
  ## *********************************

  if (FromRDataFile) {

    ## Create Bulletins object from RData file
    load(url("http://data.avalancheresearch.ca/AllBulletins_2010To2018.RData"))

    ## Filtering Bulletins
    if (!is.na(Regions) & !is.na(Seasons)) {
      BullIDs <- Bulletins$Bulletins$BULLETIN_ID[(Bulletins$Bulletins$REGION %in% Regions) & (Bulletins$Bulletins$SEASON %in% Seasons)]
      Output  <- extractFromBulletinObj(Bulletins, ByBulletinID = BullIDs)
    } else if (!is.na(Regions)) {
      BullIDs <- Bulletins$Bulletins$BULLETIN_ID[(Bulletins$Bulletins$REGION %in% Regions)]
      Output  <- extractFromBulletinObj(Bulletins, ByBulletinID = BullIDs)
    } else if (!is.na(Seasons)) {
      BullIDs <- Bulletins$Bulletins$BULLETIN_ID[(Bulletins$Bulletins$SEASON %in% Seasons)]
      Output  <- extractFromBulletinObj(Bulletins, ByBulletinID = BullIDs)
    } else {
      warning("Only Season and Region filter implemented when loading Bulletins from RData file!")
      Output <- Bulletins
    }

    ## Reformatting factors
    Output$Bulletins$REGION <- factor(Output$Bulletins$REGION)
    Output$Bulletins$AGENCY <- factor(Output$Bulletins$AGENCY)



  ## *******************************
  ## LOADING BULLETINS FROM DATABASE
  ## *******************************

  } else {


    ## ********************
    ## GETTING BULLETIN IDS
    ## ********************

    print("Determining database for requested bulletins ...")

    ## Get IDs of bulletins for given date range and forecast regions
    BulletinIDs <- getBulletinIds(Agency, Seasons, StartDate, EndDate, Regions, BestDBSourceInt, User, SeasonType, Verbose, SuppressDBWarnings)

    ## Create DB specific lists
    BulletinIDs_AvalX <- BulletinIDs$BULLETIN_ID[BulletinIDs$SOURCE_DB=="AvalX"]
    BulletinIDs_Bulletin <- BulletinIDs$BULLETIN_ID[BulletinIDs$SOURCE_DB=="Bulletin"]
    BulletinIDs_HzdWeb <- BulletinIDs$BULLETIN_ID[BulletinIDs$SOURCE_DB=="HzdWebsite"]


    ## **************
    ## AVALX DATABASE
    ## **************

    if (length(BulletinIDs_AvalX) > 0) {

      ## AVALX - BULLETIN TABLE
      ## **********************

      print("Getting bulletin table from AvalX database ...")

      if(Agency=="PkCan") {
        TimeDiff <- -7
      } else {
        TimeDiff <- -8
      }

      ## Getting bulletin table
      Query <- paste0("SELECT ABS_BULLETIN.BULLETIN_GUID AS BULLETIN_ID, ",
                             "'", Agency, "' AS AGENCY, ",
                             "'AvalX' AS SOURCE_DB, ",
                             "NULL AS SEASON, ",
                             "ABS_BULLETIN.STATUS_ID, ",
                             "convert(date, DATEADD(hh, ", TimeDiff, ", ABS_BULLETIN.PUBLISH_DATE)) AS PUBLISH_DATE, ",
                             "DATEADD(hh, ", TimeDiff, ", ABS_BULLETIN.PUBLISH_DATE) AS PUBLISH_TIME, ",
                             "convert(date, DATEADD(hh, ", TimeDiff, ", ABS_BULLETIN.VALID_UNTIL)) AS VALID_UNTIL, ",
                             "ABS_REGION.DESC_EN AS REGION, ",
                             "ABS_BULLETIN.MODIFIED_STAFF AS FORECASTER ",
                      "FROM ABS_BULLETIN INNER JOIN ABS_REGION ON ABS_BULLETIN.REGION_ID = ABS_REGION.REGION_ID")

      Query <- addIDsToSQLQuery(Query, BulletinIDs_AvalX, "ABS_BULLETIN.BULLETIN_GUID", WithQuotes=T)

      TblBulletin_AvalX <- getRecordsFromQuery(Agency, Query, "AvalX", User, ReadOnly=T, Verbose, SuppressDBWarnings)
      rm(Query)

      ## AVALX - AVALANCHE PROBLEM TABLE
      ## *******************************

      print("Getting avalanche problem table from AvalX database ...")

      ## Function for creating base query
      makeQueryBase_AvProb_AvalX <- function (ElevBand) {

        if (ElevBand=="ALP") {ElevBand <- "Alpine"}
        else if (ElevBand=="TL") {ElevBand <- "Treeline"}
        else if (ElevBand=="BTL") {ElevBand <- "Below Treeline"}

        Query <- paste0("SELECT ABS_PROBLEM.BULLETIN_GUID AS BULLETIN_ID, ",
                               "ABS_PROBLEM.PROBLEM_ID, ",
                               "ABS_PROBLEM.SORT_ORDER, ",
                               "ABS_PROB_CHARACTER.DESC_EN AS 'CHARACTER', ",
                               "TBL_ASPECT_N.N AS ASP_N, ",
                               "TBL_ASPECT_NE.NE AS ASP_NE, ",
                               "TBL_ASPECT_E.E AS ASP_E, ",
                               "TBL_ASPECT_SE.SE AS ASP_SE, ",
                               "TBL_ASPECT_S.S AS ASP_S, ",
                               "TBL_ASPECT_SW.SW AS ASP_SW, ",
                               "TBL_ASPECT_W.W AS ASP_W, ",
                               "TBL_ASPECT_NW.NW AS ASP_NW, ",
                               "ABS_PROB_DISTRIBUTION.DESC_EN AS DISTRIBUTION, ",
                               "ABS_PROB_SENSITIVITY.DESC_EN AS SENSITIVITY, ",
                               "ABS_PROBLEM.LIKELIHOOD_MIN, ",
                               "ABS_PROBLEM.LIKELIHOOD_TYPICAL AS LIKELIHOOD_TYP, ",
                               "ABS_PROBLEM.LIKELIHOOD_MAX, ",
                               "ABS_PROBLEM.SIZE_MIN, ",
                               "ABS_PROBLEM.SIZE_TYPICAL AS SIZE_TYP, ",
                               "ABS_PROBLEM.SIZE_MAX, ",
                               "ABS_PROB_ELEVATION.CONTRIBUTION AS CONTRI, ",
                               "ABS_PROBLEM.PROBLEM_2_COMMENT AS COMMENT ",
                        "FROM ABS_PROBLEM ",
                             "INNER JOIN ABS_PROB_ELEVATION ON ABS_PROBLEM.PROBLEM_ID = ABS_PROB_ELEVATION.PROBLEM_ID ",
                             "INNER JOIN ABS_PROB_ZONE_ELEVATION ON ABS_PROB_ELEVATION.ELEVATION_ID = ABS_PROB_ZONE_ELEVATION.ELEVATION_ID ",
                             "LEFT OUTER JOIN (SELECT PROBLEM_ID, 1 AS N  FROM ABS_PROB_ASPECT WHERE (ASPECT_ID = 1)) AS TBL_ASPECT_N  ON ABS_PROBLEM.PROBLEM_ID = TBL_ASPECT_N.PROBLEM_ID ",
                             "LEFT OUTER JOIN (SELECT PROBLEM_ID, 1 AS NE FROM ABS_PROB_ASPECT WHERE (ASPECT_ID = 2)) AS TBL_ASPECT_NE ON ABS_PROBLEM.PROBLEM_ID = TBL_ASPECT_NE.PROBLEM_ID ",
                             "LEFT OUTER JOIN (SELECT PROBLEM_ID, 1 AS E  FROM ABS_PROB_ASPECT WHERE (ASPECT_ID = 3)) AS TBL_ASPECT_E  ON ABS_PROBLEM.PROBLEM_ID = TBL_ASPECT_E.PROBLEM_ID ",
                             "LEFT OUTER JOIN (SELECT PROBLEM_ID, 1 AS SE FROM ABS_PROB_ASPECT WHERE (ASPECT_ID = 4)) AS TBL_ASPECT_SE ON ABS_PROBLEM.PROBLEM_ID = TBL_ASPECT_SE.PROBLEM_ID ",
                             "LEFT OUTER JOIN (SELECT PROBLEM_ID, 1 AS S  FROM ABS_PROB_ASPECT WHERE (ASPECT_ID = 5)) AS TBL_ASPECT_S  ON ABS_PROBLEM.PROBLEM_ID = TBL_ASPECT_S.PROBLEM_ID ",
                             "LEFT OUTER JOIN (SELECT PROBLEM_ID, 1 AS SW FROM ABS_PROB_ASPECT WHERE (ASPECT_ID = 6)) AS TBL_ASPECT_SW ON ABS_PROBLEM.PROBLEM_ID = TBL_ASPECT_SW.PROBLEM_ID ",
                             "LEFT OUTER JOIN (SELECT PROBLEM_ID, 1 AS W  FROM ABS_PROB_ASPECT WHERE (ASPECT_ID = 7)) AS TBL_ASPECT_W  ON ABS_PROBLEM.PROBLEM_ID = TBL_ASPECT_W.PROBLEM_ID ",
                             "LEFT OUTER JOIN (SELECT PROBLEM_ID, 1 AS NW FROM ABS_PROB_ASPECT WHERE (ASPECT_ID = 8)) AS TBL_ASPECT_NW ON ABS_PROBLEM.PROBLEM_ID = TBL_ASPECT_NW.PROBLEM_ID ",
                             "LEFT OUTER JOIN ABS_PROB_CHARACTER ON ABS_PROBLEM.CHARACTER_ID = ABS_PROB_CHARACTER.CHARACTER_ID ",
                             "LEFT OUTER JOIN ABS_PROB_DISTRIBUTION ON ABS_PROBLEM.DISTRIBUTION_ID = ABS_PROB_DISTRIBUTION.DISTRIBUTION_ID ",
                             "LEFT OUTER JOIN ABS_PROB_SENSITIVITY ON ABS_PROBLEM.SENSITIVITY_ID = ABS_PROB_SENSITIVITY.SENSITIVITY_ID ",
                        "WHERE ABS_PROBLEM.SHOW_TO_PUBLIC_IND=1 AND ABS_PROB_ZONE_ELEVATION.DESC_EN='", ElevBand, "'")

        return(Query)

      }

      ## AvProblem Tbl for Alpine elevation band
      Query <- makeQueryBase_AvProb_AvalX("ALP")
      Query <- addIDsToSQLQuery(Query, BulletinIDs_AvalX, "ABS_PROBLEM.BULLETIN_GUID", Connector="AND", InBrackets=T, WithQuotes=T)
      TblAvProblem_Alp_AvalX <- getRecordsFromQuery(Agency, Query, "AvalX", User, ReadOnly=T, Verbose, SuppressDBWarnings)
      rm(Query)

      ## AvProblem Tbl for Alpine elevation band
      Query <- makeQueryBase_AvProb_AvalX("TL")
      Query <- addIDsToSQLQuery(Query, BulletinIDs_AvalX, "ABS_PROBLEM.BULLETIN_GUID", Connector="AND", InBrackets=T, WithQuotes=T)
      TblAvProblem_Tl_AvalX <- getRecordsFromQuery(Agency, Query, "AvalX", User, ReadOnly=T, Verbose, SuppressDBWarnings)
      rm(Query)

      ## AvProblem Tbl for Alpine elevation band
      Query <- makeQueryBase_AvProb_AvalX("BTL")
      Query <- addIDsToSQLQuery(Query, BulletinIDs_AvalX, "ABS_PROBLEM.BULLETIN_GUID", Connector="AND", InBrackets=T, WithQuotes=T)
      TblAvProblem_Btl_AvalX <- getRecordsFromQuery(Agency, Query, "AvalX", User, ReadOnly=T, Verbose, SuppressDBWarnings)
      rm(Query)

      ## AvalX-specific formating of AvProblem Tbls
      TblAvProblem_Alp_AvalX$COMMENT <- substr(TblAvProblem_Alp_AvalX$COMMENT, 291, nchar(TblAvProblem_Alp_AvalX$COMMENT)-11)
      TblAvProblem_Tl_AvalX$COMMENT  <- substr(TblAvProblem_Tl_AvalX$COMMENT,  291, nchar(TblAvProblem_Tl_AvalX$COMMENT)-11)
      TblAvProblem_Btl_AvalX$COMMENT <- substr(TblAvProblem_Btl_AvalX$COMMENT, 291, nchar(TblAvProblem_Btl_AvalX$COMMENT)-11)


      ## AVALX: DANGER RATING TABLE
      ## **************************

      print("Getting danger rating table from AvalX database ...")

      ## Function for creating base query
      makeQueryBase_DngRating_AvalX <- function (ElevBand) {

        if (ElevBand=="ALP") {ElevBand <- "Alpine"}
        else if (ElevBand=="TL") {ElevBand <- "Treeline"}
        else if (ElevBand=="BTL") {ElevBand <- "Below Treeline"}

        Query <- paste0("SELECT ABS_BULLETIN_DANGER_RATING.BULLETIN_GUID AS BULLETIN_ID, ",
                               "DR_DAY0.DESC_EN AS DAY0, ",
                               "DR_DAY1.DESC_EN AS DAY1, ",
                               "DR_DAY2.DESC_EN AS DAY2, ",
                               "DR_DAY3.DESC_EN AS DAY3 ",
                        "FROM ABS_BULLETIN_DANGER_RATING ",
                             "INNER JOIN ABS_PROB_ZONE_ELEVATION ON ABS_BULLETIN_DANGER_RATING.ELEVATION_ID = ABS_PROB_ZONE_ELEVATION.ELEVATION_ID ",
                             "LEFT OUTER JOIN ABS_DANGER_RATING AS DR_DAY0 ON ABS_BULLETIN_DANGER_RATING.DANGER_RATING_DAY0_ID = DR_DAY0.DANGER_RATING_ID ",
                             "LEFT OUTER JOIN ABS_DANGER_RATING AS DR_DAY1 ON ABS_BULLETIN_DANGER_RATING.DANGER_RATING_DAY1_ID = DR_DAY1.DANGER_RATING_ID ",
                             "LEFT OUTER JOIN ABS_DANGER_RATING AS DR_DAY2 ON ABS_BULLETIN_DANGER_RATING.DANGER_RATING_DAY2_ID = DR_DAY2.DANGER_RATING_ID ",
                             "LEFT OUTER JOIN ABS_DANGER_RATING AS DR_DAY3 ON ABS_BULLETIN_DANGER_RATING.DANGER_RATING_DAY3_ID = DR_DAY3.DANGER_RATING_ID ",
                        "WHERE ABS_PROB_ZONE_ELEVATION.DESC_EN='", ElevBand, "'")

         return(Query)

      }

      ## DangerRating Tbl for Alpine elevation band
      Query <- makeQueryBase_DngRating_AvalX("ALP")
      Query <- addIDsToSQLQuery(Query, BulletinIDs_AvalX, "ABS_BULLETIN_DANGER_RATING.BULLETIN_GUID", Connector="AND", InBrackets=T, WithQuotes=T)
      TblDngRating_Alp_AvalX <- getRecordsFromQuery(Agency, Query, "AvalX", User, ReadOnly=T, Verbose, SuppressDBWarnings)
      rm(Query)

      ## DangerRating Tbl for Treeline elevation band
      Query <- makeQueryBase_DngRating_AvalX("TL")
      Query <- addIDsToSQLQuery(Query, BulletinIDs_AvalX, "ABS_BULLETIN_DANGER_RATING.BULLETIN_GUID", Connector="AND", InBrackets=T, WithQuotes=T)
      TblDngRating_Tl_AvalX <- getRecordsFromQuery(Agency, Query, "AvalX", User, ReadOnly=T, Verbose, SuppressDBWarnings)
      rm(Query)

      ## DangerRating Tbl for Below treeline elevation band
      Query <- makeQueryBase_DngRating_AvalX("BTL")
      Query <- addIDsToSQLQuery(Query, BulletinIDs_AvalX, "ABS_BULLETIN_DANGER_RATING.BULLETIN_GUID", Connector="AND", InBrackets=T, WithQuotes=T)
      TblDngRating_Btl_AvalX <- getRecordsFromQuery(Agency, Query, "AvalX", User, ReadOnly=T, Verbose, SuppressDBWarnings)
      rm(Query)


      ## AVALX - ADDING PREFIXES TO IDS
      ## ******************************

      TblBulletin_AvalX <- addIDPrefixToTblBulletin(TblBulletin_AvalX, paste0(Agency, "AvalX"))

      TblAvProblem_Alp_AvalX <- addIDPrefixToTblAvProblem(TblAvProblem_Alp_AvalX, paste0(Agency, "AvalX"))
      TblAvProblem_Tl_AvalX  <- addIDPrefixToTblAvProblem(TblAvProblem_Tl_AvalX, paste0(Agency, "AvalX"))
      TblAvProblem_Btl_AvalX <- addIDPrefixToTblAvProblem(TblAvProblem_Btl_AvalX, paste0(Agency, "AvalX"))

      TblDngRating_Alp_AvalX <- addIDPrefixToTblDngRating(TblDngRating_Alp_AvalX, paste0(Agency, "AvalX"))
      TblDngRating_Tl_AvalX  <- addIDPrefixToTblDngRating(TblDngRating_Tl_AvalX, paste0(Agency, "AvalX"))
      TblDngRating_Btl_AvalX <- addIDPrefixToTblDngRating(TblDngRating_Btl_AvalX, paste0(Agency, "AvalX"))

    }


    ## *****************
    ## BULLETIN DATABASE
    ## *****************

      if (length(BulletinIDs_Bulletin) > 0) {

      ## BULLETIN - BULLETIN TABLE
      ## *************************

      print("Getting bulletin table from bulletin database ...")

      Query <- paste0("SELECT	Bulletin.BulletinID AS BULLETIN_ID, ",
                            "'", Agency, "' AS AGENCY, ",
                            "'Bulletin' AS SOURCE_DB, ",
                            "NULL AS SEASON, ",
                            "NULL AS STATUS_ID, ",
                            "convert(date, Bulletin.DateIssued) AS PUBLISH_DATE, ",
                            "Bulletin.DateIssued AS PUBLISH_TIME, ",
                            "convert(date, Bulletin.ValidUntil) AS VALID_UNTIL, ",
                            "BulletinRegion.Name AS REGION, ",
                            "Bulletin.IssuedBy AS FORECASTER ",
                      "FROM Bulletin INNER JOIN	BulletinRegion ON Bulletin.RegionID = BulletinRegion.RegionID")

      Query <- addIDsToSQLQuery(Query, BulletinIDs_Bulletin, "Bulletin.BulletinID")

      TblBulletin_Bulletin <- getRecordsFromQuery(Agency, Query, "Bulletin", User, ReadOnly=T, Verbose, SuppressDBWarnings)
      rm(Query)


      ## BULLETIN - AVALANCHE PROBLEM TABLE
      ## **********************************

      print("Getting avalanche problem table from bulletin database ...")

      ## Base query function
      makeQueryBase_AvProb_Bulletin <- function (ElevBand) {

        if (ElevBand=="ALP") {ElevBand <- "Alpine"}
        else if (ElevBand=="TL") {ElevBand <- "Treeline"}
        else if (ElevBand=="BTL") {ElevBand <- "Below"}

        Query <- paste0("SELECT	BulletinID AS BULLETIN_ID, ",
                               "ProblemID AS PROBLEM_ID, ",
                               "NULL AS SORT_ORDER, ",
                               "Type AS 'CHARACTER', ",
                               "AspectNorth AS ASP_N, ",
                               "AspectNorthEast AS ASP_NE, ",
                               "AspectEast AS ASP_E, ",
                               "AspectSouthEast AS ASP_SE, ",
                               "AspectSouth AS ASP_S, ",
                               "AspectSouthWest AS ASP_SW, ",
                               "AspectWest AS ASP_W, ",
                               "AspectNorthWest AS ASP_NW, ",
                               "Distribution AS DISTRIBUTION, ",
                               "Sensitivity AS SENSITIVITY, ",
                               "LikelyhoodMin AS LIKELIHOOD_MIN, ",
                               "LikelyhoodTypical As LIKELIHOOD_TYP, ",
                               "LikelyhoodMax AS LIKELIHOOD_MAX, ",
                               "SizeMin AS SIZE_MIN, ",
                               "SizeTypical AS SIZE_TYP, ",
                               "SizeMax AS SIZE_MAX, ",
                               "NULL AS CONTRI, ",
                               "Comments as COMMENT ",
                        "FROM Problem ",
                        "WHERE ShowToPublic=1 AND ", ElevBand, "=1")
        return(Query)
      }

      ## AvProblem Tbl for Alpine elevation band
      Query <- makeQueryBase_AvProb_Bulletin("ALP")
      Query <- addIDsToSQLQuery(Query, BulletinIDs_Bulletin, "BulletinID", Connector="AND", InBrackets=T)
      TblAvProblem_Alp_Bulletin <- getRecordsFromQuery(Agency, Query, "Bulletin", User, ReadOnly=T, Verbose, SuppressDBWarnings)
      rm(Query)

      ## AvProblem Tbl for Treeline elevation band
      Query <- makeQueryBase_AvProb_Bulletin("TL")
      Query <- addIDsToSQLQuery(Query, BulletinIDs_Bulletin, "BulletinID", Connector="AND", InBrackets=T)
      TblAvProblem_Tl_Bulletin <- getRecordsFromQuery(Agency, Query, "Bulletin", User, ReadOnly=T, Verbose, SuppressDBWarnings)
      rm(Query)

      ## AvProblem Tbl for Below Treeline elevation band
      Query <- makeQueryBase_AvProb_Bulletin("BTL")
      Query <- addIDsToSQLQuery(Query, BulletinIDs_Bulletin, "BulletinID", Connector="AND", InBrackets=T)
      TblAvProblem_Btl_Bulletin <- getRecordsFromQuery(Agency, Query, "Bulletin", User, ReadOnly=T, Verbose, SuppressDBWarnings)
      rm(Query)

      ## AvalX-specific formating of AvProblem Tbls
      TblAvProblem_Alp_Bulletin$COMMENT <- substr(TblAvProblem_Alp_Bulletin$COMMENT, 291, nchar(TblAvProblem_Alp_Bulletin$COMMENT)-11)
      TblAvProblem_Tl_Bulletin$COMMENT  <- substr(TblAvProblem_Tl_Bulletin$COMMENT,  291, nchar(TblAvProblem_Tl_Bulletin$COMMENT)-11)
      TblAvProblem_Btl_Bulletin$COMMENT <- substr(TblAvProblem_Btl_Bulletin$COMMENT, 291, nchar(TblAvProblem_Btl_Bulletin$COMMENT)-11)


      ## BULLETIN - DANGER RATING TABLE
      ## ******************************

      print("Getting danger rating table from bulletin database ...")

      ## Get raw data
      Query <- paste0("SELECT DangerRating.BulletinID, ",
                             "DangerRatingScale_Alp.Name AS Alp, ",
                             "DangerRatingScale_Tl.Name AS Tl, ",
                             "DangerRatingScale_Btl.Name AS Btl, ",
                             "DangerRating.SortOrder ",
                      "FROM DangerRating ",
                           "left outer JOIN DangerRatingScale AS DangerRatingScale_Alp ON DangerRating.Alpine = DangerRatingScale_Alp.ID ",
                           "left outer JOIN DangerRatingScale AS DangerRatingScale_Tl ON DangerRating.Treeline = DangerRatingScale_Tl.ID ",
                           "left outer JOIN DangerRatingScale AS DangerRatingScale_Btl ON DangerRating.Below = DangerRatingScale_Btl.ID ")

      Query <- addIDsToSQLQuery(Query, BulletinIDs_Bulletin, "DangerRating.BulletinID")
      Query <- paste0(Query, " ORDER BY DangerRating.BulletinID, DangerRating.SortOrder")
      TblDngRating_Raw <- getRecordsFromQuery(Agency, Query, "Bulletin", User, ReadOnly=T, Verbose, SuppressDBWarnings)
      rm(Query)

      ## Remove white space
      TblDngRating_Raw$Alp <- trimWhiteSpace(TblDngRating_Raw$Alp)
      TblDngRating_Raw$Tl <- trimWhiteSpace(TblDngRating_Raw$Tl)
      TblDngRating_Raw$Btl <- trimWhiteSpace(TblDngRating_Raw$Btl)

      ## Create proper table for danger rating
      TblDngRating_Alp_Bulletin <- TblDngRating_Tl_Bulletin <- TblDngRating_Btl_Bulletin <- data.frame(BULLETIN_ID = numeric(0),
                                                                                                       DAY0 = character(0),
                                                                                                       DAY1 = character(0),
                                                                                                       DAY2 = character(0),
                                                                                                       DAY3 = character(0), stringsAsFactors=F)

      ## Transpose danger ratings to proper table format
      for (Index_BulletinID in 1:length(BulletinIDs_Bulletin)) {

        TempDngRating <- TblDngRating_Raw[TblDngRating_Raw$BulletinID==BulletinIDs_Bulletin[Index_BulletinID],]

        TblDngRating_Alp_Bulletin <- rbind(TblDngRating_Alp_Bulletin,
                                           data.frame(BULLETIN_ID=BulletinIDs_Bulletin[Index_BulletinID],
                                                      DAY0=TempDngRating$Alp[1],
                                                      DAY1=TempDngRating$Alp[2],
                                                      DAY2=TempDngRating$Alp[3],
                                                      DAY3=TempDngRating$Alp[4]))

        TblDngRating_Tl_Bulletin <- rbind(TblDngRating_Tl_Bulletin,
                                          data.frame(BULLETIN_ID=BulletinIDs_Bulletin[Index_BulletinID],
                                                     DAY0=TempDngRating$Tl[1],
                                                     DAY1=TempDngRating$Tl[2],
                                                     DAY2=TempDngRating$Tl[3],
                                                     DAY3=TempDngRating$Tl[4]))

        TblDngRating_Btl_Bulletin <- rbind(TblDngRating_Btl_Bulletin,
                                           data.frame(BULLETIN_ID=BulletinIDs_Bulletin[Index_BulletinID],
                                                      DAY0=TempDngRating$Btl[1],
                                                      DAY1=TempDngRating$Btl[2],
                                                      DAY2=TempDngRating$Btl[3],
                                                      DAY3=TempDngRating$Btl[4]))
        rm(TempDngRating)

      }

      ## Properly sorting rows
      TblDngRating_Alp_Bulletin <- TblDngRating_Alp_Bulletin[order(TblDngRating_Alp_Bulletin$BULLETIN_ID),]
      TblDngRating_Tl_Bulletin  <- TblDngRating_Tl_Bulletin[order(TblDngRating_Tl_Bulletin$BULLETIN_ID),]
      TblDngRating_Btl_Bulletin <- TblDngRating_Btl_Bulletin[order(TblDngRating_Btl_Bulletin$BULLETIN_ID),]


      ## BULLETIN - ADDING PREFIXES TO IDS
      ## *********************************

      TblBulletin_Bulletin <- addIDPrefixToTblBulletin(TblBulletin_Bulletin, paste0(Agency, "Bulletin"))

      TblAvProblem_Alp_Bulletin <- addIDPrefixToTblAvProblem(TblAvProblem_Alp_Bulletin, paste0(Agency, "Bulletin"))
      TblAvProblem_Tl_Bulletin  <- addIDPrefixToTblAvProblem(TblAvProblem_Tl_Bulletin, paste0(Agency, "Bulletin"))
      TblAvProblem_Btl_Bulletin <- addIDPrefixToTblAvProblem(TblAvProblem_Btl_Bulletin, paste0(Agency, "Bulletin"))

      TblDngRating_Alp_Bulletin <- addIDPrefixToTblDngRating(TblDngRating_Alp_Bulletin, paste0(Agency, "Bulletin"))
      TblDngRating_Tl_Bulletin  <- addIDPrefixToTblDngRating(TblDngRating_Tl_Bulletin, paste0(Agency, "Bulletin"))
      TblDngRating_Btl_Bulletin <- addIDPrefixToTblDngRating(TblDngRating_Btl_Bulletin, paste0(Agency, "Bulletin"))


    }


    ## ***********************
    ## HAZARD WEBSITE DATABASE
    ## ***********************

    if (length(BulletinIDs_HzdWeb) > 0) {

      ## HZDWEB - BULLETIN TABLE
      ## ***********************

      print("Getting bulletin table from hzd website database ...")

      Query <- paste0("SELECT `tbl_assessments`.`ID` AS BULLETIN_ID, ",
                             "'", Agency, "' AS AGENCY, ",
                             "'HzdWebsite' AS SOURCE_DB, ",
                             "NULL AS SEASON, ",
                             "`tbl_assessments`.`Show` AS STATUS_ID, ",
                             "`tbl_assessments`.`AssessDate` AS PUBLISH_DATE, ",
                             "CONCAT(`tbl_assessments`.`AssessDate`, ' ', `tbl_assessments`.`AssessTime`) AS PUBLISH_TIME, ",
                             "`tbl_assessments`.`ValidToDate` AS VALID_UNTIL, ",
                             "`tbl_regions`.`Name` AS REGION, ",
                             "`tbl_forecasters`.`Name` AS FORECASTER ",
                      "FROM `tbl_assessments` ",
                            "INNER JOIN `tbl_regions` ON `tbl_regions`.`ID` = `tbl_assessments`.`RegionID` ",
                            "INNER JOIN `tbl_forecasters` ON `tbl_forecasters`.`ID` = `tbl_assessments`.`ForecasterID`")

      Query <- addIDsToSQLQuery(Query, BulletinIDs_HzdWeb, "`tbl_assessments`.`ID`")

  #     Query <- paste0("SELECT `tbl_assessments`.`ID` AS BULLETIN_ID, ",
  #                     "'HzdWebsite' AS SOURCE_DB, ",
  #                     "`tbl_assessments`.`Show` AS STATUS_ID, ",
  #                     "`tbl_assessments`.`AssessDate` AS PUBLISH_DATE, ",
  #                     "`tbl_regions`.`Name` AS REGION, ",
  #                     "`tbl_forecasters`.`Name` AS FORECASTER ",
  #                     "FROM `tbl_assessments` ",
  #                     "INNER JOIN `tbl_regions` ON `tbl_regions`.`ID` = `tbl_assessments`.`RegionID` ",
  #                     "INNER JOIN `tbl_forecasters` ON `tbl_forecasters`.`ID` = `tbl_assessments`.`ForecasterID` ",
  #                     "WHERE `tbl_assessments`.`FormatWizard`='AlpTlnBtl'")
  #
  #     Query <- addIDsToSQLQuery(Query, BulletinIDs_HzdWeb, "`tbl_assessments`.`ID`", Connector = "AND", InBrackets = T)

      TblBulletin_HzdWeb <- getRecordsFromQuery(Agency, Query, "HzdWebsite", User, ReadOnly=T, Verbose, SuppressDBWarnings)
      rm(Query)


      ## HZDWEB - AVALANCHE PROBLEM TABLES
      ## *********************************

      print("Getting avalanche problem table from hzd website database ...")

      ## Function for creating base queries
      makeQueryBase_AvProb_HzWeb <- function (Prefix, SortOrder) {

        Query <- paste0("SELECT tbl_assessments.ID AS BULLETIN_ID, ",
                                "concat(tbl_assessments.ID, '_", Prefix, "') AS PROBLEM_ID, ",
                                SortOrder, " AS SORT_ORDER, ",
                                "tbl_enum_avcharacter.Label AS `CHARACTER`, ",
                                "tbl_assessments.",Prefix, "Aspect AS ASP,",
                                "NULL AS ASP_N, ",
                                "NULL AS ASP_NE, ",
                                "NULL AS ASP_E, ",
                                "NULL AS ASP_SE, ",
                                "NULL AS ASP_S, ",
                                "NULL AS ASP_SW, ",
                                "NULL AS ASP_W, ",
                                "NULL AS ASP_NW, ",
                                "tbl_enum_distribution.Label AS DISTRIBUTION, ",
                                "tbl_enum_sensitivity.Label AS SENSITIVITY, ",
                                "tbl_assessments.", Prefix, "LikeliMin AS LIKELIHOOD_MIN, ",
                                "tbl_assessments.", Prefix, "LikeliTyp AS LIKELIHOOD_TYP, ",
                                "tbl_assessments.", Prefix, "LikeliMax AS LIKELIHOOD_MAX, ",
                                "tbl_assessments.", Prefix, "ExpSizeMin AS SIZE_MIN, ",
                                "tbl_assessments.", Prefix, "ExpSizeTyp AS SIZE_TYP, ",
                                "tbl_assessments.", Prefix, "ExpSizeMax AS SIZE_MAX, ",
                                "tbl_assessments.", Prefix, "Contri AS CONTRI, ",
                                "tbl_assessments.", Prefix, "Comment AS COMMENT ",
                        "FROM tbl_assessments ",
                              "INNER JOIN tbl_enum_avcharacter ON tbl_enum_avcharacter.`Value` = tbl_assessments.", Prefix, "Character ",
                              "INNER JOIN tbl_enum_distribution ON tbl_enum_distribution.`Value` = tbl_assessments.", Prefix, "Dist ",
                              "INNER JOIN tbl_enum_sensitivity ON tbl_enum_sensitivity.`Value` = tbl_assessments.", Prefix, "Sens ",
                        "WHERE tbl_assessments.", Prefix, "Character IS NOT NULL AND tbl_assessments.", Prefix, "Character !=-99 AND tbl_assessments.", Prefix, "Character !=0")
        return(Query)

      }

      ## Function for combining base queries with assessment IDs and union
      makeQuery_AvProb_HzWeb <- function (ElevBand, BulletinIDs_HzdWeb) {

        if (ElevBand=="ALP") {ElevBand <- "Alp"}
        else if (ElevBand=="TL") {ElevBand <- "Tln"}
        else if (ElevBand=="BTL") {ElevBand <- "Btl"}

        Query <- makeQueryBase_AvProb_HzWeb(paste0(ElevBand, "01"), 1)
        Query <- addIDsToSQLQuery(Query, BulletinIDs_HzdWeb, "tbl_assessments.ID", Connector="AND", InBrackets=T)
        Query <- paste0(Query, "UNION ", makeQueryBase_AvProb_HzWeb(paste0(ElevBand, "02"), 2))
        Query <- addIDsToSQLQuery(Query, BulletinIDs_HzdWeb, "tbl_assessments.ID", Connector="AND", InBrackets=T)
        Query <- paste0(Query, "UNION ", makeQueryBase_AvProb_HzWeb(paste0(ElevBand, "03"), 3))
        Query <- addIDsToSQLQuery(Query, BulletinIDs_HzdWeb, "tbl_assessments.ID", Connector="AND", InBrackets=T)

        return(Query)

      }

      ## Function for parsing aspect values from string and put into proper columns
      parseAspect_HzdWed <- function(TblAvProblem_HzdWeb) {

        if(nrow(TblAvProblem_HzdWeb)>0) {

          for (Index_AvProb in 1:nrow(TblAvProblem_HzdWeb)) {

            if (nchar(TblAvProblem_HzdWeb$ASP[Index_AvProb])==15) {

                ASP_Parsed <- as.numeric(unlist(strsplit(TblAvProblem_HzdWeb$ASP[Index_AvProb], ",")))
                TblAvProblem_HzdWeb$ASP_N[Index_AvProb]  <- ASP_Parsed[1]
                TblAvProblem_HzdWeb$ASP_NE[Index_AvProb] <- ASP_Parsed[2]
                TblAvProblem_HzdWeb$ASP_E[Index_AvProb]  <- ASP_Parsed[3]
                TblAvProblem_HzdWeb$ASP_SE[Index_AvProb] <- ASP_Parsed[4]
                TblAvProblem_HzdWeb$ASP_S[Index_AvProb]  <- ASP_Parsed[5]
                TblAvProblem_HzdWeb$ASP_SW[Index_AvProb] <- ASP_Parsed[6]
                TblAvProblem_HzdWeb$ASP_W[Index_AvProb]  <- ASP_Parsed[7]
                TblAvProblem_HzdWeb$ASP_NW[Index_AvProb] <- ASP_Parsed[8]

              }
          }

        }

        TblAvProblem_HzdWeb <- deleteDFColumnsByName(TblAvProblem_HzdWeb, "ASP")

      return(TblAvProblem_HzdWeb)

      }

      ## AvProblem Tbl for alpine elevation band
      Query <- makeQuery_AvProb_HzWeb("ALP", BulletinIDs_HzdWeb)
      TblAvProblem_Alp_HzdWeb <- getRecordsFromQuery(Agency, Query, "HzdWebsite", User, ReadOnly=T, Verbose, SuppressDBWarnings)
      TblAvProblem_Alp_HzdWeb <- parseAspect_HzdWed(TblAvProblem_Alp_HzdWeb)
      rm(Query)

      ## AvProblem Tbl for treeline elevation band
      Query <- makeQuery_AvProb_HzWeb("TL", BulletinIDs_HzdWeb)
      TblAvProblem_Tl_HzdWeb <- getRecordsFromQuery(Agency, Query, "HzdWebsite", User, ReadOnly=T, Verbose, SuppressDBWarnings)
      TblAvProblem_Tl_HzdWeb <- parseAspect_HzdWed(TblAvProblem_Tl_HzdWeb)
      rm(Query)

      ## AvProblem Tbl for below treeline elevation band
      Query <- makeQuery_AvProb_HzWeb("BTL", BulletinIDs_HzdWeb)
      TblAvProblem_Btl_HzdWeb <- getRecordsFromQuery(Agency, Query, "HzdWebsite", User, ReadOnly=T, Verbose, SuppressDBWarnings)
      TblAvProblem_Btl_HzdWeb <- parseAspect_HzdWed(TblAvProblem_Btl_HzdWeb)
      rm(Query)


      ## HZDWEB - DANGER RATING TABLES
      ## *****************************

      print("Getting danger rating table from hzd website database ...")

      ## Function for creating base query
      makeQueryBase_DngRating_HzWeb <- function (ElevBand) {

        if (ElevBand=="ALP") {ElevBand <- "Alp"}
        else if (ElevBand=="TL") {ElevBand <- "Tln"}
        else if (ElevBand=="BTL") {ElevBand <- "Btl"}

        Query <- paste0("SELECT tbl_assessments.ID AS BULLETIN_ID, ",
                        "tbl_enum_rating_dng.Label AS DAY0, ",
                        "NULL AS DAY1, ",
                        "NULL AS DAY2, ",
                        "NULL AS DAY3 ",
                        "FROM tbl_assessments ",
                        "INNER JOIN tbl_enum_rating_dng ON tbl_enum_rating_dng.`Value` = tbl_assessments.ATBDngRating", ElevBand)

        return(Query)

      }

      ## Danger Rating alpine elevation band
      Query <- makeQueryBase_DngRating_HzWeb("ALP")
      Query <- addIDsToSQLQuery(Query, BulletinIDs_HzdWeb, "tbl_assessments.ID")
      TblDngRating_Alp_HzdWeb <- getRecordsFromQuery(Agency, Query, "HzdWebsite", User, ReadOnly=T, Verbose, SuppressDBWarnings)
      rm(Query)

      ## Danger Rating treeline elevation band
      Query <- makeQueryBase_DngRating_HzWeb("TL")
      Query <- addIDsToSQLQuery(Query, BulletinIDs_HzdWeb, "tbl_assessments.ID")
      TblDngRating_Tl_HzdWeb <- getRecordsFromQuery(Agency, Query, "HzdWebsite", User, ReadOnly=T, Verbose, SuppressDBWarnings)
      rm(Query)

      ## Danger Rating below treeline elevation band
      Query <- makeQueryBase_DngRating_HzWeb("BTL")
      Query <- addIDsToSQLQuery(Query, BulletinIDs_HzdWeb, "tbl_assessments.ID")
      TblDngRating_Btl_HzdWeb <- getRecordsFromQuery(Agency, Query, "HzdWebsite", User, ReadOnly=T, Verbose, SuppressDBWarnings)
      rm(Query)


      ## HZDWEB - ADDING PREFIXES TO IDS
      ## *******************************
      TblBulletin_HzdWeb <- addIDPrefixToTblBulletin(TblBulletin_HzdWeb, paste0(Agency, "HzdWebsite"))

      TblAvProblem_Alp_HzdWeb <- addIDPrefixToTblAvProblem(TblAvProblem_Alp_HzdWeb, paste0(Agency, "HzdWebsite"))
      TblAvProblem_Tl_HzdWeb  <- addIDPrefixToTblAvProblem(TblAvProblem_Tl_HzdWeb, paste0(Agency, "HzdWebsite"))
      TblAvProblem_Btl_HzdWeb <- addIDPrefixToTblAvProblem(TblAvProblem_Btl_HzdWeb, paste0(Agency, "HzdWebsite"))

      TblDngRating_Alp_HzdWeb <- addIDPrefixToTblDngRating(TblDngRating_Alp_HzdWeb, paste0(Agency, "HzdWebsite"))
      TblDngRating_Tl_HzdWeb  <- addIDPrefixToTblDngRating(TblDngRating_Tl_HzdWeb, paste0(Agency, "HzdWebsite"))
      TblDngRating_Btl_HzdWeb <- addIDPrefixToTblDngRating(TblDngRating_Btl_HzdWeb, paste0(Agency, "HzdWebsite"))


      ## FIX SPELLING OF AVALANCHE PROBLEMS
      ## ***********************************

      TblAvProblem_Alp_HzdWeb$CHARACTER[TblAvProblem_Alp_HzdWeb$CHARACTER=="Wind Slab"] <- "Wind Slabs"
      TblAvProblem_Alp_HzdWeb$CHARACTER[TblAvProblem_Alp_HzdWeb$CHARACTER=="Storm Slab"] <- "Storm Slabs"
      TblAvProblem_Alp_HzdWeb$CHARACTER[TblAvProblem_Alp_HzdWeb$CHARACTER=="Persistent Slab"] <- "Persistent Slabs"
      TblAvProblem_Alp_HzdWeb$CHARACTER[TblAvProblem_Alp_HzdWeb$CHARACTER=="Persistent Deep Slab"] <- "Deep Persistent Slabs"
      TblAvProblem_Alp_HzdWeb$CHARACTER[TblAvProblem_Alp_HzdWeb$CHARACTER=="Cornice"] <- "Cornices"
      TblAvProblem_Alp_HzdWeb$CHARACTER[TblAvProblem_Alp_HzdWeb$CHARACTER=="Wet Slab"] <- "Wet Slabs"

      TblAvProblem_Tl_HzdWeb$CHARACTER[TblAvProblem_Tl_HzdWeb$CHARACTER=="Wind Slab"] <- "Wind Slabs"
      TblAvProblem_Tl_HzdWeb$CHARACTER[TblAvProblem_Tl_HzdWeb$CHARACTER=="Storm Slab"] <- "Storm Slabs"
      TblAvProblem_Tl_HzdWeb$CHARACTER[TblAvProblem_Tl_HzdWeb$CHARACTER=="Persistent Slab"] <- "Persistent Slabs"
      TblAvProblem_Tl_HzdWeb$CHARACTER[TblAvProblem_Tl_HzdWeb$CHARACTER=="Persistent Deep Slab"] <- "Deep Persistent Slabs"
      TblAvProblem_Tl_HzdWeb$CHARACTER[TblAvProblem_Tl_HzdWeb$CHARACTER=="Cornice"] <- "Cornices"
      TblAvProblem_Tl_HzdWeb$CHARACTER[TblAvProblem_Tl_HzdWeb$CHARACTER=="Wet Slab"] <- "Wet Slabs"

      TblAvProblem_Btl_HzdWeb$CHARACTER[TblAvProblem_Btl_HzdWeb$CHARACTER=="Wind Slab"] <- "Wind Slabs"
      TblAvProblem_Btl_HzdWeb$CHARACTER[TblAvProblem_Btl_HzdWeb$CHARACTER=="Storm Slab"] <- "Storm Slabs"
      TblAvProblem_Btl_HzdWeb$CHARACTER[TblAvProblem_Btl_HzdWeb$CHARACTER=="Persistent Slab"] <- "Persistent Slabs"
      TblAvProblem_Btl_HzdWeb$CHARACTER[TblAvProblem_Btl_HzdWeb$CHARACTER=="Persistent Deep Slab"] <- "Deep Persistent Slabs"
      TblAvProblem_Btl_HzdWeb$CHARACTER[TblAvProblem_Btl_HzdWeb$CHARACTER=="Cornice"] <- "Cornices"
      TblAvProblem_Btl_HzdWeb$CHARACTER[TblAvProblem_Btl_HzdWeb$CHARACTER=="Wet Slab"] <- "Wet Slabs"


    }

    ## *****************
    ## FINALIZING OUTPUT
    ## *****************

    ## Merge tables
    ## ************

    TblBulletin <- mergeTables("TblBulletin")

    TblAvProblem_Alp <- mergeTables("TblAvProblem_Alp")
    TblAvProblem_Tl  <- mergeTables("TblAvProblem_Tl")
    TblAvProblem_Btl <- mergeTables("TblAvProblem_Btl")

    TblDngRating_Alp <- mergeTables("TblDngRating_Alp")
    TblDngRating_Tl  <- mergeTables("TblDngRating_Tl")
    TblDngRating_Btl <- mergeTables("TblDngRating_Btl")


    ## Formatting output tables
    ## ************************

    ## Bulletin table
    if(!is.null(TblBulletin)) {
      TblBulletin <- formatTblBulletin(TblBulletin)
    }

    ## AvProblem table
    TblAvProblem_Alp <- formatTblAvProblem(TblAvProblem_Alp)
    TblAvProblem_Tl <- formatTblAvProblem(TblAvProblem_Tl)
    TblAvProblem_Btl <- formatTblAvProblem(TblAvProblem_Btl)

    ## DngRating table
    TblDngRating_Alp <- formatTblDngRating(TblDngRating_Alp)
    TblDngRating_Tl <- formatTblDngRating(TblDngRating_Tl)
    TblDngRating_Btl <- formatTblDngRating(TblDngRating_Btl)


    ## Ensuring the danger rating and avalanche problem entries are tied to bulletin
    ## *****************************************************************************

    ## AvProblem table
    TblAvProblem_Alp <- TblAvProblem_Alp[(TblAvProblem_Alp$BULLETIN_ID %in% TblBulletin$BULLETIN_ID),]
    TblAvProblem_Tl  <- TblAvProblem_Tl[(TblAvProblem_Tl$BULLETIN_ID %in% TblBulletin$BULLETIN_ID),]
    TblAvProblem_Btl <- TblAvProblem_Btl[(TblAvProblem_Btl$BULLETIN_ID %in% TblBulletin$BULLETIN_ID),]

    ## DngRating table
    TblDngRating_Alp <- TblDngRating_Alp[(TblDngRating_Alp$BULLETIN_ID %in% TblBulletin$BULLETIN_ID),]
    TblDngRating_Tl  <- TblDngRating_Tl[(TblDngRating_Tl$BULLETIN_ID %in% TblBulletin$BULLETIN_ID),]
    TblDngRating_Btl <- TblDngRating_Btl[(TblDngRating_Btl$BULLETIN_ID %in% TblBulletin$BULLETIN_ID),]


    ## Create Bulletins object
    ## ***********************

    ## Create list
    Output <- list(Bulletins = TblBulletin,
                   AvProblems = list (Alp = TblAvProblem_Alp,
                                      Tl  = TblAvProblem_Tl,
                                      Btl = TblAvProblem_Btl),
                   DngRating = list (Alp = TblDngRating_Alp,
                                     Tl  = TblDngRating_Tl,
                                     Btl = TblDngRating_Btl))

    ## Add class label
    class(Output) <- append(class(Output), "Bulletins")


    ## Eliminates non final bulletins
    ## ******************************
    if (GetOnlyFinalBulletins) {

      ## Identify duplicates
      Duplicates <- Output$Bulletins[duplicated(Output$Bulletins[c("PUBLISH_DATE", "REGION")]),]

      if(nrow(Duplicates)>0) {

        ## Initialize DF with only singles
        Singles <- Output$Bulletins[!(paste0(Output$Bulletins$VALID_UNTIL, Output$Bulletins$REGION) %in% paste0(Duplicates$VALID_UNTIL, Duplicates$REGION)),]

        ## Choose last bulletin from duplicates
        for (Index in 1:nrow(Duplicates)) {

          Temp <- Output$Bulletins[(Output$Bulletins$VALID_UNTIL==Duplicates$VALID_UNTIL[Index] & Output$Bulletins$REGION==Duplicates$REGION[Index]),]
          Temp <- Temp[!is.na(Temp$BULLETIN_ID),]

          if(nrow(Temp)>0) {
            if (sum(is.na(Temp$PUBLISH_TIME)) < nrow(Temp)) {
              Choice <- Temp[Temp$PUBLISH_TIME==max(Temp$PUBLISH_TIME, na.rm = TRUE),][1,]
            } else {
              Choice <- Temp[nrow(Temp),]
            }
            Singles <- rbind(Singles, Choice)
          }
        }

        print(paste0("WARNING: Eliminated ", (nrow(Output$Bulletins)-nrow(Singles)), " non-final bulletin records!"))
        Output <- extractFromBulletinObj(Output, ByBulletinID = Singles$BULLETIN_ID)

      }

    }

  }

  ## Return result
  ## *************
  return(Output)

}
