#' Get all bulletins.
#'
#' Wrapper function for getting all bulletins in a convenient way.
#' @param Agency Optional parameter for specifying agency: NA (default) includes all agencies, 'AvCan' or 'PkCan' includes only bulletins from Avalanche Canada or Parks Canada.
#' @param SeasonType Parameter passed on to deriveAvSeasonInfo() from SARPGeneral package to specify start and end date of a season. Values are 'Operation' (Dec. 15 to Apr. 15) (default) or 'AvYear' (Oct. 1 to Sept. 30).
#' @return Object of type Bulletins
#'
#' @examples
#'
#' require(SarpBulletinTools)
#' require(SarpBulletinToolsPrivate)
#'
#' ## Get bulletin
#' Bulletins <- getBulletinsAll()
#' summary(Bulletins)
#'
#' ## For Mac and Linux users who cannot access the DB directly, you can get all bulletins with
#' ## load(url("http://data.avalancheresearch.ca/AllBulletins_2010To2017.RData"))
#'
#' ## Format for hazard analysis
#' AnalysisDF <- formatBulletinObjForHzdAnalysis(Bulletins)
#'
#' @export

getBulletinsAll <- function(Agency=NA, SeasonType="Operation") {

  if (is.na(Agency)) {
    BestDBSourceInt <- BestDBSource
  } else if (Agency=="AvCan"|Agency=="PkCan") {
    BestDBSourceInt <- BestDBSource[BestDBSource$Agency==Agency,]
  } else {
    stop("Agency parameter specified incorrectly. Only accepts NA, 'AvCan' and 'PkCan'.")
  }

  ## Getting bulletins
  List <- list()
  for (index in 1:nrow(BestDBSource)) {
     print(paste0(BestDBSource$Agency[index], " - ", BestDBSource$Season[index]))
     print("************")
     List[[index]] <- getBulletins(Agency=BestDBSource$Agency[index], Seasons = BestDBSource$Season[index], SeasonType = SeasonType)
     print("")
  }

  ## Combining bulletins
  Output <- List[[1]]
  for (index in 2:nrow(BestDBSource)) {
    Output <- combineBulletinObjects(Output, List[[index]])
  }

  ## Return
  return(Output)

}
