#' Gets computed daily weather observations from WeatherDB
#'
#' Gets computed daily weather observations from WeatherDB
#' @param Source The source of the weather data. Must be 'InfoEx', BCMoTI' or 'EnvCan'
#' @param OperName Name of operation
#' @param WxSiteName Name of weather site
#' @param DateStart Start date of time period. Default value is NA.
#' @param DateEnd End date of time period. Default value is same as DateStart.
#' @return Dataframe with result
#'
#' @examples
#' require(SarpBulletinTools)
#' require(SarpBulletinToolsPrivate)
#'
#' WxObs_All  <- getDailyWxObsFromWxDB("InfoEx", "Whitewater Ski Resort", "Summit Study Plot")
#' WxObs_2016 <- getDailyWxObsFromWxDB("InfoEx", "Whitewater Ski Resort", "Summit Study Plot", DateStart = "2015-10-01", DateEnd = "2016-05-01")
#'
#' @export

getDailyWxObsFromWxDB <- function(Source, OperName, WxSiteName, DateStart=NA, DateEnd=DateStart) {

  Source <- tolower(Source)
  OperName <- tolower(OperName)
  WxSiteName <- tolower(WxSiteName)

  ## Source specific base query
  if(Source=='infoex') {

    Query <- paste0("SELECT * FROM infoex.daily WHERE lower(oper_name)='", OperName, "' and lower(wxsite_name)='", WxSiteName, "'")

  } else if (Source=='bcmoti') {

    stop("Functionality not yet implemented for BCMoTI!")

  } else if (Source=='envcan') {

    stop("Functionality not yet implemented for EnvCan!")

  } else {

    stop("Unknown value for source parameter!")

  }

  ## Adding date to query
  if (!is.na(DateStart)) {
    Query <- paste0(Query, " and date>='", DateStart, "' and date<='", DateEnd, "'")
  }

  ## Retrieving data from database
  DailyWxObs <- getRecordsFromQueryWxDB(Query)

  ## Calculating Day of the Season
  if(nrow(DailyWxObs)>0) {
    DailyWxObs$DaySinceOct1 <- convertDate2DaySinceStartOfSeason(DailyWxObs$date)
  }

  return(DailyWxObs)

}
