#' Gets dataframe with list of available forecast regions
#'
#' Gets dataframe with list of available forecast regions
#' @param Agency Name of agency (PkCan or AvCan)
#' @param DBFormat Name of database type. Needs to be either 'AvalX' (default), 'Bulletin' or 'HzdWebsite'
#' @param User Name of user accessing the database. This name is used to record database interactions and to potentially access local resources (i.e., local folder JDBC driver)
#' @param Verbose Switch for printing SQL query. Default value is FALSE.
#' @param SuppressDBWarnings Switch for turning warnings off. Default value is TRUE.
#' @return Dataframe with result
#'
#' @examples
#' require(SarpBulletinToolsPrivate)
#' require(SarpBulletinTools)
#'
#' getListForecastRegions("PkCan")
#' getListForecastRegions("AvCan")
#' getListForecastRegions("AvCan", "Bulletin")
#' getListForecastRegions("AvCan", "HzdWebsite")
#'
#' @export

getListForecastRegions <- function(Agency, DBFormat="AvalX", User="", Verbose=F, SuppressDBWarnings=T) {

  ## Preparation
  DBFormat <- toupper(DBFormat)

  ## Building of query

  if (DBFormat=="AVALX") {

    Query <- paste0("SELECT ABS_BULLETIN.REGION_ID AS REGION_ID, ",
                    "ABS_REGION.DESC_EN AS REGION_NAME, ",
                    "COUNT(ABS_BULLETIN.BULLETIN_GUID) AS BULLETIN_NUM, ",
                    "convert(date, MIN(ABS_BULLETIN.PUBLISH_DATE)) AS PUBLISH_DATE_FIRST, ",
                    "convert(date, MAX(ABS_BULLETIN.PUBLISH_DATE)) AS PUBLISH_DATE_LAST ",
                    "FROM ABS_BULLETIN INNER JOIN ABS_REGION ON ABS_BULLETIN.REGION_ID = ABS_REGION.REGION_ID ",
                    "WHERE ABS_BULLETIN.STATUS_ID=1 ",
                    "GROUP BY ABS_BULLETIN.REGION_ID, ABS_REGION.DESC_EN ",
                    "ORDER BY ABS_BULLETIN.REGION_ID")

  } else if (DBFormat=="BULLETIN") {

    Query <- paste0("SELECT BulletinMaster.RegionID AS REGION_ID, ",
                    "BulletinRegion.Name AS REGION_NAME, ",
                    "count(BulletinMaster.bulletinID) AS BULLETIN_NUM, ",
                    "convert(date, min(BulletinMaster.DateIssued)) AS PUBLISH_DATE_FIRST, ",
                    "convert(date, max(BulletinMaster.DateIssued)) AS PUBLISH_DATE_LAST ",
                    "FROM BulletinMaster INNER JOIN BulletinRegion ON BulletinMaster.RegionID = BulletinRegion.RegionID ",
                    "WHERE (BulletinMaster.Archived = 1) ",
                    "GROUP BY BulletinMaster.RegionID, BulletinRegion.Name ",
                    "ORDER BY BulletinRegion.Name")

    warning("Query for bulletin DBformat currently does not eliminate multiple bulletins per date issued!", immediate. = T)

  } else if (DBFormat=="HZDWEBSITE") {

    Query <- paste0("SELECT tbl_regions.ID AS REGION_ID, ",
                    "tbl_regions.`Name` AS REGION_NAME, ",
                    "Count(tbl_assessments.ID) AS BULLETIN_NUM, ",
                    "Min(tbl_assessments.AssessDate) AS PUBLISH_DATE_FIRST, ",
                    "Max(tbl_assessments.AssessDate) AS PUBLISH_DATE_LAST ",
                    "FROM tbl_regions INNER JOIN tbl_assessments ON tbl_regions.ID = tbl_assessments.RegionID ",
                    "WHERE tbl_assessments.`Show` = 1 ",
                    "GROUP BY tbl_regions.ID, tbl_regions.`Name`")

  } else {

    stop("Incorrect value for DBFormat. Can only be 'AvalX' (default), 'Bulletin' or 'HzdWebsite'!")

  }

  ResultDF <- getRecordsFromQuery(Agency, Query, DBFormat, User, ReadOnly=T, Verbose, SuppressDBWarnings)

  ResultDF$PUBLISH_DATE_FIRST <- dates(ResultDF$PUBLISH_DATE_FIRST, format="y-m-d")
  ResultDF$PUBLISH_DATE_LAST  <- dates(ResultDF$PUBLISH_DATE_LAST, format="y-m-d")

  return(ResultDF)

}
