#' Gets list of dataframes what lists number of bulletins in different database formats for agency
#'
#' Gets list of dataframes what lists number of bulletins in different database formats for agency
#' @param Agency Name of agency (PkCan or AvCan)
#' @param Season Single season number (e.g., '2015' for 2014/15 winter season)
#' @param User Name of user accessing the database. This name is used to record database interactions and to potentially access local resources (i.e., local folder JDBC driver)
#' @param Verbose Switch for printing SQL query. Default value is FALSE.
#' @param SuppressDBWarnings Switch for turning warnings off. Default value is TRUE.
#' @return List with results
#'
#' @examples
#' require(SarpBulletinToolsPrivate)
#' require(SarpBulletinTools)
#'
#' getNumBulletinsForSeason("PkCan", 2015)
#' getNumBulletinsForSeason("AvCan", 2015)
#'
#' @export

getNumBulletinsForSeason <- function(Agency, Season, User="", Verbose=F, SuppressDBWarnings=T) {

  ## Preparation
  SeasonInfo <- deriveAvSeasonInfo(Season)

  ## AvalX Query
  ## ***********
  Query_AvalX <- paste0("SELECT ABS_BULLETIN.REGION_ID AS REGION_ID, ",
                        "ABS_REGION.DESC_EN AS REGION_NAME, ",
                        "COUNT(ABS_BULLETIN.BULLETIN_GUID) AS BULLETIN_NUM, ",
                        "convert(date, MIN(ABS_BULLETIN.PUBLISH_DATE)) AS PUBLISH_DATE_FIRST, ",
                        "convert(date, MAX(ABS_BULLETIN.PUBLISH_DATE)) AS PUBLISH_DATE_LAST ",
                        "FROM ABS_BULLETIN INNER JOIN ABS_REGION ON ABS_BULLETIN.REGION_ID = ABS_REGION.REGION_ID ",
                        "WHERE ABS_BULLETIN.STATUS_ID = 1 AND convert(date, ABS_BULLETIN.PUBLISH_DATE) >= ", convertDateToSQL(SeasonInfo$StartDate), " AND convert(date, ABS_BULLETIN.PUBLISH_DATE) <= ", convertDateToSQL(SeasonInfo$EndDate), " ",
                        "GROUP BY ABS_BULLETIN.REGION_ID, ABS_REGION.DESC_EN ",
                        "ORDER BY ABS_BULLETIN.REGION_ID")

  ## Bulletin Query
  ## **************
  ## Create list of only last bulletin for specific DateIssued and RegionID
  Query_Bull1 <- paste0("SELECT max(BulletinID) AS LastBulletinID, count(BulletinID) AS BullCount ",
                        "FROM BulletinMaster ",
                        "WHERE BulletinMaster.Archived = 1 AND convert(date, BulletinMaster.DateIssued) >= ", convertDateToSQL(SeasonInfo$StartDate), " AND convert(date, BulletinMaster.DateIssued) <= ", convertDateToSQL(SeasonInfo$EndDate), " ",
                        "Group by RegionID, convert(date, DateIssued) ",
                        "order by count(BulletinID) desc")

  ## Create query with last BulletinIDs in where conditions (date and archive condition would not be necessary)
  Query_Bull2 <- paste0("SELECT BulletinMaster.RegionID AS REGION_ID, ",
                        "BulletinRegion.Name AS REGION_NAME, ",
                        "count(BulletinMaster.BulletinID) AS BULLETIN_NUM, ",
                        "convert(date, min(BulletinMaster.DateIssued)) AS PUBLISH_DATE_FIRST, ",
                        "convert(date, max(BulletinMaster.DateIssued)) AS PUBLISH_DATE_LAST ",
                        "FROM BulletinMaster INNER JOIN BulletinRegion ON BulletinMaster.RegionID = BulletinRegion.RegionID ",
                        "WHERE BulletinMaster.Archived = 1 AND convert(date, BulletinMaster.DateIssued) >= ", convertDateToSQL(SeasonInfo$StartDate), " AND convert(date, BulletinMaster.DateIssued) <= ", convertDateToSQL(SeasonInfo$EndDate), " ")


  ## HzdWebsite Query
  ## ****************
  Query_Hzd <- paste0("SELECT tbl_regions.ID AS REGION_ID, ",
                      "tbl_regions.`Name` AS REGION_NAME, ",
                      "Count(tbl_assessments.ID) AS BULLETIN_NUM, ",
                      "Min(tbl_assessments.AssessDate) AS PUBLISH_DATE_FIRST, ",
                      "Max(tbl_assessments.AssessDate) AS PUBLISH_DATE_LAST ",
                      "FROM tbl_regions INNER JOIN tbl_assessments ON tbl_regions.ID = tbl_assessments.RegionID ",
                      "WHERE tbl_assessments.`Show` = 1 AND tbl_assessments.AssessDate >=", convertDateToSQL(SeasonInfo$StartDate), " AND tbl_assessments.AssessDate <=", convertDateToSQL(SeasonInfo$EndDate), " ",
                      "GROUP BY tbl_regions.ID, tbl_regions.`Name`")

  ## Parks Canada
  if (Agency=="PkCan") {

    AvalX <- getRecordsFromQuery(Agency, Query_AvalX, "AvalX", User, ReadOnly=T, Verbose, SuppressDBWarnings)
    AvalX$PUBLISH_DATE_FIRST <- dates(AvalX$PUBLISH_DATE_FIRST, format="y-m-d")
    AvalX$PUBLISH_DATE_LAST  <- dates(AvalX$PUBLISH_DATE_LAST, format="y-m-d")

    Output <- list(AvalX=AvalX,
                   Bulletin=NA,
                   HzdWebsite=NA)


  ## Avalanche Canada
  } else if (Agency=="AvCan") {

    ## AvalX
    AvalX <- getRecordsFromQuery(Agency, Query_AvalX, "AvalX", User, ReadOnly=T, Verbose, SuppressDBWarnings)
    AvalX$PUBLISH_DATE_FIRST <- dates(AvalX$PUBLISH_DATE_FIRST, format="y-m-d")
    AvalX$PUBLISH_DATE_LAST  <- dates(AvalX$PUBLISH_DATE_LAST, format="y-m-d")

    ## Bulletin
    IDList_Bull1 <- getRecordsFromQuery(Agency, Query_Bull1, "Bulletin", User, ReadOnly=T, Verbose, SuppressDBWarnings)$LastBulletinID

    if (length(IDList_Bull1)>0) {

      Query_Bull2 <- addIDsToSQLQuery(Query_Bull2, IDList_Bull1, "BulletinMaster.BulletinID", Connector = "AND", InBrackets=T)
      Query_Bull2 <- paste0(Query_Bull2, " GROUP BY BulletinMaster.RegionID, BulletinRegion.Name ", "ORDER BY BulletinRegion.Name")

      Bull <- getRecordsFromQuery(Agency, Query_Bull2, "Bulletin", User, ReadOnly=T, Verbose, SuppressDBWarnings)

    } else {

      Bull <- data.frame(REGION_ID=numeric(0), REGION_NAME=character(0), BULLETIN_NUM=numeric(0), PUBLISH_DATE_FIRST=character(0), PUBLISH_DATE_LAST=character(0))
      Bull$REGION_NAME <- as.character(Bull$REGION_NAME)

    }

    Bull$PUBLISH_DATE_FIRST <- dates(Bull$PUBLISH_DATE_FIRST, format="y-m-d")
    Bull$PUBLISH_DATE_LAST  <- dates(Bull$PUBLISH_DATE_LAST, format="y-m-d")

    # warning("Query for bulletin DBformat currently does not eliminate multiple bulletins per date issued!", immediate. = T)

    ## Hazard website
    Hzd <- getRecordsFromQuery(Agency, Query_Hzd, "HzdWebsite", User, ReadOnly=T, Verbose, SuppressDBWarnings)
    Hzd$PUBLISH_DATE_FIRST <- dates(Hzd$PUBLISH_DATE_FIRST, format="y-m-d")
    Hzd$PUBLISH_DATE_LAST  <- dates(Hzd$PUBLISH_DATE_LAST, format="y-m-d")

    Output <- list(AvalX=AvalX,
                   Bulletin=Bull,
                   HzdWebsite=Hzd)

  }

  return(Output)

}
