#' Gets provinces geometry as spatial dataframe
#'
#' Gets provinces geometry as spatial dataframe
#' @param AsCentroid Switch for whether returned geometry should be the centroid. Default value is F.
#' @param Verbose Switch for printing of query
#' @param SuppressDBWarnings Switch for supressing warning messages from postgresqlExecStatement. Turn on for debugging!
#'
#' @return Appropriate sp spatial objects (SpatialPoints, SpatialLines, SpatialPolygons)
#'
#' @examples
#' require(SarpBulletinTools)
#' require(SarpBulletinToolsPrivate)
#'
#' ProvincesSPDF <- getProvincesGeomSPDF()
#' plot(ProvincesSPDF)
#'
#' @export

getProvincesGeomSPDF <- function(AsCentroid=F, Verbose=F, SuppressDBWarnings=T) {

  require(sp)

  ## Supporting functions
  getProvincesGeom <- function(AsCentroid=F, Verbose=F, SuppressDBWarnings=T) {

    ## Build query - Base
    if (AsCentroid) {
      Query <- paste0("SELECT uuid, ST_AsText(ST_Centroid(geom)) AS wkt, ST_SRID(geom) AS srid FROM bulletin.provinces")
    } else {
      Query <- paste0("SELECT uuid, ST_AsText(geom) AS wkt, ST_SRID(geom) AS srid FROM bulletin.provinces")
    }

    ## Send query to database
    ResultDF <- getRecordsFromQueryBulletinRegionsDB(Query=Query, Verbose=Verbose, SuppressDBWarnings=SuppressDBWarnings)

    if(nrow(ResultDF)>0) {

      ## Create spatial objects
      for (Index_Rows in 1:nrow(ResultDF)) {

        ## Convert WKT to spatial object (requires rgeos package)
        SpatialObject <- rgeos::readWKT(ResultDF$wkt[Index_Rows], id=ResultDF$uuid[Index_Rows])

        ## Add projection information (SRID=4617)
        proj4string(SpatialObject) <- "+proj=longlat +ellps=GRS80 +towgs84=0,0,0,0,0,0,0 +no_defs"

        ## Combining spatial objects (requires maptools package)
        if (Index_Rows==1) {
          SpatialObjects <- SpatialObject
        } else {
          SpatialObjects <- maptools::spRbind(SpatialObjects, SpatialObject)
        }
      }
      rm(Index_Rows)

    } else {

      SpatialObjects <- NULL

    }

    ## Return final result
    return (SpatialObjects)

  }


  ## Retrieve spatial objects
  SpatialObjects <- getProvincesGeom(AsCentroid=AsCentroid, Verbose=Verbose, SuppressDBWarnings=SuppressDBWarnings)

  if (!is.null(SpatialObjects)) {

    ## Retrieve attributes
    Query <- "SELECT uuid, name, flag_delete, dtcre, dtmod FROM bulletin.provinces where "


    if (class(SpatialObjects)=='SpatialPolygons') {
      NumRegions <- length(SpatialObjects@polygons)
      for (Region_Index in 1:NumRegions) {
        Query <- paste0(Query, "(uuid='", SpatialObjects@polygons[[Region_Index]]@ID, "')")
        if (Region_Index < NumRegions) {
          Query <- paste0(Query, " OR ")
        }
      }
    } else if (class(SpatialObjects)=='SpatialPoints') {
      NumRegions <- length(row.names(SpatialObjects@coords))
      for (Region_Index in 1:NumRegions) {
        Query <- paste0(Query, "(uuid='", row.names(SpatialObjects@coords)[Region_Index], "')")
        if (Region_Index < NumRegions) {
          Query <- paste0(Query, " OR ")
        }
      }
    } else {
      stop(paste0("Class '", class(SpatialObjects), "' not supported!"), immediate. = T)
    }

    AttributesDF <- getRecordsFromQueryBulletinRegionsDB(Query, Verbose=Verbose, SuppressDBWarnings=SuppressDBWarnings)
    row.names(AttributesDF) <- AttributesDF$uuid

    ## Create spatial objects dataframe
    if (class(SpatialObjects)=="SpatialPoints") {
      SPDF <- SpatialPointsDataFrame(SpatialObjects, AttributesDF)
    } else if (class(SpatialObjects)=="SpatialPolygons") {
      SPDF <- SpatialPolygonsDataFrame(SpatialObjects, AttributesDF)
    } else {
      SPDF <- NULL
      warning(paste0("Class '", class(SpatialObjects), "' not supported."), immediate. = T)
    }

  } else {

    SPDF <- NULL

  }

  ## Return SPDF
  return(SPDF)
}
