#' Gets record from bulletin database based on SQL query
#'
#' Gets record from bulletin database based on SQL query
#' @param Agency Name of agency (PkCan or AvCan)
#' @param Query Properly formatted SQL query
#' @param DBFormat Name of database type. Needs to be either 'AvalX' (default), 'Bulletin' or 'HzdWebsite'
#' @param User Name of user accessing the database. This name is used to record database interactions and to potentially access local resources (i.e., local folder JDBC driver)
#' @param ReadOnly Switch for specifiying whether query is sent with read only or read and write credentials.
#' @param Verbose Switch for printing SQL query. Default value is FALSE.
#' @param SuppressDBWarnings Switch for turning warnings off. Default value is TRUE.
#' @return Dataframe with result
#'
#' @examples
#' require(SarpBulletinToolsPrivate)
#' require(SarpBulletinTools)
#'
#' RegionNamesPkCan_AvalX <- getRecordsFromQuery("PkCan", "SELECT DESC_EN AS REGION_NAME FROM ABS_REGION")
#' RegionNamesAvCan_AvalX <- getRecordsFromQuery("AvCan", "SELECT DESC_EN AS REGION_NAME FROM ABS_REGION")
#' RegionNamesAvCan_HzdWebsite <- getRecordsFromQuery("AvCan", "SELECT tbl_regions.`Name` AS REGION_NAME FROM tbl_regions", DBFormat="HzdWebsite")
#'
#' @export

getRecordsFromQuery <- function(Agency, Query, DBFormat="AvalX", User="", ReadOnly=T, Verbose=F, SuppressDBWarnings=T, MaxNumTries=10) {

  ## Preparation
  DBInfo <- getBulletinDBInfo(Agency, DBFormat)

  if(Verbose) {cat(Query, ";\n\n")}

  if (SuppressDBWarnings) {options(warn=-1)}

  ## Connect to DB
  # DBCon <- connectToBulletinDB(Agency, DBFormat, User, ReadOnly)
  DBCon <- tryMultipleTimes(connectToBulletinDB(Agency, DBFormat, User, ReadOnly), MaxNumTries=MaxNumTries)

  if(DBInfo$DBType=="MSSQL") {

    ResultDF <- RJDBC::dbGetQuery(DBCon, Query)
    dbDisconnect(DBCon)

  } else if (DBInfo$DBType=="MySQL") {

    Result <- RMySQL::dbSendQuery(DBCon, Query)
    ResultDF <- RMySQL::dbFetch(Result, n=-1)
    dbDisconnect(DBCon)
    rm(Result)

  } else {

    stop("Incorrect value for DBType. Only supports 'MSSQL' and 'MySQL'!")

  }

  ## Finish
  if (SuppressDBWarnings) {options(warn=0)}
  rm(DBCon)

  return(ResultDF)

}
