#' Plots assessments and CTree predictions on hazard chart
#'
#' Plots assessments and CTree predictions on hazard chart.
#' This function was created for Taylor Clarke's research in 2018.
#' @param CTreeModel CTreeModel object created with createDRCTreeAndValidate().
#' @param ScenGen List with parameters for scenario (e.g., list(AGENCY="AvCan", AVPROB="PERS")). Default is empty list.
#' @param PlotType Specifies which plot is shown. Options are 'Predictions' (default), 'Observations' or 'Probabilities'.
#' @param ProbPlotDR Danger rating for probability plot. Values can only be low, moderate (default), considerable or high.
#' @param OnlyWhereObs Switch for whether tree rules should only be shown where observations are available (default) or across entire chart.
#' @param WithNodeNum Switch for whether node numbers are shown in prediction chart. Default is TRUE.
#' @param WithProb Switch for whether probability pie charts are shown in prediction chart. Default is TRUE.
#' @param RuleLwd Width of lines between rules. Default is 3
#' @param DRClr Array for danger rating colors. Default are standard colors as defined in SarpGeneralVis package
#' @param DRClrTransp Transparency for background danger rating colors in prediction chart. Must be between 10 and 99. Default is 50.
#' @param DRClrProdSum Switch for specifying what danger rating is shown in the background: product sum of proportions (TRUE, default) or largest proportion (FALSE)
#' @param main Title for chart that overwrites default
#'
#' @examples
#' require(SarpBulletinTools)
#'
#' ## Get data
#' load(url("http://data.avalancheresearch.ca/2018_Analysis_Taylor.RData"))
#' nrow(TblAnalysis)
#'
#' ## Extract single problems and stack
#' SingleProbStacked <- extractAndStackSingleAvProblems(TblAnalysis)
#' table(SingleProbStacked$PROB_COMBINATION)
#'
#' ## Extract columns of interest and rename
#' TblTree <- SingleProbStacked[,c("AGENCY", "REGION", "MTNRANGE", "DAY0", "PROB_COMBINATION", "LIKELIHOOD_TYP", "SIZE_TYP")]
#' names(TblTree)
#' names(TblTree) <- c("AGENCY", "REGION", "MTNRANGE", "DAY0", "AVPROB", "LH", "SZ")
#'
#' ## Tree 1: Just LH and SZ for all av problems
#' ## ******************************************
#' ## Create tree model
#' TreeFormula <- "DAY0 ~ LH + SZ"
#' TreeModel1 <- createDRCTreeAndValidate(TreeFormula, TblTree)
#'
#' ## Plot tree
#' plot(TreeModel1$CTree)
#'
#' ## Plot observations and predictions on hzd chart
#' plotCTreeModelSingleProbOnHzdChart(TreeModel1, PlotType = "Obs")
#' plotCTreeModelSingleProbOnHzdChart(TreeModel1)
#'
#' ## Tree 2: LH and SZ for all av problems with agency
#' ## *************************************************
#' ## Create tree model
#' TreeFormula <- "DAY0 ~ LH + SZ + AGENCY"
#' TreeModel2 <- createDRCTreeAndValidate(TreeFormula, TblTree)
#'
#' ## Plot tree
#' plot(TreeModel2$CTree)
#'
#' ## Plot observations and predictions on hzd chart for different scenarios: AvCan or PkCan
#' par(mfrow=c(2,2))
#' plotCTreeModelSingleProbOnHzdChart(TreeModel2, ScenGen = list(AGENCY="AvCan"), PlotType = "Obs")
#' plotCTreeModelSingleProbOnHzdChart(TreeModel2, ScenGen = list(AGENCY="PkCan"), PlotType = "Obs")
#'
#' plotCTreeModelSingleProbOnHzdChart(TreeModel2, ScenGen = list(AGENCY="AvCan"))
#' plotCTreeModelSingleProbOnHzdChart(TreeModel2, ScenGen = list(AGENCY="PkCan"))
#' par(mfrow=c(1,1))
#'
#' ## Compare with prediction for combined model
#' plotCTreeModelSingleProbOnHzdChart(TreeModel1)
#'
#' ## Tree 3: Add avalanche problem
#' ## *****************************
#' ## Create tree model
#' TreeFormula <- "DAY0 ~ LH + SZ + AGENCY + AVPROB"
#' TreeModel3 <- createDRCTreeAndValidate(TreeFormula, TblTree)
#'
#' ## Plot tree
#' plot(TreeModel3$CTree)
#'
#' ## Compare assessments of PRES, STORM and LDRY of AvCan and PkCan
#' par(mfrow=c(2,3))
#' plotCTreeModelSingleProbOnHzdChart(TreeModel3, ScenGen = list(AGENCY="AvCan", AVPROB="PERS"))
#' plotCTreeModelSingleProbOnHzdChart(TreeModel3, ScenGen = list(AGENCY="AvCan", AVPROB="STORM"))
#' plotCTreeModelSingleProbOnHzdChart(TreeModel3, ScenGen = list(AGENCY="AvCan", AVPROB="LDRY"))
#' plotCTreeModelSingleProbOnHzdChart(TreeModel3, ScenGen = list(AGENCY="PkCan", AVPROB="PERS"))
#' plotCTreeModelSingleProbOnHzdChart(TreeModel3, ScenGen = list(AGENCY="PkCan", AVPROB="STORM"))
#' plotCTreeModelSingleProbOnHzdChart(TreeModel3, ScenGen = list(AGENCY="PkCan", AVPROB="LDRY"))
#' par(mfrow=c(1,1))
#'
#' @export

plotCTreeModelSingleProbOnHzdChart <- function(CTreeModel, ScenGen=list(), PlotType="Predictions", ProbPlotDR="Moderate", OnlyWhereObs=TRUE, WithNodeNum=TRUE, WithProb=TRUE, RuleLwd=3, DRClrProdSum=FALSE, DRClr=getHzdColor(c(1, 2, 3, 4)), DRClrTrans=50, main="") {

  ## Preparation
  ## ***********

  ## Extract relevant elements
  CTree <- CTreeModel$CTree
  Data <- CTreeModel$TrainData

  ## Check for LH and SZ
  if (!"LH" %in% names(Data)) {stop("The likelihood term needs to be called 'LH'!")}
  if (!"SZ" %in% names(Data)) {stop("The size term needs to be called 'SZ'!")}


  ## Check whether scenario contains all of the necessary terms
  ## **********************************************************
  TermsTree <- attr(formula(CTree), "term.labels")
  TermsTree <- TermsTree[TermsTree!="LH" & TermsTree!="SZ"]

  if (length(TermsTree)==0) {

    warning("Only LH and SZ incuded in tree! Additional parameters in scenario ignored!", immediate. = T)
    WithScenario <- FALSE

  } else {

    WithScenario <- TRUE

    TermsTree <- TermsTree[order(TermsTree)]

    TermsScen <- names(ScenGen)
    TermsScen <- TermsScen[order(TermsScen)]

    if (sum(TermsScen==TermsTree) < length(TermsTree)) {
      stop(paste0("Your Scenarion variable does not include all of the necessary parameters! Needed are ", paste(TermsTree, collapse = ", "), "!"))
    }

  }

  ## Filters data down to given scenario
  ## ***********************************
  if(WithScenario) {
    for (i in 1:length(ScenGen)) {
      Data <- Data[Data[,names(ScenGen)[i]]==ScenGen[[i]],]
    }
  }

  ## Count observation for different hazard ratings
  ObsSummary <- data.frame(expand.grid(LH=levels(Data$LH), SZ=levels(Data$SZ)))
  ObsSummary$ObsCountL <- 0
  ObsSummary$ObsCountM <- 0
  ObsSummary$ObsCountC <- 0
  ObsSummary$ObsCountH <- 0

  for (i in 1: nrow(Data)) {

    ObsSummaryIndex <- which(as.character(ObsSummary$LH)==as.character(Data$LH[i]) & as.character(ObsSummary$SZ)==as.character(Data$SZ[i]))

    if(Data$DAY0[i]=="Low") {
      ObsSummary$ObsCountL[ObsSummaryIndex] <- ObsSummary$ObsCountL[ObsSummaryIndex] + 1
    } else if(Data$DAY0[i]=="Moderate") {
      ObsSummary$ObsCountM[ObsSummaryIndex] <- ObsSummary$ObsCountM[ObsSummaryIndex] + 1
    } else if(Data$DAY0[i]=="Considerable") {
      ObsSummary$ObsCountC[ObsSummaryIndex] <- ObsSummary$ObsCountC[ObsSummaryIndex] + 1
    } else if(Data$DAY0[i]=="High/Extreme") {
      ObsSummary$ObsCountH[ObsSummaryIndex] <- ObsSummary$ObsCountH[ObsSummaryIndex] + 1
    }
  }
  rm(ObsSummaryIndex, i)

  ObsSummary$ObsCountTotal <- ObsSummary$ObsCountL + ObsSummary$ObsCountM + ObsSummary$ObsCountC + ObsSummary$ObsCountH


  ## Predictions
  ## ***********
  ## Create new data
  NewData <- data.frame(expand.grid(LH=levels(Data$LH), SZ=levels(Data$SZ)))
  NewData$LH <- ordered(NewData$LH, levels=levels(Data$LH))
  NewData$SZ <- ordered(NewData$SZ, levels=levels(Data$SZ))

  ## Adds all of the necessary variables and formats them properly
  if (WithScenario) {
    for (i in 1:length(ScenGen)) {
      NewData[,names(ScenGen)[i]] <- ScenGen[[i]]
      if(class(Data[,names(ScenGen)[i]])[1]=="factor") {
        NewData[,names(ScenGen)[i]] <- factor(NewData[,names(ScenGen)[i]], levels=levels(Data[,names(ScenGen)[i]]))
      } else if (class(Data[,names(ScenGen)[i]])[1]=="ordered") {
        NewData[,names(ScenGen)[i]] <- ordered(NewData[,names(ScenGen)[i]], levels=levels(Data[,names(ScenGen)[i]]))
      }
    }
    rm(i)
  }

  ## Calculates predictions
  Predict.Rate <- predict(CTree, newdata = NewData, type = "response")
  Predict.Dist <- predict(CTree, newdata = NewData, type = "prob")
  Predict.Node <- predict(CTree, newdata = NewData, type = "node")

  ## Adds to NewData
  NewData$PreRate <- Predict.Rate
  NewData$PreProbL <- Predict.Dist[,1]
  NewData$PreProbM <- Predict.Dist[,2]
  NewData$PreProbC <- Predict.Dist[,3]
  NewData$PreProbH <- Predict.Dist[,4]
  NewData$PreNode <- Predict.Node
  rm(Predict.Dist, Predict.Rate, Predict.Node)

  ## Calculates single numerical rating for gradual colors
  NewData$PreNumRate <- NewData$PreProbL*1 + NewData$PreProbM*2 + NewData$PreProbC*3 + NewData$PreProbH*4
  NewData$PreNumRateCat <- cut(NewData$PreNumRate, seq(1, 4, 0.1), include.lowest = TRUE)
  NewData$PreNumRateCat <- cut(NewData$PreNumRate, c(1, seq(1.25, 3.75, 0.5), 4), include.lowest = TRUE)
  RatingColPalette <- colorRampPalette(colors = DRClr)(7)

  ## Adds number of observations
  NewData <- merge(NewData, ObsSummary, by.x = c("LH", "SZ"), by.y = c("LH", "SZ"), all.x = TRUE)

  ## Extract relevant paths and rules
  PathsRules <- CTreeModel$PathsRules[CTreeModel$PathsRules$node %in% unique(NewData$PreNode),]

  ## Create output
  Output <- list(ScenGen=ScenGen,
                 Obs=ObsSummary,
                 Pred=NewData,
                 PathsRules=PathsRules)

  ## PLOTTING
  ## ********

  ## Create title string
  if (WithScenario) {
    TitleStart <- paste(ScenGen, collapse = " - ")
  } else {
    TitleStart <- "No additional parameters"
  }

  ## Base plot
  plotHzdChartBase()

  ## Observation plot
  if (toupper(substr(PlotType, 1, 3))=="OBS") {

    if(main=="") {
      title(paste(TitleStart, "(Observations)"))
    } else {
      title(main)
    }

    ObsSummary <- ObsSummary[ObsSummary$ObsCountTotal>0,]
    for (i in 1:nrow(ObsSummary)) {
      Prop <- as.numeric(ObsSummary[i,c("ObsCountL", "ObsCountM", "ObsCountC", "ObsCountH")])
      add.pie(z=Prop, x = as.numeric(ObsSummary$SZ[i]), y = as.numeric(ObsSummary$LH[i]), radius=0.25, labels = NA, col = DRClr)
    }
    text(as.numeric(ObsSummary$SZ)+0.3, as.numeric(ObsSummary$LH)+0.3, labels = ObsSummary$ObsCountTotal, cex=.75)


  ## Prediction plot
  } else if (toupper(substr(PlotType, 1, 4))=="PRED") {

    if(main=="") {
      title(paste(TitleStart, "(Predictions)"))
    } else {
      title (main)
    }

    if(OnlyWhereObs) {
      NewData <- NewData[NewData$ObsCountTotal>0,]
    }

    if (DRClrProdSum) {
      rect(xleft = as.numeric(NewData$SZ)-0.5, xright = as.numeric(NewData$SZ)+0.5, ybottom = as.numeric(NewData$LH)-0.5, ytop = as.numeric(NewData$LH)+0.5, col = paste0(RatingColPalette[as.numeric(NewData$PreNumRateCat)], as.character(DRClrTrans)), border = NA)
    } else {
      rect(xleft = as.numeric(NewData$SZ)-0.5, xright = as.numeric(NewData$SZ)+0.5, ybottom = as.numeric(NewData$LH)-0.5, ytop = as.numeric(NewData$LH)+0.5, col = paste0(DRClr[as.numeric(NewData$PreRate)], as.character(DRClrTrans)), border = NA)
    }

    if(WithProb) {
      for (i in 1:nrow(NewData)) {
        PredProp <- as.numeric(round(NewData[i,c("PreProbL", "PreProbM", "PreProbC", "PreProbH")]*100,0))
        add.pie(z=PredProp, x = as.numeric(NewData$SZ[i]), y = as.numeric(NewData$LH[i]), radius=0.25, labels = NA, col = DRClr)
      }
      if(WithNodeNum) {
        text(as.numeric(NewData$SZ)+0.3, as.numeric(NewData$LH)+0.3, labels = NewData$PreNode, cex=0.75)
      }
    } else if (WithNodeNum) {
      text(as.numeric(NewData$SZ), as.numeric(NewData$LH), labels = NewData$PreNode, cex=1)
    }

    ## Horizontal and verical lines between rules
    if(WithNodeNum) {
      for (lh in 1:9) {
        for (sz in 1:8) {
          NodeCurr <- NewData$PreNode[(as.numeric(NewData$LH)==lh & as.numeric(NewData$SZ)==sz)]
          NodeNext <- NewData$PreNode[(as.numeric(NewData$LH)==lh & as.numeric(NewData$SZ)==sz+1)]
          if(length(NodeCurr)>0 & length(NodeNext)>0) {
            if(NodeCurr != NodeNext) {
              arrows(x0 = sz+0.5, x1 = sz+0.5, y0 = lh-0.5, y1 = lh+0.5, lwd=RuleLwd, length = 0)
            }
          }
        }
      }
      for (lh in 1:8) {
        for (sz in 1:9) {
          NodeCurr <- NewData$PreNode[(as.numeric(NewData$LH)==lh & as.numeric(NewData$SZ)==sz)]
          NodeNext <- NewData$PreNode[(as.numeric(NewData$LH)==lh+1 & as.numeric(NewData$SZ)==sz)]
          if(length(NodeCurr)>0 & length(NodeNext)>0) {
            if(NodeCurr != NodeNext) {
              arrows(x0 = sz-0.5, x1 = sz+0.5, y0 = lh+0.5, y1 = lh+0.5, lwd=RuleLwd, length = 0)
            }
          }
        }
      }
    }

    box()

    ## Prediction plot
  } else if (toupper(substr(PlotType, 1, 4))=="PROB") {

    if(main=="") {
      title(paste0(TitleStart, " (Prob. for ", ProbPlotDR, ")"))
    } else {
      title (main)
    }

    if(OnlyWhereObs) {
      NewData <- NewData[NewData$ObsCountTotal>0,]
    }

    ## Determing values to be plotted and background color
    if (tolower(substr(ProbPlotDR, 1, 3))=="low") {
      NewData$ProbPlotValue <- round(100*NewData$PreProbL)
      ProbPlotCol <- DRClr[1]
    } else if (tolower(substr(ProbPlotDR, 1, 3))=="mod") {
      NewData$ProbPlotValue <- round(100*NewData$PreProbM)
      ProbPlotCol <- DRClr[2]
    } else if (tolower(substr(ProbPlotDR, 1, 3))=="con") {
      NewData$ProbPlotValue <- round(100*NewData$PreProbC)
      ProbPlotCol <- DRClr[3]
    } else if (tolower(substr(ProbPlotDR, 1, 3))=="hig") {
      NewData$ProbPlotValue <- round(100*NewData$PreProbH)
      ProbPlotCol <- DRClr[4]
    } else {
      stop ("Invalid value for ProbPlotDr!")
    }

    ## Formatting of probability
    NewData$ProbPlotFormCol <- "black"
    NewData$ProbPlotFormCol[NewData$ProbPlotValue < 25] <- "#808080"
    NewData$ProbPlotFormFont <- 1
    NewData$ProbPlotFormFont[NewData$ProbPlotValue >= 50] <- 2
    NewData$ProbPlotFormCex <- 1
    NewData$ProbPlotFormCex[NewData$ProbPlotValue < 75] <- 0.5 + 0.5*NewData$ProbPlotValue[NewData$ProbPlotValue < 75]/75
    NewData$ProbPlotTrans <- round(10 + (90/100*NewData$ProbPlotValue))


    ## Background shading
    rect(xleft = as.numeric(NewData$SZ)-0.5, xright = as.numeric(NewData$SZ)+0.5, ybottom = as.numeric(NewData$LH)-0.5, ytop = as.numeric(NewData$LH)+0.5, col = paste0(ProbPlotCol, as.character(NewData$ProbPlotTrans)), border = NA)

    ## Probability values
    text(as.numeric(NewData$SZ), as.numeric(NewData$LH), labels = NewData$ProbPlotValue, font=NewData$ProbPlotFormFont, col=NewData$ProbPlotFormCol, cex=NewData$ProbPlotFormCex)

    ## Horizontal and verical lines between rules
    if(WithNodeNum) {
      for (lh in 1:9) {
        for (sz in 1:8) {
          NodeCurr <- NewData$PreNode[(as.numeric(NewData$LH)==lh & as.numeric(NewData$SZ)==sz)]
          NodeNext <- NewData$PreNode[(as.numeric(NewData$LH)==lh & as.numeric(NewData$SZ)==sz+1)]
          if(length(NodeCurr)>0 & length(NodeNext)>0) {
            if(NodeCurr != NodeNext) {
              arrows(x0 = sz+0.5, x1 = sz+0.5, y0 = lh-0.5, y1 = lh+0.5, lwd=RuleLwd, length = 0)
            }
          }
        }
      }
      for (lh in 1:8) {
        for (sz in 1:9) {
          NodeCurr <- NewData$PreNode[(as.numeric(NewData$LH)==lh & as.numeric(NewData$SZ)==sz)]
          NodeNext <- NewData$PreNode[(as.numeric(NewData$LH)==lh+1 & as.numeric(NewData$SZ)==sz)]
          if(length(NodeCurr)>0 & length(NodeNext)>0) {
            if(NodeCurr != NodeNext) {
              arrows(x0 = sz-0.5, x1 = sz+0.5, y0 = lh+0.5, y1 = lh+0.5, lwd=RuleLwd, length = 0)
            }
          }
        }
      }
    }


  ## PlotType not supported
  } else {

    warning(paste("Plot type", PlotType, "not supported!"), immediate. = T)
    title(paste("Plot type", PlotType, "not supported!"))

  }

  ## Output
  invisible(Output)

}
