#' Plots assessments and CTree predictions on hazard chart for trees that include two problems.
#'
#' Plots assessments and CTree predictions on hazard chart for trees that include two problems.
#' This function was created for Taylor Clarke's research in 2018.
#' @param CTreeModel CTreeModel object created with createDRCTreeAndValidate().
#' @param Prob Array with problem abbreviations. The hazard chart is plotted for the first problem at the scenarios that are specificed for the second problem in parameter ScenProb2.
#' @param ScenProb2 List of lists with scenarios for problem 2. Default is list(list(LH="P", SZ="2.0")).
#' @param ScenGen List with parameters for scenario (e.g., list(AGENCY="AvCan", AVPROB="PERS")). Default is empty list.
#' @param PlotType Specifies which plot is shown. Options are 'Predictions' (default) and 'Observations'.
#' @param OnlyWhereObs Switch for whether tree rules should only be shown where observations are available (default) or across entire chart.
#' @param WithNodeNum Switch for whether node numbers are shown in prediction chart. Default is TRUE.
#' @param RuleLwd Width of lines between rules. Default is 3
#' @param WithProb Switch for whether probability pie charts are shown in prediction chart. Default is TRUE.
#' @param DRClr Array for danger rating colors. Default are standard colors as defined in SarpGeneralVis package
#' @param DRClrTransp Transparency for background danger rating colors in prediction chart. Must be between 10 and 99. Default is 50.
#' @param DRClrProdSum Switch for specifying what danger rating is shown in the background: product sum of proportions (TRUE, default) or largest proportion (FALSE)
#' @param main Title for chart that overwrites default
#'
#' @examples
#' require(SarpBulletinTools)
#'
#' ## Get data
#' load(url("http://data.avalancheresearch.ca/2018_Analysis_Taylor.RData"))
#' nrow(TblAnalysis)
#'
#' ## Extract storm slab and persistent slab
#' table(TblAnalysis$PROB_COMBINATION)
#' Tbl <- TblAnalysis[TblAnalysis$PROB_COMBINATION=="PERS_STORM",c("AGENCY", "REGION", "MTNRANGE", "DAY0", "PROB_COMBINATION", "PERS_LIKELIHOOD_TYP", "PERS_SIZE_TYP", "STORM_LIKELIHOOD_TYP", "STORM_SIZE_TYP")]
#' names(Tbl) <- c("AGENCY", "REGION", "MTNRANGE", "DAY0", "PROB_COMBINATION", "PERS_LH", "PERS_SZ", "STORM_LH", "STORM_SZ")
#'
#' ## Create tree
#' TreeFormula <- "DAY0 ~ PERS_LH + PERS_SZ + STORM_LH + STORM_SZ"
#' CTreeModel <- createDRCTreeAndValidate(TreeFormula, Tbl)
#'
#' ## Plot tree
#' plot(CTreeModel$CTree)
#'
#' ## Plot observations
#' plotCTreeModelTwoProbOnHzdChart(CTreeModel = CTreeModel, Prob = c("PERS", "STORM"), PlotType = "Observations")
#'
#' ## Plot PERS @ STORM default scenario (LH=P and SZ=2.0)
#' plotCTreeModelTwoProbOnHzdChart(CTreeModel = CTreeModel, Prob = c("PERS", "STORM"))
#'
#' ## Plot STORM @ PERS default scenario (LH=P and SZ=2.0)
#' plotCTreeModelTwoProbOnHzdChart(CTreeModel = CTreeModel, Prob = c("STORM", "PERS"))
#'
#' ## Plot STORM @ PERS with a range of likelihoods
#' ScenProb2 <- list(list(LH="UL", SZ="1.5"), list(LH="P", SZ="1.5"), list(LH="L", SZ="1.5"), list(LH="VL", SZ="1.5"))
#' plotCTreeModelTwoProbOnHzdChart(CTreeModel = CTreeModel, Prob = c("STORM", "PERS"), ScenProb2 = ScenProb2)
#'
#' ## Plot STORM @ PERS with a range of sizes
#' ScenProb2 <- list(list(LH="P", SZ="1.0"), list(LH="P", SZ="2.0"), list(LH="P", SZ="3.0"), list(LH="P", SZ="4.0"))
#' plotCTreeModelTwoProbOnHzdChart(CTreeModel = CTreeModel, Prob = c("STORM", "PERS"), ScenProb2 = ScenProb2)
#'
#' @export

plotCTreeModelTwoProbOnHzdChart <- function(CTreeModel, Prob, ScenProb2=list(list(LH="P", SZ="2.0")), ScenGen=list(), PlotType="Predictions", OnlyWhereObs=TRUE, WithNodeNum=TRUE, WithProb=TRUE, RuleLwd=3, DRClrProdSum=TRUE, DRClr=getHzdColor(c(1, 2, 3, 4)), DRClrTransp=50, main="") {

  ## Preparation
  ## ***********

  warning("Variable names are not checked! Be careful when you specify the scenarios!", immediate. = TRUE)

  ## Extract relevant elements
  CTree <- CTreeModel$CTree
  Data <- CTreeModel$TrainData

  ## Check whether general scenario is included
  if (length(ScenGen)>0) {
    WithScenario <- TRUE
  } else {
    WithScenario <- FALSE
  }

  ## Create title string
  if (WithScenario) {
    TitleStart <- paste(ScenGen, collapse = " - ")
  } else {
    TitleStart <- "No additional parameters"
  }

  ## Filters data down to given general scenario
  if(WithScenario) {
    for (i in 1:length(ScenGen)) {
      Data <- Data[Data[,names(ScenGen)[i]]==ScenGen[[i]],]
    }
  }

  ## Summarize observations
  ## **********************

  ## Count observation for different hazard ratings
  ObsSummaryList <- list()

  for (p in 1:length(Prob)) {

    ## Set up dataframe
    LHColName <- paste0(Prob[p], "_LH")
    SZColName <- paste0(Prob[p], "_SZ")
    Temp <- data.frame(expand.grid(LH=levels(Data[,LHColName]), SZ=levels(Data[,SZColName])))
    Temp$ObsCountL <- 0
    Temp$ObsCountM <- 0
    Temp$ObsCountC <- 0
    Temp$ObsCountH <- 0

    ## Count observations
    for (i in 1:nrow(Data)) {

      TempIndex <- which(as.character(Temp$LH)==as.character(Data[i,LHColName]) & as.character(Temp$SZ)==as.character(Data[i, SZColName]))

      if(Data$DAY0[i]=="Low") {
        Temp$ObsCountL[TempIndex] <- Temp$ObsCountL[TempIndex] + 1
      } else if(Data$DAY0[i]=="Moderate") {
        Temp$ObsCountM[TempIndex] <- Temp$ObsCountM[TempIndex] + 1
      } else if(Data$DAY0[i]=="Considerable") {
        Temp$ObsCountC[TempIndex] <- Temp$ObsCountC[TempIndex] + 1
      } else if(Data$DAY0[i]=="High/Extreme") {
        Temp$ObsCountH[TempIndex] <- Temp$ObsCountH[TempIndex] + 1
      }
    }
    rm(TempIndex, i)
    Temp$ObsCountTotal <- Temp$ObsCountL + Temp$ObsCountM + Temp$ObsCountC + Temp$ObsCountH

    ObsSummaryList[[p]] <- Temp
    rm (Temp)

  }

  ## Plotting
  ## ********

  ## OBSERVATION PLOT
  if (toupper(substr(PlotType, 1, 3))=="OBS") {

    ## Plotting observations
    if (length(Prob)>1) {
      par(mfrow=c(1,length(Prob)))
    }

    for (p in 1:length(Prob)) {

      ## Base plot
      plotHzdChartBase()
      if (main[1]=="") {
        title(paste0("Obs: ", Prob[p], " of ", paste(Prob, collapse = " & "), " (", TitleStart, ")"))
      } else {
        title(main[p])
      }

      ## plot observations
      ObsSummary <- ObsSummaryList[[p]]
      ObsSummary <- ObsSummary[ObsSummary$ObsCountTotal>0,]
      for (i in 1:nrow(ObsSummary)) {
        Prop <- as.numeric(ObsSummary[i,c("ObsCountL", "ObsCountM", "ObsCountC", "ObsCountH")])
        add.pie(z=Prop, x = as.numeric(ObsSummary$SZ[i]), y = as.numeric(ObsSummary$LH[i]), radius=0.25, labels = NA, col = DRClr)
      }
      text(as.numeric(ObsSummary$SZ)+0.3, as.numeric(ObsSummary$LH)+0.3, labels = ObsSummary$ObsCountTotal, cex=.75)

    }

    ## Reset
    if (length(Prob)>1) {
      par(mfrow=c(1,1))
    }

    ## Output
    NewDataList <- NULL


  ## PREDICTION PLOT
  } else if (toupper(substr(PlotType, 1, 4))=="PRED") {

    ## Plot layout
    NumScenarios <- length(ScenProb2)

    if (NumScenarios==1) {
      ## par(mfrow=c(1,1))
    } else if (NumScenarios==2) {
      par(mfrow=c(1,2))
    } else if (NumScenarios<=4) {
      par(mfrow=c(2,2))
    } else if (NumScenarios<=6) {
      par(mfrow=c(2,3))
    } else if (NumScenarios<=9) {
      par(mfrow=c(3,3))
    } else {
      stop("You can only include up to 9 problem 2 scenarios!")
    }

    ## Calculate scenarios
    NewDataList <- list()

    for (s in 1:NumScenarios) {

      ScenProb2Spec <- ScenProb2[[s]]

      cat(paste0("Plotting problem 2 scenario ", s, "...\n"))

      ## Col names
      LHColName <- paste0(Prob[1], "_LH")
      SZColName <- paste0(Prob[1], "_SZ")

      ## Create new data
      NewData <- data.frame(expand.grid(LH=levels(Data[,LHColName]), SZ=levels(Data[,SZColName])))
      NewData$LH <- ordered(NewData$LH, levels=levels(Data[,LHColName]))
      NewData$SZ <- ordered(NewData$SZ, levels=levels(Data[,SZColName]))
      names(NewData) <- c(LHColName, SZColName)

      ## Add problem 2
      LHColName <- paste0(Prob[2], "_LH")
      SZColName <- paste0(Prob[2], "_SZ")
      NewData[[LHColName]] <- ScenProb2Spec$LH
      NewData[[SZColName]] <- ScenProb2Spec$SZ
      NewData[[LHColName]] <- ordered(NewData[[LHColName]], levels=levels(Data[,LHColName]))
      NewData[[SZColName]] <- ordered(NewData[[SZColName]], levels=levels(Data[,SZColName]))

      Point_X <- NewData[[SZColName]][1]
      Point_Y <- NewData[[LHColName]][1]

      ## Identify combinations where the scanerio exists
      ScenProb2Data <- Data[Data[[LHColName]] == ScenProb2Spec$LH & Data[[SZColName]] == ScenProb2Spec$SZ,]
      LHColName <- paste0(Prob[1], "_LH")
      SZColName <- paste0(Prob[1], "_SZ")
      ScenProb2Data$Comb <- paste(ScenProb2Data[[LHColName]], ScenProb2Data[[SZColName]], sep = " - ")

      ## Adds all of the necessary variables and formats them properly
      if (WithScenario) {
        for (i in 1:length(ScenGen)) {
          NewData[,names(ScenGen)[i]] <- ScenGen[[i]]
          if(class(Data[,names(ScenGen)[i]])[1]=="factor") {
            NewData[,names(ScenGen)[i]] <- factor(NewData[,names(ScenGen)[i]], levels=levels(Data[,names(ScenGen)[i]]))
          } else if (class(Data[,names(ScenGen)[i]])[1]=="ordered") {
            NewData[,names(ScenGen)[i]] <- ordered(NewData[,names(ScenGen)[i]], levels=levels(Data[,names(ScenGen)[i]]))
          }
        }
        rm(i)
      }

      ## Calculates predictions
      Predict.Rate <- predict(CTree, newdata = NewData, type = "response")
      Predict.Dist <- predict(CTree, newdata = NewData, type = "prob")
      Predict.Node <- predict(CTree, newdata = NewData, type = "node")

      ## Adds to NewData
      NewData$PreRate <- Predict.Rate
      NewData$PreProbL <- Predict.Dist[,1]
      NewData$PreProbM <- Predict.Dist[,2]
      NewData$PreProbC <- Predict.Dist[,3]
      NewData$PreProbH <- Predict.Dist[,4]
      NewData$PreNode <- Predict.Node
      rm(Predict.Dist, Predict.Rate, Predict.Node)

      ## Calculates single numerical rating for gradual colors
      NewData$PreNumRate <- NewData$PreProbL*1 + NewData$PreProbM*2 + NewData$PreProbC*3 + NewData$PreProbH*4
      NewData$PreNumRateCat <- cut(NewData$PreNumRate, c(1, seq(1.25, 3.75, 0.5), 4), include.lowest = TRUE)
      RatingColPalette <- colorRampPalette(colors = DRClr)(7)

      ## Adds number of observations of first problem
      LHColName <- paste0(Prob[1], "_LH")
      SZColName <- paste0(Prob[1], "_SZ")
      NewData <- merge(NewData, ObsSummaryList[[1]], by.x = c(LHColName, SZColName), by.y = c("LH", "SZ"), all.x = TRUE)

      ## Base plot
      Title <- paste0(Prob[1], " @ ", Prob[2], " with LH=", ScenProb2Spec$LH, " and SZ=", ScenProb2Spec$SZ, " (", TitleStart, ")")
      plotHzdChartBase()

      if(main[1]=="") {
        title(Title)
      } else {
        title(main[s])
      }

      ## Filter data
      if(OnlyWhereObs) {
        NewData <- NewData[NewData$ObsCountTotal>0,]
        LHColName <- paste0(Prob[1], "_LH")
        SZColName <- paste0(Prob[1], "_SZ")
        NewData$Exists[paste(NewData[[LHColName]], NewData[[SZColName]], sep = " - ") %in% ScenProb2Data$Com] <- 1
      }

      ## Col names
      LHColName <- paste0(Prob[1], "_LH")
      SZColName <- paste0(Prob[1], "_SZ")

      ## Rating background
      if (DRClrProdSum) {
        rect(xleft = as.numeric(NewData[[SZColName]][!is.na(NewData$Exists)])-0.5, xright = as.numeric(NewData[[SZColName]][!is.na(NewData$Exists)])+0.5, ybottom = as.numeric(NewData[[LHColName]][!is.na(NewData$Exists)])-0.5, ytop = as.numeric(NewData[[LHColName]][!is.na(NewData$Exists)])+0.5, col = paste0(RatingColPalette[as.numeric(NewData$PreNumRateCat[!is.na(NewData$Exists)])], as.character(DRClrTransp)), border = NA)
        rect(xleft = as.numeric(NewData[[SZColName]][is.na(NewData$Exists)])-0.5, xright = as.numeric(NewData[[SZColName]][is.na(NewData$Exists)])+0.5, ybottom = as.numeric(NewData[[LHColName]][is.na(NewData$Exists)])-0.5, ytop = as.numeric(NewData[[LHColName]][is.na(NewData$Exists)])+0.5, col = paste0("#A9A9A9", as.character(DRClrTransp)), border = NA)
      } else {
        rect(xleft = as.numeric(NewData[[SZColName]][!is.na(NewData$Exists)])-0.5, xright = as.numeric(NewData[[SZColName]][!is.na(NewData$Exists)])+0.5, ybottom = as.numeric(NewData[[LHColName]][!is.na(NewData$Exists)])-0.5, ytop = as.numeric(NewData[[LHColName]][!is.na(NewData$Exists)])+0.5, col = paste0(DRClr[as.numeric(NewData$PreRate[!is.na(NewData$Exists)])], as.character(DRClrTransp)), border = NA)
        rect(xleft = as.numeric(NewData[[SZColName]][is.na(NewData$Exists)])-0.5, xright = as.numeric(NewData[[SZColName]][is.na(NewData$Exists)])+0.5, ybottom = as.numeric(NewData[[LHColName]][is.na(NewData$Exists)])-0.5, ytop = as.numeric(NewData[[LHColName]][is.na(NewData$Exists)])+0.5, col = paste0("#A9A9A9", as.character(DRClrTransp)), border = NA)
      }

      ## Reduce NewData to only existing combinations
      ## NewData <- NewData[!is.na(NewData$Exists),]

      if(nrow(NewData) > 0) {

        ## Plotting of pie charts
        if(WithProb) {
          for (i in 1:nrow(NewData)) {
            PredProp <- as.numeric(round(NewData[i,c("PreProbL", "PreProbM", "PreProbC", "PreProbH")]*100,0))
            if (is.na(NewData$Exists[i])) {
              add.pie(z=PredProp, x = as.numeric(NewData[i, SZColName]), y = as.numeric(NewData[i, LHColName]), radius=0.25, labels = NA, col = rev(grey.colors(4)))
            } else {
              add.pie(z=PredProp, x = as.numeric(NewData[i, SZColName]), y = as.numeric(NewData[i, LHColName]), radius=0.25, labels = NA, col = DRClr)
            }
          }
          if(WithNodeNum) {
            text(as.numeric(NewData[[SZColName]])+0.3, as.numeric(NewData[[LHColName]])+0.3, labels = NewData$PreNode, cex=0.75)
          }
        } else if (WithNodeNum) {
          text(as.numeric(NewData[[SZColName]]), as.numeric(NewData[[LHColName]]), labels = NewData$PreNode, cex=1)
        }

        ## Horizontal and verical lines between rules
        if(WithNodeNum) {
          for (lh in 1:9) {
            for (sz in 1:8) {
              NodeCurr <- NewData$PreNode[(as.numeric(NewData[[LHColName]])==lh & as.numeric(NewData[[SZColName]])==sz)]
              NodeNext <- NewData$PreNode[(as.numeric(NewData[[LHColName]])==lh & as.numeric(NewData[[SZColName]])==sz+1)]
              if(length(NodeCurr)>0 & length(NodeNext)>0) {
                if(NodeCurr != NodeNext) {
                  arrows(x0 = sz+0.5, x1 = sz+0.5, y0 = lh-0.5, y1 = lh+0.5, lwd=RuleLwd, length = 0)
                }
              }
            }
          }
          for (lh in 1:8) {
            for (sz in 1:9) {
              NodeCurr <- NewData$PreNode[(as.numeric(NewData[[LHColName]])==lh & as.numeric(NewData[[SZColName]])==sz)]
              NodeNext <- NewData$PreNode[(as.numeric(NewData[[LHColName]])==lh+1 & as.numeric(NewData[[SZColName]])==sz)]
              if(length(NodeCurr)>0 & length(NodeNext)>0) {
                if(NodeCurr != NodeNext) {
                  arrows(x0 = sz-0.5, x1 = sz+0.5, y0 = lh+0.5, y1 = lh+0.5, lwd=RuleLwd, length = 0)
                }
              }
            }
          }
        }

      }

      ## Plotting location of Problem 2
      # LHColName <- paste0(Prob[2], "_LH")
      # SZColName <- paste0(Prob[2], "_SZ")
      # points(x=NewData[1,SZColName], y=NewData[1,LHColName], pch=23, bg="blue", cex=3, lwd=5, col="white")
      points(x=Point_X, y=Point_Y, pch=23, bg="blue", cex=3, lwd=5, col="white")

      box()

      ## Pass New data
      NewDataList[[s]] <- list(ScenProb2=ScenProb2Spec,
                               Predictions=NewData)

    } ## End of scenario loop

    ## Reset
    if (NumScenarios>1) {
      par(mfrow=c(1,1))
    }

  } ## End of PlotType if-else

  ## Output
  invisible(list(Prob=Prob,
                 ScenGen=ScenGen,
                 Obs=ObsSummaryList,
                 Pred=NewDataList))

}
