#' Creates season plot for a group of forecast regions
#'
#' Creates season plot of avalanche bulletins
#' @param bulletins getBulletins() object
#' @param StartDate Start date in dates() format i.e. dates("2015-11-01", format="y-m-d").
#' @param EnDate End date in dates() format i.e. dates("2016-05-01", format="y-m-d").
#'
#' @examples
#' 
#' require(SarpBulletinTools)
#' require(SarpBulletinToolsPrivate)
#' require(SarpGeneral)
#' if (!require(tidyr)) {install.packages("tidyr"); require(tidyr)}             ## package for data wrangling
#' if (!require(DescTools)) {install.packages("DescTools"); require(DescTools)} ## package for data wrangling
#' 
#' 
#' ## Inputs
#' Regions    <- "Jasper"
#' season     <- 2017
#' bulletins <- getBulletins(Agency = "PkCan",
#'                           Regions = Regions,
#'                           Seasons = season)
#' 
#' StartDate  <- dates(paste0(season-1,"-11-01"), format="y-m-d")
#' EndDate    <- dates(paste0(season,"-05-01"), format="y-m-d")
#' 
#' ## PLOT
#' OutputFilename <- paste0("C:/Users/bshan/sfuvault/Documents/ParksCanada/",season,"_", Regions,".png")
#' 
#' png(filename = OutputFilename, width=16.18, height=10, units="cm", pointsize=8, bg="white", res=600)
#' 
#' plotDngProb_TS(bulletins = bulletins,
#'                StartDate = StartDate,
#'                EndDate = EndDate)
#' #END PNG
#' dev.off()
#' 
#' @export

plotDngProb_TS <- function(bulletins, StartDate, EndDate){

  bulletins$DngRating$Alp$DAY0 <- factor(bulletins$DngRating$Alp$DAY0, ordered = TRUE, levels = c("No Rating","'Spring'","Low","Moderate","Considerable","High","Extreme"))
  bulletins$DngRating$Tl$DAY0 <- factor(bulletins$DngRating$Tl$DAY0, ordered = TRUE, levels = c("No Rating","'Spring'","Low","Moderate","Considerable","High","Extreme"))
  bulletins$DngRating$Btl$DAY0 <- factor(bulletins$DngRating$Btl$DAY0, ordered = TRUE, levels = c("No Rating","'Spring'","Low","Moderate","Considerable","High","Extreme"))
  
  hzd.all <- list(Alp = NA,
                  Tl = NA,
                  Btl = NA)
  
  for(elv in names(hzd.all)){
    # format data to tally problems
    problems <- bulletins$AvProblems[[elv]][,c('BULLETIN_ID','CHARACTER')]
    problems <- problems[!duplicated(problems),]
    problems$value <- 1
    
    problem.wide <- spread(problems, CHARACTER, value)
    problem.wide[is.na(problem.wide)]<-0  
    
    problem.wide <- merge(x = problem.wide,
                              y = bulletins$Bulletin[,c("BULLETIN_ID", "PUBLISH_DATE")],
                              by = "BULLETIN_ID")
    problem.wide$count <- 1
    
    prob <- problem.wide[,!names(problem.wide) %in% c("BULLETIN_ID")]
    prob <- aggregate( .~ PUBLISH_DATE, data = prob, sum)
    
    # one line per day
    names(problem.wide) <- gsub(" ", "", names(problem.wide))
    
    dr <- bulletins$DngRating[[elv]][,c('BULLETIN_ID','DAY0')]
    dr <- dr[!duplicated(dr$BULLETIN_ID),]
    dr <- merge(dr, bulletins$Bulletin[,c("BULLETIN_ID", "PUBLISH_DATE")])
    dr <- aggregate(DAY0 ~ PUBLISH_DATE, data = dr, median.factor)
    
    # merge in danger rating
    prob <- merge(x = prob,
                  y = dr,
                  by = "PUBLISH_DATE", all.y = TRUE)
    prob[is.na(prob)] <- 0
    
    prob$DAY0 <- as.character(prob$DAY0)
    prob$PUBLISH_DATE <- as.Date(prob$PUBLISH_DATE, format = "%Y/%m/%d") 
    
    # ensure only one problem per day of each type
    denom <- prob[,"count"]
    denom[denom==0] <- 1
    prob <- cbind(prob[,c('PUBLISH_DATE','DAY0')], prob[,names(prob)[!names(prob) %in% c('PUBLISH_DATE','DAY0','count')]] / denom)
    names(prob) <- gsub(" ","",names(prob))
    hzd.all[[elv]] <- prob
    rm(prob)
    rm(problem.wide)
    rm(dr)
    rm(denom)
  }
  prob.alp <- hzd.all[["Alp"]]
  prob.tl <- hzd.all[["Tl"]]
  prob.btl <- hzd.all[["Btl"]]
  
 
  ## Functions for plotting 
  
  # Plotting of avalanche problems
  plotAvProblem <- function(DateArray, ProbArray, YAxis, pch,Color) {
    # color depending on prevalence
    # Plot points
    trans <- as.character(cut(ProbArray[ProbArray>0], breaks = c(0,0.34,0.67,0.99,1), labels = c(40,60,50,99)))
    Color <- paste0(Color,trans)
    points(DateArray[ProbArray>0], rep(YAxis, sum(ProbArray>0)), pch=pch, cex=.9, col=NA, bg = Color)
  }
  
  ## Constructing the plot
  
  StartDate <- as.Date(StartDate)
  EndDate   <- as.Date(EndDate)
  # Output device
  # ChartWidth <- 16
  # ChartHeight <- 10
  # png(paste0(Output.dir, "/",filename, ".png"), width=ChartWidth, height=ChartHeight, units="cm", pointsize=8, bg="white", res=600)
  
  # par defaults and chart size and margins
  def.par <- par(no.readonly = TRUE) # save default, for resetting...
  
  Margins   <- c(0.75,   0.1,0.75,0.1)
  
  par(mai=Margins)
  
  plot(x = c(StartDate:EndDate),
       y = rep(3,length(c(StartDate:EndDate))),
       type='n',
       axes=FALSE,
       ann=FALSE,
       xlim = c(StartDate,EndDate),
       ylim = c(0,11))
  
  # Grid lines
  abline(h=c(0:8), col="#636363", lty=2, lwd=0.5)
  abline(v=seq.Date(from = StartDate,
                    to = EndDate,
                    by = "weeks"),
         col="#cccccc", 
         lty=2, 
         lwd=0.5)
  abline(v=seq.Date(from = StartDate,
                    to = EndDate,
                    by = "months"),
         col="#636363", 
         lty=5, 
         lwd=0.5)
  for(d in 1:length(seq.Date(from = StartDate, to = EndDate, by = 1))){
    
    vecDate <- seq.Date(from = StartDate, to = EndDate, by = 1)
    day <- vecDate[d]
    
    rect(xleft = day-0.5,
         ybottom = 10,
         xright = 0.5+day,
         ytop = 11,
         col= ifelse(!is.na(prob.alp[prob.alp$PUBLISH_DATE == day,"DAY0"]), getHzdColor(prob.alp[prob.alp$PUBLISH_DATE == day,"DAY0"]),NA),
         lwd=0)
    
    rect(xleft = day-0.5,
         ybottom = 9,
         xright = 0.5+day,
         ytop = 10,
         col= ifelse(!is.na(prob.tl[prob.tl$PUBLISH_DATE == day,"DAY0"]), getHzdColor(prob.tl[prob.tl$PUBLISH_DATE == day,"DAY0"]),NA),
         lwd=0)
    
    rect(xleft = day-0.5,
         ybottom = 8,
         xright = 0.5+day,
         ytop = 9,
         col= ifelse(!is.na(prob.btl[prob.btl$PUBLISH_DATE == day,"DAY0"]), getHzdColor(prob.btl[prob.btl$PUBLISH_DATE == day,"DAY0"]),NA),
         lwd=0)
    
  }
  # add Danger rating lables
  text(x = StartDate -7, y = 8.5, labels = "Below Treeline", pos = 4)
  text(x = StartDate -7, y = 9.5, labels = "Treeline", pos = 4)
  text(x = StartDate -7, y = 10.5, labels = "Alpine", pos = 4)

  
  ## Avalanche problems
  
  # labels
  text(x = StartDate + 14, y = 7.25, labels = "BTL", pos = 4, cex =0.75)
  text(x = StartDate + 14, y = 7.5, labels = "TL", pos = 4, cex =0.75)
  text(x = StartDate + 14, y = 7.75, labels = "ALP", pos = 4, cex =0.75)
  
  avp.levels  <- c("Deep Persistent Slabs","Persistent Slabs","Storm Slabs","Wind Slabs","Cornices","Wet Slabs","Loose Wet","Loose Dry")
  
  for(avp in 1:length(avp.levels)){
    if(gsub(" ","",avp.levels[avp]) %in% names(prob.alp)){
      plotAvProblem(prob.alp$PUBLISH_DATE, 
                    prob.alp[,gsub(" ","",avp.levels[avp])], 
                    avp-1+0.75, 
                    pch = 24, 
                    Color = getAvCharColor(avp.levels[avp]))
    }
    if(gsub(" ","",avp.levels[avp]) %in% names(prob.tl)){
      plotAvProblem(prob.tl$PUBLISH_DATE, 
                    prob.tl[,gsub(" ","",avp.levels[avp])], 
                    avp-1+0.5, 
                    pch = 22, 
                    Color = getAvCharColor(avp.levels[avp]))
    }
    if(gsub(" ","",avp.levels[avp]) %in% names(prob.btl)){
      plotAvProblem(prob.btl$PUBLISH_DATE, 
                    prob.btl[,gsub(" ","",avp.levels[avp])], 
                    avp-1+0.25, 
                    pch = 25, 
                    Color = getAvCharColor(avp.levels[avp]))
    }
    
    # add avalanche probelm lables
    text(x = StartDate -7,
         y = avp-1+0.5,
         labels = avp.levels[avp],
         pos = 4)
  }
  
  ## Date axis
  axis.Date(3, 
            at = seq.Date(from = StartDate,
                          to = EndDate,
                          by = "weeks"),
            format = "%d-%m-%y",
            las=2)
  axis.Date(1, 
            at = seq.Date(from = StartDate+15,
                          to = EndDate+15,
                          by = "months"),
            labels = months(seq.Date(from = StartDate,
                                     to = EndDate,
                                     by = "months")),
            tick = FALSE,
            las=1)
  axis.Date(1, 
            at = seq.Date(from = StartDate,
                          to = EndDate,
                          by = "months"),
            labels = "",
            tick = TRUE,
            las=1)
  
  
  box(col="#252525")
}