#' Creates season plot to compare forecast regions
#'
#' Creates season plot to compare forecast regions
#' @param bulletins getBulletins() object
#' @param StartDate Start date in dates() format i.e. dates("2015-11-01", format="y-m-d").
#' @param EnDate End date in dates() format i.e. dates("2016-05-01", format="y-m-d").
#'
#' @examples
#' 
#' require(SarpBulletinTools)
#' require(SarpBulletinToolsPrivate)
#' require(SarpGeneral)
#' if (!require(tidyr)) {install.packages("tidyr"); require(tidyr)}             ## package for data wrangling
#' if (!require(DescTools)) {install.packages("DescTools"); require(DescTools)} ## package for data wrangling
#' 
#' 
#' ## Inputs
#' avp.levels  <- c("Deep Persistent Slabs","Persistent Slabs","Storm Slabs","Wind Slabs","Cornices","Wet Slabs","Loose Wet","Loose Dry")
#' AvalancheProblem <- avp.levels[4]
#' Regions1    <- "Jasper"
#' Regions2    <- "Banff, Yoho and Kootenay"
#' Regions3    <- "Kananaskis Country"
#' season     <- 2017
#' 
#' bulletins1 <- getBulletins(Agency = "PkCan",
#'                           Regions = Regions1,
#'                           Seasons = season)
#' 
#' bulletins2 <- getBulletins(Agency = "PkCan",
#'                            Regions = Regions2,
#'                            Seasons = season)
#' 
#' bulletins3 <- getBulletins(Agency = "AvCan",
#'                            Regions = Regions3,
#'                            Seasons = season)
#' 
#' StartDate  <- dates(paste0(season-1,"-11-01"), format="y-m-d")
#' EndDate    <- dates(paste0(season,"-05-01"), format="y-m-d")
#' 
#' ## PLOT
#' OutputFilename <- paste0("C:/Users/bshan/sfuvault/Documents/ParksCanada/", AvalancheProblem, ".png")
#' 
#' png(filename = OutputFilename, width=16.18, height=5.5, units="cm", pointsize=8, bg="white", res=600)
#' 
#' plotDngProb_TS_COMP(bulletins1 = bulletins1,
#'                     bulletins2 = bulletins2,
#'                     bulletins3 = bulletins3,
#'                     AvalancheProblem = AvalancheProblem,
#'                     StartDate = StartDate,
#'                     EndDate = EndDate)
#' #END PNG
#' dev.off()
#' 
#' @export

plotDngProb_TS_COMP <- function(bulletins1, bulletins2, bulletins3, AvalancheProblem, StartDate, EndDate){
  
  ## FUNCTION FOR SORTING BULLETIN INFO
  sortBulletins <- function(bulletins){
    hzd.all <- list(Alp = NA,
                Tl = NA,
                Btl = NA)
    
    bulletins$DngRating$Alp$DAY0 <- factor(bulletins$DngRating$Alp$DAY0, ordered = TRUE, levels = c("No Rating","'Spring'","Low","Moderate","Considerable","High","Extreme"))
    bulletins$DngRating$Tl$DAY0 <- factor(bulletins$DngRating$Tl$DAY0, ordered = TRUE, levels = c("No Rating","'Spring'","Low","Moderate","Considerable","High","Extreme"))
    bulletins$DngRating$Btl$DAY0 <- factor(bulletins$DngRating$Btl$DAY0, ordered = TRUE, levels = c("No Rating","'Spring'","Low","Moderate","Considerable","High","Extreme"))
    for(elv in names(hzd.all)){# format data to tally problems
      problems <- bulletins$AvProblems[[elv]][,c('BULLETIN_ID','CHARACTER')]
      problems <- problems[!duplicated(problems),]
      problems$value <- 1
      
      problem.wide <- spread(problems, CHARACTER, value)
      problem.wide[is.na(problem.wide)]<-0  
      
      problem.wide <- merge(x = problem.wide,
                            y = bulletins$Bulletin[,c("BULLETIN_ID", "PUBLISH_DATE")],
                            by = "BULLETIN_ID")
      problem.wide$count <- 1
      
      prob <- problem.wide[,!names(problem.wide) %in% c("BULLETIN_ID")]
      prob <- aggregate( .~ PUBLISH_DATE, data = prob, sum)
      
      # one line per day
      names(problem.wide) <- gsub(" ", "", names(problem.wide))
      
      dr <- bulletins$DngRating[[elv]][,c('BULLETIN_ID','DAY0')]
      dr <- dr[!duplicated(dr$BULLETIN_ID),]
      dr <- merge(dr, bulletins$Bulletin[,c("BULLETIN_ID", "PUBLISH_DATE")])
      dr <- aggregate(DAY0 ~ PUBLISH_DATE, data = dr, median.factor)
      
      # merge in danger rating
      prob <- merge(x = prob,
                    y = dr,
                    by = "PUBLISH_DATE", all.y = TRUE)
      prob[is.na(prob)] <- 0
      
      prob$DAY0 <- as.character(prob$DAY0)
      prob$PUBLISH_DATE <- as.Date(prob$PUBLISH_DATE, format = "%Y/%m/%d") 
      
      # ensure only one problem per day of each type
      denom <- prob[,"count"]
      denom[denom==0] <- 1
      prob <- cbind(prob[,c('PUBLISH_DATE','DAY0')], prob[,names(prob)[!names(prob) %in% c('PUBLISH_DATE','DAY0','count')]] / denom)
      names(prob) <- gsub(" ","",names(prob))
      hzd.all[[elv]] <- prob
      rm(prob)
      rm(problem.wide)
      rm(dr)
      rm(denom)
    }
    return(hzd.all)
  }

  ## Functions for plotting 
  
  # Plotting of avalanche problems
  plotAvProblem <- function(DateArray, ProbArray, YAxis, pch,Color) {
    # color depending on prevalence
    # Plot points
    trans <- as.character(cut(ProbArray[ProbArray>0], breaks = c(0,0.34,0.67,0.99,1), labels = c(40,60,50,99)))
    Color <- paste0(Color,trans)
    points(DateArray[ProbArray>0], rep(YAxis, sum(ProbArray>0)), pch=pch, cex=.9, col=NA, bg = Color)
  }
  
  ## SORTING BULLETIN INFO
  list_bul1 <- sortBulletins(bulletins1)
  list_bul2 <- sortBulletins(bulletins2)
  list_bul3 <- sortBulletins(bulletins3)
  
  prob.alp1 <- list_bul1[["Alp"]]
  prob.tl1 <- list_bul1[["Tl"]]
  prob.btl1 <- list_bul1[["Btl"]]
  
  prob.alp2 <- list_bul2[["Alp"]]
  prob.tl2 <- list_bul2[["Tl"]]
  prob.btl2 <- list_bul2[["Btl"]]
  
  prob.alp3 <- list_bul3[["Alp"]]
  prob.tl3 <- list_bul3[["Tl"]]
  prob.btl3 <- list_bul3[["Btl"]]
  
  ## PLOT
  
  StartDate <- as.Date(StartDate)
  EndDate   <- as.Date(EndDate)
  # Output device
  # ChartWidth <- 16
  # ChartHeight <- 10
  # png(paste0(Output.dir, "/",filename, ".png"), width=ChartWidth, height=ChartHeight, units="cm", pointsize=8, bg="white", res=600)
  
  # par defaults and chart size and margins
  def.par <- par(no.readonly = TRUE) # save default, for resetting...
  
  Margins   <- c(0.75,   0.5,0.75,0.1)
  
  par(mai=Margins)
  
  plot(x = c(StartDate:EndDate),
       y = rep(3,length(c(StartDate:EndDate))),
       type='n',
       axes=FALSE,
       ann=FALSE,
       xlim = c(StartDate,EndDate),
       ylim = c(0,3))
  
  # Grid lines
  abline(h=c(0:8), col="#636363", lty=2, lwd=0.5)
  abline(v=seq.Date(from = StartDate,
                    to = EndDate,
                    by = "weeks"),
         col="#cccccc", 
         lty=2, 
         lwd=0.5)
  abline(v=seq.Date(from = StartDate,
                    to = EndDate,
                    by = "months"),
         col="#636363", 
         lty=5, 
         lwd=0.5)

  
  ## Avalanche problems
  
  # labels
  text(x = EndDate-2 , y = 2.2, labels = "BTL", pos = 4, cex =0.75)
  text(x = EndDate-2 , y = 2.5, labels = "TL", pos = 4, cex =0.75)
  text(x = EndDate-2 , y = 2.8, labels = "ALP", pos = 4, cex =0.75)
  

  # plot problem for the FIRST region
  if(gsub(" ","",AvalancheProblem) %in% names(prob.alp1)){
    plotAvProblem(prob.alp1$PUBLISH_DATE, # ALPINE
                  prob.alp1[,gsub(" ","",AvalancheProblem)], 
                  0.75, 
                  pch = 24, 
                  Color = getAvCharColor(AvalancheProblem))
  }
  if(gsub(" ","",AvalancheProblem) %in% names(prob.tl1)){
    plotAvProblem(prob.tl1$PUBLISH_DATE, # TREELINE
                  prob.tl1[,gsub(" ","",AvalancheProblem)], 
                  0.5, 
                  pch = 22, 
                  Color = getAvCharColor(AvalancheProblem))
  }
  if(gsub(" ","",AvalancheProblem) %in% names(prob.btl1)){
    plotAvProblem(prob.btl1$PUBLISH_DATE, # BTL
                  prob.btl1[,gsub(" ","",AvalancheProblem)], 
                  0.25, 
                  pch = 25, 
                  Color = getAvCharColor(AvalancheProblem))
  }
  # add avalanche probelm lables
  text(x = StartDate -7,
       y = 0.5,
       labels = as.character(bulletins1$Bulletins$REGION[1]),
       pos = 4)
  
  # plot problem for the SECOND region
  if(gsub(" ","",AvalancheProblem) %in% names(prob.alp2)){
    plotAvProblem(prob.alp2$PUBLISH_DATE, # ALPINE
                  prob.alp2[,gsub(" ","",AvalancheProblem)], 
                  1+0.75, 
                  pch = 24, 
                  Color = getAvCharColor(AvalancheProblem))
  }
  if(gsub(" ","",AvalancheProblem) %in% names(prob.tl2)){
    plotAvProblem(prob.tl2$PUBLISH_DATE, # TREELINE
                  prob.tl2[,gsub(" ","",AvalancheProblem)], 
                  1+0.5, 
                  pch = 22, 
                  Color = getAvCharColor(AvalancheProblem))
  }
  if(gsub(" ","",AvalancheProblem) %in% names(prob.btl2)){
    plotAvProblem(prob.btl2$PUBLISH_DATE, # BTL
                  prob.btl2[,gsub(" ","",AvalancheProblem)], 
                  1+0.25, 
                  pch = 25, 
                  Color = getAvCharColor(AvalancheProblem))
  }
  # add avalanche probelm lables
  text(x = StartDate -7,
       y = 1+0.5,
       labels = as.character(bulletins2$Bulletins$REGION[1]),
       pos = 4)

  # plot problem for the THIRD region
  if(gsub(" ","",AvalancheProblem) %in% names(prob.alp3)){
    plotAvProblem(prob.alp3$PUBLISH_DATE, # ALPINE
                  prob.alp3[,gsub(" ","",AvalancheProblem)], 
                  2+0.75, 
                  pch = 24, 
                  Color = getAvCharColor(AvalancheProblem))
  }
  if(gsub(" ","",AvalancheProblem) %in% names(prob.tl3)){
    plotAvProblem(prob.tl3$PUBLISH_DATE, # TREELINE
                  prob.tl3[,gsub(" ","",AvalancheProblem)], 
                  2+0.5, 
                  pch = 22, 
                  Color = getAvCharColor(AvalancheProblem))
  }
  if(gsub(" ","",AvalancheProblem) %in% names(prob.btl3)){
    plotAvProblem(prob.btl3$PUBLISH_DATE, # BTL
                  prob.btl3[,gsub(" ","",AvalancheProblem)], 
                  2+0.25, 
                  pch = 25, 
                  Color = getAvCharColor(AvalancheProblem))
  }
  # add avalanche probelm lables
  text(x = StartDate -7,
       y = 2+0.5,
       labels = as.character(bulletins3$Bulletins$REGION[1]),
       pos = 4)
  
  ## Date axis
  axis.Date(3, 
            at = seq.Date(from = StartDate,
                          to = EndDate,
                          by = "weeks"),
            format = "%d-%m-%y",
            las=2)
  axis.Date(1, 
            at = seq.Date(from = StartDate+15,
                          to = EndDate+15,
                          by = "months"),
            labels = months(seq.Date(from = StartDate,
                                     to = EndDate,
                                     by = "months")),
            tick = FALSE,
            las=1)
  axis.Date(1, 
            at = seq.Date(from = StartDate,
                          to = EndDate,
                          by = "months"),
            labels = "",
            tick = TRUE,
            las=1)
  axis(2, at = 1.5,labels = AvalancheProblem, tick = FALSE)
  
  box(col="#252525")
}