#' Plot hazard chart from avalanche problem component of hazard assessment object.
#'
#' Plot hazard chart from avalanche problem component of hazard assessment object.
#' @param AvProblems DF with avalanche problems
#' @param Title Title of chart. Default value is 'Hazard Chart'.
#' @param WithPoints Switch whether chart should be plotted with points. Default is TRUE.
#' @param Transparency The transparency of the blob background. Value can be between 00 and 99. Default value is 50.
#' @param Shape For specifying the shape of the blob. Values can be 'ellipse' or 'square' (default).
#' @param NumPoints Number of point per quater section of ellipse. Default value is 50.
#' @param MedianShape Switch for showing only median ellipse only. Default value is F.
#' @param SuppressWarnings Switch for suppressing warning message about application of is.na to S4 object. Default is TRUE.
#'
#' @examples
#' require(SarpBulletinToolsPrivate)
#' require(SarpBulletinTools)
#' 
#' Operation <- "AvCan"
#' Season <- 2015
#' Region <- "Sea-to-Sky"
#' 
#' ## Entire Season - alpine elevation band
#' BulletinsSeason <- getBulletins(Agency=Operation, Seasons=Season, Regions=Region)
#' AvProblemsAlp <- BulletinsSeason$AvProblems$Alp
#' 
#' plotHzdChart(AvProblems = AvProblemsAlp[AvProblemsAlp$CHARACTER=="Wind Slabs",], Title="Wind Slabs")
#' plotHzdChart(AvProblemsAlp[AvProblemsAlp$CHARACTER=="Deep Persistent Slabs",], Title="Deep Persistent Slabs")
#' plotHzdChart(AvProblemsAlp[AvProblemsAlp$CHARACTER=="Persistent Slabs",], Title="Persistent Slabs")
#'
#' ## Median chart
#' plotHzdChart(AvProblemsAlp[AvProblemsAlp$CHARACTER=="Wind Slabs",], Title="Wind Slabs - Median", MedianShape=T)
#' plotHzdChart(AvProblemsAlp, Title="All Avalanche Problems - Median", MedianShape=T)
#'
#' ## Single day
#' Date <- dates("2015-02-05", format="y-m-d")
#' BulletinSingleDay <- getBulletins(Agency=Operation, StartDate=Date, Regions=Region)
#' summary(BulletinSingleDay)
#'
#' plotHzdChart(BulletinSingleDay$AvProblems$Alp, Title=paste0(Region, " - ALP - ", Date))
#' plotHzdChart(BulletinSingleDay$AvProblems$Tl, Title=paste0(Region, " - TL - ", Date))
#' plotHzdChart(BulletinSingleDay$AvProblems$Btl, Title=paste0(Region, " - BTL - ", Date))
#'
#' @export

plotHzdChart <- function(AvProblems, 
                         AvProblemsFaded=NULL, 
                         Title="Hazard Chart", 
                         WithPoints=T, 
                         Transparency='50', 
                         Shape='square', 
                         NumPoints=50, 
                         MedianShape=F, 
                         SuppressWarnings=T, 
                         cex.AdditionalText=0.66,
                         sumProb=T,
                         ylab="Likelihood of Avalanche(s)",
                         xlab="Destructive Size",
                         cex.lab = 0.5,
                         cex.lab.a = 1) {

  ## Number of problems
  NumAvProblems <- nrow(AvProblems)

  ## Calculate median if MedianShape==T
  if(MedianShape) {

    if(length(unique(AvProblems$CHARACTER))==1) {
      MedianCharacter <- AvProblems$CHARACTER[1]
    } else {
      MedianCharacter <- "Multiple"
    }

    AvProblems <- data.frame(PROBLEM_ID = createUuid(),
                             CHARACTER = MedianCharacter,
                             LIKELIHOOD_MIN=median(AvProblems$LIKELIHOOD_MIN),
                             LIKELIHOOD_TYP=median(AvProblems$LIKELIHOOD_TYP),
                             LIKELIHOOD_MAX=median(AvProblems$LIKELIHOOD_MAX),
                             SIZE_MIN=median(AvProblems$SIZE_MIN),
                             SIZE_TYP=median(AvProblems$SIZE_TYP),
                             SIZE_MAX=median(AvProblems$SIZE_MAX))

  }

  ## Convert avalanche problems to SPDF
  AvProbSPDF <- convertAvProbToSPDF(AvProblems=AvProblems, Shape=Shape, NumPoints=NumPoints)
  AvProbFadedSPDF <- convertAvProbToSPDF(AvProblems=AvProblemsFaded, Shape=Shape, NumPoints=NumPoints)
  
  if (SuppressWarnings) {options(warn=-1)}

  ## Calculate colors
  if(!is.na(AvProbSPDF)) {
    AvProbSPDF@data$ColBord <- sapply(AvProbSPDF@data$CHARACTER, function(x) getAvCharColor(x))
    AvProbSPDF@data$ColPoly <- paste0(AvProbSPDF@data$ColBord, Transparency)
  }
  
  if(!is.na(AvProbFadedSPDF)) {
    FadedBaseCol <- sapply(AvProbFadedSPDF@data$CHARACTER, function(x) getAvCharColor(x))
    AvProbFadedSPDF@data$ColBord <- paste0(FadedBaseCol, '50')
    AvProbFadedSPDF@data$ColPoly <- paste0(FadedBaseCol, '25')
    rm(FadedBaseCol)
  }
  
  ## Dummy polygon outside plotting area for creating empty chart
  DummySpObjWKT <- "POLYGON ((-1 -1, -1 -2, -2 -2, -2 -1, -1 -1))"
  DummySpObj <- rgeos::readWKT(DummySpObjWKT, id="Dummy")

  ## Base chart without any blobs
  sp::plot(DummySpObj, xlim=c(1,9), ylim=c(1,9), yaxt="n", ylab=ylab, xaxt="n", xlab=xlab, cex.lab=cex.lab.a)
  axis(1, at=c(1,3,5,7,9), labels=c("Sz 1", "Sz 2", "Sz 3", "Sz 4", "Sz 5"), cex.lab=cex.lab)
  axis(2, at=c(1,3,5,7,9), labels=c("Unl", "Poss", "Lik", "VLik", "ACer"), cex.lab=cex.lab)
  title(Title)
  abline(h = c(1:9), lty=2, lwd=0.5, col="dark grey")
  abline(v = c(1:9), lty=2, lwd=0.5, col="dark grey")
  box()

  if(sumProb){
    mtext(paste("Total number of avalanche problems:", NumAvProblems), side=3, adj=0, cex=cex.AdditionalText)
  }
  
  if (MedianShape) {
    mtext("MEDIAN SHAPE", side=3, adj=1, cex=cex.AdditionalText)
  }

  ## Plotting of faded blobs
  if(!is.na(AvProbFadedSPDF)) {
    sp::plot(AvProbFadedSPDF, add=T, col=AvProbFadedSPDF@data$ColPoly, border=AvProbFadedSPDF@data$ColBord, lwd=2)
  }
  
  ## Plotting of blobs
  if(!is.na(AvProbSPDF)) {
    sp::plot(AvProbSPDF, add=T, col=AvProbSPDF@data$ColPoly, border=AvProbSPDF@data$ColBord, lwd=3)
  } else {
    text(5,5,"No avalanche problems specified!", adj=0.5)
  }

  ## Plotting of points
  if (WithPoints & !is.na(AvProbSPDF)) {

    for(Index_AvProb in 1:nrow(AvProbSPDF)) {

      Points <- data.frame(x=c(AvProbSPDF@data$size_min[Index_AvProb], AvProbSPDF@data$size_typ[Index_AvProb], AvProbSPDF@data$size_max[Index_AvProb], AvProbSPDF@data$size_typ[Index_AvProb], AvProbSPDF@data$size_typ[Index_AvProb]),
                           y=c(AvProbSPDF@data$like_typ[Index_AvProb], AvProbSPDF@data$like_max[Index_AvProb], AvProbSPDF@data$like_typ[Index_AvProb], AvProbSPDF@data$like_min[Index_AvProb], AvProbSPDF@data$like_typ[Index_AvProb]))


      points(Points, pch=19, cex=1.25, col=AvProbSPDF@data$ColBord[Index_AvProb])

    }

  }

  if (SuppressWarnings) {options(warn=-1)}

}
