#' Plot raster in hazard chart with counts or percentages of avalanche problems.
#'
#' Plot raster in hazard chart with counts or percentages of avalanche problems.
#' @param AvProblems DF with avalanche problems
#' @param Title Title of chart. Default value is 'Hazard Chart'.
#' @param WithContours Switch whether chart should be plotted with contour lines. Default is TRUE.
#' @param Percent Switch whether plotted values should be percentages instead of counts. Default is F.
#' @param PercentInterval If percentage values are plotted with contour lines, this parameter specifies the contour intervals.
#' @param Shape For specifying the shape of the blob. Values can be 'ellipse' or 'square' (default).
#' @param cex.AdditionalText Scalling factor for additional text. Default value is 0.66.
#' @param NumPoints Number of point per quater section of ellipse. Default value is 50.
#' @param SuppressWarnings Switch for suppressing warning message about application of is.na to S4 object. Default is TRUE.
#'
#' @examples
#' require(SarpBulletinToolsPrivate)
#' require(SarpBulletinTools)
#'
#' Bulletins_BYK <- getBulletins(Agency="PkCan", Seasons=2015, Regions="Banff, Yoho and Kootenay")
#' AlpAvProb_BYK <- Bulletins_BYK$AvProblems$Alp
#'
#' Bulletins_NCol <- getBulletins(Agency="AvCan", Seasons=2015, Regions="North Columbia - Monashees & Selkirks")
#' AlpAvProb_NCol <- Bulletins_NCol$AvProblems$Alp
#'
#' par(mfrow=c(2,2))
#'
#' plotHzdChart(AlpAvProb_BYK[AlpAvProb_BYK$CHARACTER=="Wind Slabs",], Shape = "Square", Title="2015 B/Y/K: Alp Wind Slabs", cex.AdditionalText=0.5)
#' plotHzdChart(AlpAvProb_BYK[AlpAvProb_BYK$CHARACTER=="Deep Persistent Slabs",], Shape = "Square", Title="2015 B/Y/K: Alp Deep Pers Slabs", cex.AdditionalText=0.5)
#' plotHzdChart(AlpAvProb_NCol[AlpAvProb_NCol$CHARACTER=="Wind Slabs",], Shape = "Square", Title="2015 NCol: Alp Wind Slabs", cex.AdditionalText=0.5)
#' plotHzdChart(AlpAvProb_NCol[AlpAvProb_NCol$CHARACTER=="Deep Persistent Slabs",], Shape = "Square", Title="2015 NCol: Alp Deep Pers Slabs", cex.AdditionalText=0.5)
#'
#' plotHzdChartCount(AlpAvProb_BYK[AlpAvProb_BYK$CHARACTER=="Wind Slabs",], Shape = "Square", Percent = T, PercentInterval=10, Title="2015 B/Y/K: Alp Wind Slabs", cex.AdditionalText=0.5)
#' plotHzdChartCount(AlpAvProb_BYK[AlpAvProb_BYK$CHARACTER=="Deep Persistent Slabs",], Shape = "Square", Percent = T, PercentInterval=10, Title="2015 B/Y/K: Alp Deep Pers Slabs", cex.AdditionalText=0.5)
#' plotHzdChartCount(AlpAvProb_NCol[AlpAvProb_NCol$CHARACTER=="Wind Slabs",], Shape = "Square", Percent = T, PercentInterval=10, Title="2015 NCol: Alp Wind Slabs", cex.AdditionalText=0.5)
#' plotHzdChartCount(AlpAvProb_NCol[AlpAvProb_NCol$CHARACTER=="Deep Persistent Slabs",], Shape = "Square", Percent = T, PercentInterval=10, Title="2015 NCol: Alp Deep Pers Slabs", cex.AdditionalText=0.5)
#'
#' par(mfrow=c(1,1))
#'
#' @export

plotHzdChartCount <- function(AvProblems, Title="Hazard Chart", WithContours=T, Percent=F, PercentInterval=5, Shape='square', cex.AdditionalText=0.66, NumPoints=50, IDCol="PROBLEM_ID", SuppressWarnings=T) {

  ## Calculate raster
  RasterList <- calculateHzdChartCountRasterPoints(AvProblems=AvProblems, Shape=Shape, NumPoints=NumPoints, OutputRaster=T, Percent=Percent, IDCol=IDCol, SuppressWarnings=SuppressWarnings)
  Raster <- RasterList$Raster

#
#   Shape <- toupper(Shape)
#
#   ## Check for necessary packages
#   if (!isPackageInstalled("sp")) {stop("This function requires package 'sp' to be installed!")}
#   if (!isPackageInstalled("raster")) {stop("This function requires package 'raster' to be installed!")}
#
#   ## Convert avalanche problems to SPDF
#   AvProbSPDF <- convertAvProbToSPDF(AvProblems=AvProblems, Shape=Shape, NumPoints=NumPoints)
#
#   if (SuppressWarnings) {options(warn=-1)}
#
#   ## Create raster
#   if (Shape=="SQUARE") {
#     Raster <- raster::raster(ncol=8, nrow=8, xmn=1, xmx=9, ymn=1, ymx=9)
#     RasterCoord <- raster::xyFromCell(Raster, cell=c(1:64))
#   } else if (Shape=="ELLIPSE") {
#     Raster <- raster::raster(ncol=40, nrow=40, xmn=0, xmx=8, ymn=0, ymx=8)
#     RasterCoord <- raster::xyFromCell(Raster, cell=c(1:1600))
#   }
#
#   RasterPoints <- data.frame(x=RasterCoord[,1],
#                              y=RasterCoord[,2],
#                              count=0,
#                              percent=0)
#
#   ## Counting of blobs in cell
#   if(!is.na(AvProbSPDF)) {
#
#     for (Index_AvProb in 1:nrow(AvProbSPDF)) {
#       for (Index_Points in 1:nrow(RasterPoints)) {
#         RasterPoints$count[Index_Points] <- ifelse(sp::point.in.polygon(RasterPoints$x[Index_Points],
#                                                                         RasterPoints$y[Index_Points],
#                                                                         AvProbSPDF@polygons[[Index_AvProb]]@Polygons[[1]]@coords[,1],
#                                                                         AvProbSPDF@polygons[[Index_AvProb]]@Polygons[[1]]@coords[,2])==1,
#                                                    RasterPoints$count[Index_Points]+1, RasterPoints$count[Index_Points])
#       }
#     }
#
#     RasterPoints$percent <- RasterPoints$count/nrow(AvProbSPDF)*100
#
#     if (Percent==F) {
#       raster::values(Raster) <- RasterPoints$count
#     } else {
#       RasterPoints$percent <- RasterPoints$count/nrow(AvProbSPDF)*100
#       raster::values(Raster) <- RasterPoints$percent
#     }
#
#   }
#
#   TestRaster <<- Raster
#   TestRasterPoints <<- RasterPoints

  ## Chart with data
  if(!is.null(RasterList)) {

    if(Percent) {
      zLabels <- seq(0, 100, by=20)
      sp::plot(Raster, xlim=c(1,9), ylim=c(1,9), breaks=seq(0, 100, by=1), col=rev(heat.colors(100)), yaxt="n", ylab="Likelihood of avalanche(s)", xaxt="n", xlab="Destructive size", axis.args=list(at=zLabels, labels=zLabels))
    } else {
      sp::plot(Raster, xlim=c(1,9), ylim=c(1,9), yaxt="n", ylab="Likelihood of avalanche(s)", xaxt="n", xlab="Destructive size")
    }
    title(Title)
    abline(h = c(1:9), lty=2, lwd=0.5, col="dark grey")
    abline(v = c(1:9), lty=2, lwd=0.5, col="dark grey")
    box()
    axis(1, at=c(1,3,5,7,9), labels=c("Sz 1", "Sz 2", "Sz 3", "Sz 4", "Sz 5"))
    axis(2, at=c(1,3,5,7,9), labels=c("Unl", "Poss", "Lik", "VLik", "ACer"))

    mtext(paste("Total number of avalanche problems:", RasterList$NumAvProb), side=3, adj=0, cex=cex.AdditionalText)
    if(Percent) {
      mtext(("Percent"), side=3, adj=1, cex=cex.AdditionalText)
    } else {
      mtext(("Counts"), side=3, adj=1, cex=cex.AdditionalText)
    }

    ## Contour
    if(WithContours) {
      if(Percent) {
        raster::contour(Raster, add=T, zlim=c(0,100), levels=seq(0, 100, by=PercentInterval))
      } else {
        raster::contour(Raster, add=T)
      }
    }

  ## Empty chart without data
  } else {

    if(Percent) {
      zLabels <- seq(0, 100, by=20)
      sp::plot(NULL, xlim=c(1,9), ylim=c(1,9), breaks=seq(0, 100, by=1), col=rev(heat.colors(100)), yaxt="n", ylab="Likelihood of triggering", xaxt="n", xlab="Destructive size", axis.args=list(at=zLabels, labels=zLabels))
    } else {
      sp::plot(NULL, xlim=c(1,9), ylim=c(1,9), yaxt="n", ylab="Likelihood of avalanche(s)", xaxt="n", xlab="Destructive size")
    }
    title(Title)
    abline(h = c(1:9), lty=2, lwd=0.5, col="dark grey")
    abline(v = c(1:9), lty=2, lwd=0.5, col="dark grey")
    box()
    axis(1, at=c(1,3,5,7,9), labels=c("Sz 1", "Sz 2", "Sz 3", "Sz 4", "Sz 5"))
    axis(2, at=c(1,3,5,7,9), labels=c("Unl", "Poss", "Lik", "VLik", "ACer"))

    mtext(paste("Total number of avalanche problems:", RasterList$NumAvProb), side=3, adj=0, cex=cex.AdditionalText)
    if(Percent) {
      mtext(("Percent"), side=3, adj=1, cex=cex.AdditionalText)
    } else {
      mtext(("Counts"), side=3, adj=1, cex=cex.AdditionalText)
    }

    text(5,5,"No avalanche problems specified!", adj=0.5)

  }

  if (SuppressWarnings) {options(warn=0)}

}
