#' Plot raster in hazard chart with counts or percentages of avalanche problems.
#'
#' Plot raster in hazard chart with counts or percentages of avalanche problems.
#' @param AvProblems1 DF with first set of avalanche problems
#' @param AvProblems2 DF with second set of avalanche problems
#' @param Title Title of chart. Default value is 'Hazard Chart'.
#' @param WithContours Switch whether chart should be plotted with contour lines. Default is TRUE.
#' @param Percent Switch whether plotted values should be percentages instead of counts. Default is T.
#' @param PercentInterval If percentage values are plotted with contour lines, this parameter specifies the contour intervals.
#' @param Shape For specifying the shape of the blob. Values can be 'ellipse' (default) or 'square'.
#' @param cex.AdditionalText Scalling factor for additional text. Default value is 0.66.
#' @param NumPoints Number of point per quater section of ellipse. Default value is 50.
#' @param SuppressWarnings Switch for suppressing warning message about application of is.na to S4 object. Default is TRUE.
#'
#' @examples
#' require(SarpBulletinToolsPrivate)
#' require(SarpBulletinTools)
#'
#' Bulletins_BYK <- getBulletins(Agency="PkCan", Seasons=2015, Regions="Banff, Yoho and Kootenay")
#' AlpAvProb_BYK <- Bulletins_BYK$AvProblems$Alp
#'
#' plotHzdChartCount(AlpAvProb_BYK[AlpAvProb_BYK$CHARACTER=="Wind Slabs",], Shape = "Square", Percent = T, PercentInterval=10, Title="2015 B/Y/K: Alp Wind Slabs", cex.AdditionalText=0.5)
#' plotHzdChartCount(AlpAvProb_BYK[AlpAvProb_BYK$CHARACTER=="Deep Persistent Slabs",], Shape = "Square", Percent = T, PercentInterval=10, Title="2015 B/Y/K: Alp Deep Pers Slabs", cex.AdditionalText=0.5)
#'
#' plotHzdChartCountDiff(AlpAvProb_BYK[AlpAvProb_BYK$CHARACTER=="Wind Slabs",],
#'                       AlpAvProb_BYK[AlpAvProb_BYK$CHARACTER=="Deep Persistent Slabs",],
#'                       Shape = "Square", Percent = T, PercentInterval=10, Title="2015 B/Y/K: Alp Wind - DPers Slabs", cex.AdditionalText=0.5)
#'
#' @export

plotHzdChartCountDiff <- function(AvProblems1, AvProblems2, Title="Hazard Chart", WithContours=T, Percent=T, PercentInterval=5, Difference=25, Shape='square', cex.AdditionalText=0.66, NumPoints=50, SuppressWarnings=T) {

  warning("Fct should be updated to use calculateHzdChartCountRasterPoints() function!", immediate. = T)

  Shape <- toupper(Shape)

  ## Check for necessary packages
  if (!isPackageInstalled("sp")) {stop("This function requires package 'sp' to be installed!")}
  if (!isPackageInstalled("raster")) {stop("This function requires package 'raster' to be installed!")}

  ## Convert avalanche problems to SPDF
  AvProb1SPDF <- convertAvProbToSPDF(AvProblems=AvProblems1, Shape=Shape, NumPoints=NumPoints)
  AvProb2SPDF <- convertAvProbToSPDF(AvProblems=AvProblems2, Shape=Shape, NumPoints=NumPoints)

  if (SuppressWarnings) {options(warn=-1)}

  ## Create raster
  if (Shape=="SQUARE") {
    Raster1 <- raster::raster(ncol=8, nrow=8, xmn=1, xmx=9, ymn=1, ymx=9)
    Raster1Coord <- raster::xyFromCell(Raster1, cell=c(1:64))
  } else if (Shape=="ELLIPSE") {
    Raster1 <- raster::raster(ncol=40, nrow=40, xmn=0, xmx=8, ymn=0, ymx=8)
    Raster1Coord <- raster::xyFromCell(Raster1, cell=c(1:1600))
  }

  Raster2 <- Raster1
  RasterD <- Raster1

  RasterPoints1 <- data.frame(x=Raster1Coord[,1], y=Raster1Coord[,2], count=0, percent=0)
  RasterPoints2 <- RasterPoints1

  ## Counting of blobs in cell
  if(!is.na(AvProb1SPDF)) {

    ## First set of avalanche problems
    for (Index_AvProb in 1:nrow(AvProb1SPDF)) {
      for (Index_Points in 1:nrow(RasterPoints1)) {
        RasterPoints1$count[Index_Points] <- ifelse(sp::point.in.polygon(RasterPoints1$x[Index_Points],
                                                                         RasterPoints1$y[Index_Points],
                                                                         AvProb1SPDF@polygons[[Index_AvProb]]@Polygons[[1]]@coords[,1],
                                                                         AvProb1SPDF@polygons[[Index_AvProb]]@Polygons[[1]]@coords[,2])==1,
                                                    RasterPoints1$count[Index_Points]+1, RasterPoints1$count[Index_Points])
      }
    }

    RasterPoints1$percent <- RasterPoints1$count/nrow(AvProb1SPDF)*100

    ## Second set of avalanche problems
    for (Index_AvProb in 1:nrow(AvProb2SPDF)) {
      for (Index_Points in 1:nrow(RasterPoints2)) {
        RasterPoints2$count[Index_Points] <- ifelse(sp::point.in.polygon(RasterPoints2$x[Index_Points],
                                                                         RasterPoints2$y[Index_Points],
                                                                         AvProb2SPDF@polygons[[Index_AvProb]]@Polygons[[1]]@coords[,1],
                                                                         AvProb2SPDF@polygons[[Index_AvProb]]@Polygons[[1]]@coords[,2])==1,
                                                    RasterPoints2$count[Index_Points]+1, RasterPoints2$count[Index_Points])
      }
    }

    RasterPoints2$percent <- RasterPoints2$count/nrow(AvProb2SPDF)*100


    if (Percent==F) {
      raster::values(Raster1) <- RasterPoints1$count
      raster::values(Raster2) <- RasterPoints2$count
    } else {
      raster::values(Raster1) <- RasterPoints1$percent
      raster::values(Raster2) <- RasterPoints2$percent
      raster::values(RasterD) <- (RasterPoints1$percent - RasterPoints2$percent)
    }

  }

  PlotRaster <- RasterD

  ## Base chart without any blobs
  if(Percent) {
    zLabels <- seq(-Difference, Difference, by=Difference/4)
    sp::plot(PlotRaster, xlim=c(1,9), ylim=c(1,9), 
             breaks=seq(-Difference, Difference, by=2), 
             col=rev(cm.colors(Difference)), 
             yaxt="n", ylab="Likelihood of triggering", 
             xaxt="n", xlab="Destructive size", 
             legend.shrink= 0.75, 
             axis.args=list(at=zLabels, labels=zLabels), cex.axis = 1)
  } else {
    sp::plot(PlotRaster, xlim=c(1,9), ylim=c(1,9), 
             yaxt="n", ylab="Likelihood of triggering", 
             xaxt="n", xlab="Destructive size")
  }
  title(Title)
  abline(h = c(1:9), lty=2, lwd=0.5, col="dark grey")
  abline(v = c(1:9), lty=2, lwd=0.5, col="dark grey")
  box()
  axis(1, at=c(1,3,5,7,9), labels=c("Sz 1", "Sz 2", "Sz 3", "Sz 4", "Sz 5"))
  axis(2, at=c(1,3,5,7,9), labels=c("Unl", "Poss", "Lik", "VLik", "ACer"))

  mtext(paste("Total number of avalanche problems:", nrow(AvProb1SPDF), " / ", nrow(AvProb2SPDF)), side=3, adj=0, cex=cex.AdditionalText)
  if(Percent) {
    mtext(("Percent"), side=3, adj=1, cex=cex.AdditionalText)
  } else {
    mtext(("Counts"), side=3, adj=1, cex=cex.AdditionalText)
  }

  ## Contour
  if(WithContours) {
    if(Percent) {
      raster::contour(PlotRaster, add=T, zlim=c(-Difference,Difference), levels=seq(-Difference, Difference, by=PercentInterval))
    } else {
      raster::contour(PlotRaster, add=T)
    }
  }

  ## Error message
  if(is.na(AvProb1SPDF)) {
    text(5,5,"No avalanche problems specified!", adj=0.5)
  }

  if (SuppressWarnings) {options(warn=0)}

}
