#' Plot bulletin region map with hazard ratings
#'
#' Plot bulletin region map with hazard ratings
#' @param Agency Instead of RegionsNames, it is possible to provide the Agency name. Either AvCan or PkCan.
#' @param RegionNames Single or array of bulletin region names. Default is 'All'
#' @param Date Date of day.
#'
#' @examples
#' require(SarpBulletinTools)
#' require(SarpBulletinToolsPrivate)
#'
#' Agency <- "AvCan"
#' RegionNames <- c("South Rockies", "South Columbias", "Kootenay Boundary", "Lizard Range")
#' Date <- dates("2015-02-05", format="y-m-d")
#'
#' plotMapHzdRating(Date, Agency, RegionNames)
#' @export

plotMapHzdRating <- function(Date, Agency, RegionNames="All") {

  ## Initial check
  if(is.na(RegionNames[1]) & is.na(Agency)) { stop("values for either RegionNames or Agency must be provided!")}

  ## Get RegionNames is necessary
  if(RegionNames[1]=="All") {
    RegionNames <- getListForecastRegions(Agency)$REGION_NAME
  }

  ## Get Maps
  BulletinRegionSPDF <- getBulletinRegionGeomSPDF(RegionNames, Season=deriveAvSeasonFromDate(Date))
  BulletinRegionSPDF_AsCentroid <- getBulletinRegionGeomSPDF(RegionNames, Season=deriveAvSeasonFromDate(Date), AsCentroid = T)

  ## Get bulletin info
  Bulletin <- getBulletins(Agency="AvCan", StartDate=Date, Regions=RegionNames)

  ## Adding relevant hazard ratings to SPDF
  BulletinRegionSPDF_AsCentroid@data$BulletinID <- NA
  BulletinRegionSPDF_AsCentroid@data$HzdAlp <- NA
  BulletinRegionSPDF_AsCentroid@data$HzdTl <- NA
  BulletinRegionSPDF_AsCentroid@data$HzdBtl <- NA

  for(Index in 1:nrow(BulletinRegionSPDF_AsCentroid@data)) {

    BulletinID <- Bulletin$Bulletins$BULLETIN_ID[Bulletin$Bulletins$REGION==BulletinRegionSPDF_AsCentroid@data$name[Index]]

    if (length(BulletinID)==1) {
      BulletinRegionSPDF_AsCentroid@data$BulletinID[Index] <- BulletinID
      BulletinRegionSPDF_AsCentroid@data$HzdAlp[Index] <- as.character(Bulletin$DngRating$Alp$DAY0[Bulletin$DngRating$Alp$BULLETIN_ID==BulletinID])
      BulletinRegionSPDF_AsCentroid@data$HzdTl[Index]  <- as.character(Bulletin$DngRating$Tl$DAY0[Bulletin$DngRating$Tl$BULLETIN_ID==BulletinID])
      BulletinRegionSPDF_AsCentroid@data$HzdBtl[Index] <- as.character(Bulletin$DngRating$Btl$DAY0[Bulletin$DngRating$Btl$BULLETIN_ID==BulletinID])
      rm(BulletinID)
    }

  }

  BulletinRegionSPDF_AsCentroid@data$HzdAlpClr <- getHzdColor(BulletinRegionSPDF_AsCentroid@data$HzdAlp)
  BulletinRegionSPDF_AsCentroid@data$HzdTlClr  <- getHzdColor(BulletinRegionSPDF_AsCentroid@data$HzdTl)
  BulletinRegionSPDF_AsCentroid@data$HzdBtlClr <- getHzdColor(BulletinRegionSPDF_AsCentroid@data$HzdBtl)

  ## Plotting
  plot(BulletinRegionSPDF)
  plot(BulletinRegionSPDF_AsCentroid, pch=21, cex=5, bg=BulletinRegionSPDF_AsCentroid@data$HzdBtlClr, add=T)
  plot(BulletinRegionSPDF_AsCentroid, pch=21, cex=3.2, bg=BulletinRegionSPDF_AsCentroid@data$HzdTlClr, add=T)
  plot(BulletinRegionSPDF_AsCentroid, pch=21, cex=1.5, bg=BulletinRegionSPDF_AsCentroid@data$HzdAlpClr, add=T)

  box()
  title(paste("Danger Ratings: ", Date))

}
