#' Creates plot of Avalanche Problems across multiple seasons for two regions
#'
#' Creates plot of Avalanche Problems across multiple seasons for two regions
#' @param Data1 List created by createAvProblemStats.R
#' @param Data2 List created by createAvProblemStats.R
#' @param Season Range of Seasons to be plotted. i.e. 2012:2015
#' @param OutputFolder Folder path for output PNG files
#' @return PNG file in specified output folder
#'
#' @export

plotTwoRegionAllProblems <- function(Data1, Data2, OutputFolder)
{
	## 1. Set graphics dimensions 
	ChartWidth <- 16
  	ChartHeight <- 25.89
  	FigName <- paste0(OutputFolder, Region1, " & ",Region2, " - All_AvProblems")
  	png(paste0(FigName, ".png"), width=ChartWidth, height=ChartHeight, units="cm", pointsize=8, bg="white", res=600)
  	## 1a. Identify Region
  	Region1 <- as.character(Data1$Alp$Region[1])
  	Region2 <- as.character(Data2$Alp$Region[1])

	## 2. Set graphical parameters (bottom, left, top, right)
	def.par <- par(no.readonly = TRUE) # save default, for resetting...
	nf <- layout(matrix(c(1:9), 9, 1, byrow=T), 
                 width=c(lcm(15),lcm(15),lcm(15),lcm(15),lcm(15),lcm(15),lcm(15),lcm(15),lcm(15)), 
                 height=c(lcm(2.66),lcm(2.66),lcm(2.66),lcm(2.66),lcm(2.66),lcm(2.66),lcm(2.66),lcm(2.66),lcm(2.66)))

	MarginsAPers <- c(0   ,0.5,0   ,0.25)
	MarginsDPesS <- c(0   ,0.5,0   ,0.25)
	MarginsPersS <- c(0   ,0.5,0   ,0.25)
	MarginsStorm <- c(0   ,0.5,0   ,0.25)
	MarginsWind  <- c(0   ,0.5,0   ,0.25)
	MarginsCorn  <- c(0   ,0.5,0   ,0.25)
	MarginsWetS  <- c(0   ,0.5,0   ,0.25)
	MarginsLWet  <- c(0   ,0.5,0   ,0.25)
	MarginsLDry  <- c(0   ,0.5,0   ,0.25)

	### Create margin matrix
	Margins <- matrix(c(MarginsAPers, 
						MarginsDPesS, 
						MarginsPersS, 
						MarginsStorm,
						MarginsWind,
						MarginsCorn,
						MarginsWetS, 
						MarginsLWet,
						MarginsLDry),
					nrow = 9,
					ncol = 4,
					byrow = TRUE)

	### Create vector with names of avalanche problems
	AvProbNames <- c("All Persistent", 	#1
					 "Deep Persistent", #2
					 "Persistent Slab", #3
					 "Storm Slab", 		#4
					 "Wind Slab", 		#5
					 "Cornices", 		#6
					 "Wet Slab", 		#7
					 "Loose Wet", 		#8
					 "Loose Dry")		#9
	AvProbs 	<- c("PersPl",
					 "DPers",
					 "Pers",
					 "Storm",
					 "wind",
					 "Corn",
					 "Wet",
					 "LWet",
					 "LDry")
	### Create a vector with AvProblem color
	color1 		<- c("#575757",
					 "#9F000F",
					 "#F62217",
					 "#FDD017",
					 "#4CC417",
					 "#347235",
					 "#2B65EC",
					 "#79BAEC",
					 "#C8A2C8")
	color2 		<- c("#181818",
					 "#7a000c",
					 "#a71912",
					 "#c3a010",
					 "#3ea113",
					 "#1e471f",
					 "#214fba",
					 "#5091c3",
					 "#9a659a")
	# min and max season for x axis
		YRstart <- min(c(Data1$Alp$Season, Data2$Alp$Season))
		YRend   <- max(c(Data1$Alp$Season, Data2$Alp$Season))

	## Create a loop to plot all av problems
	for(index in 1:9)
	{
		## 3. Setup empty graphic
		par(mai=Margins[index,])

		PlotTitle <- paste0(Region1, " & ", Region2, " - Avalanche Problems ", YRstart, " - ", YRend)

		plot (Data1$Alp$Season, Data1$Alp[,(index+3)], 
			  type="n",
	          ylim=c(0,100), 
	          yaxt="n", 
	          ylab="",
	          xlab="",
	          xlim=c(YRstart, YRend),
	          axes=FALSE)

		## 4. Add axes
		if(index==9)
		{
			axis(1,at = seq(YRstart,YRend, by = 1), lwd = 0, lwd.tick = 1, col = "grey50")
			mtext("Season", side = 1, line = 2.75)
		}

		axis(2, at = seq(0, 100, by=10), labels = c("0 %", "10 %", "20 %", "30 %", "40 %", "50 %", "60 %", "70 %", "80 %", "90 %", ""), las = 1, lwd = 0, lwd.ticks = 1, col = "grey50")
		box(col = "grey50")

		## 5. Add Grid lines
		abline(h=seq(0,100, by = 20), col="grey", lty=2, lwd=0.5)
		abline(v=seq(YRstart, YRend, by=1), col="grey", lty=2, lwd=0.5)
		
		### Add line to seperate two data sets
		abline(v=max(Data1$Alp$Season + 1), col="grey35", lty=3, lwd=1.0)

		## 6. Add AvProb time series
		### Region1
		###########
		### Alpine
		lines(x = c(Data1$Alp[,"Season"], Data2$Alp$Season[1]), y = c((Data1$Alp[,(index+3)] / Data1$Alp[,3])*100, (Data2$Alp[1,(index+3)] / Data2$Alp[1,3])*100), col = color1[index], lwd = 2)
		points(x = Data1$Alp[,"Season"], y = (Data1$Alp[,(index+3)] / Data1$Alp[,3])*100, pch=16, cex=1.25, col=color1[index])

		### Treeline
		lines(x = c(Data1$Tln[,"Season"], Data2$Tln$Season[1]), y = c((Data1$Tln[,(index+3)] / Data1$Tln[,3])*100, (Data2$Tln[1,(index+3)] / Data2$Tln[1,3])*100), col = color1[index], lwd = 2, lty = 2)
		points(x = Data1$Tln[,"Season"], y = (Data1$Tln[,(index+3)] / Data1$Tln[,3]*100), pch=16, cex=1.25, col=color1[index])

		### Below Treeline
		lines(x = c(Data1$Btl[,"Season"], Data2$Btl$Season[1]), y = c((Data1$Btl[,(index+3)] / Data1$Btl[,3])*100, (Data2$Btl[1,(index+3)] / Data2$Btl[1,3])*100), col = color1[index], lwd = 2, lty = 3)
		points(x = Data1$Btl[,"Season"], y = (Data1$Btl[,(index+3)] / Data1$Btl[,3]*100), pch=16, cex=1.25, col=color1[index])

		### Region2
		###########
		### Alpine
		lines(x = Data2$Alp[,"Season"], y = (Data2$Alp[,(index+3)] / Data2$Alp[,3])*100, col = color2[index], lwd = 2)
		points(x = Data2$Alp[,"Season"], y = (Data2$Alp[,(index+3)] / Data2$Alp[,3])*100, pch=16, cex=1.25, col=color2[index])

		### Treeline
		lines(x = Data2$Tln[,"Season"], y = (Data2$Tln[,(index+3)] / Data2$Tln[,3])*100, col = color2[index], lwd = 2, lty=2)
		points(x = Data2$Tln[,"Season"], y = (Data2$Tln[,(index+3)] / Data2$Tln[,3]*100), pch=16, cex=1.25, col=color2[index])

		### Below Treeline
		lines(x = Data2$Btl[,"Season"], y = (Data2$Btl[,(index+3)] / Data2$Btl[,3])*100, col = color2[index], lwd = 2, lty=3)
		points(x = Data2$Btl[,"Season"], y = (Data2$Btl[,(index+3)] / Data2$Btl[,3]*100), pch=16, cex=1.25, col=color2[index])

		## 8. Add axis labels
		if(index == 1)
		{
			mtext(PlotTitle, side=3, line = 2)
		}
		mtext(AvProbNames[index], side = 2, line = 3.25)
		## 9. Add legend
		if(index==1)
		{
			legend("topright", 
					legend = c("Alpine", "Treeline", "Below Treeline"),
					col = "black",
					lwd = 2,
					lty = c(1,2,3),
					cex = 0.8,
					box.lty = 1)
			# add labels for data source
			text(2011.5, 80, "Hzdwebsite Database")
			text(2012.4, 80, "Bulletin Database")
		}
	}

	## Add CHart Title

	# Close png file
  	dev.off()
  	par(def.par)
}