#' Combines and formats Environment Canada weather obs files in preperation for Mock & Birkeland (2000) snow climate classification
#'
#' Combines and formats Env Canada weather obs .csv file(s) downloaded from http://climate.weather.gc.ca/index_e.html#access in preperation for Mock & Birkeland (2000) snow climate classification.
#' @param Filepath Folder containing WX Station Data i.e. "/user/you/Documents/WXstation/"
#' @param Year_First First year of data in format YYYY
#' @param Year_Last Last year of data in format YYYY
#'
#' @return Dataframe weather oberservation for combined season(s)
#'
#' @export

prepareWxDataEnvCan <- function(Filepath, Year_First, Year_Last)  {

  ## Loop to bind years of data from Environemnt Canada Weather Observation files
  ## *****************************************************************

  for (Year_Index in Year_First:Year_Last) 
  {
    FileName <- paste0(Filepath,"/eng-daily-0101", Year_Index, "-1231", Year_Index, ".csv")

    if(!file.exists(FileName)) {stop("Filepath for WXdata does not exist")}

    Tbl_Year <- read.csv(FileName, header = T, skip=25, as.is = T)

    if(exists("Year_WX_Combined")) {
      colnames(Tbl_Year) <- colnames(Year_WX_Combined) 
      Year_WX_Combined <- rbind(Year_WX_Combined, Tbl_Year)
    } else {
      Year_WX_Combined <- Tbl_Year
    }

  } # end of for loop combining years
  rm(Tbl_Year)

  ## Check input .csv for correct variables
  CombinedWxData_ColNames <- names(Year_WX_Combined)
  if (!("Date.Time" %in% CombinedWxData_ColNames)) {stop("Input file needs to have a 'Date' column!")}
  if (!(which(substr(CombinedWxData_ColNames, 1,9) == "Mean.Temp" & substr(CombinedWxData_ColNames, 1,11) == "Mean.Temp.F")) > 0) {stop("Input file needs to have a 'Mean Temp' column!")}
  if (!("Total.Rain..mm." %in% CombinedWxData_ColNames)) {stop("Input file needs to have a 'Rain' column!")}
  if (!("Total.Snow..cm." %in% CombinedWxData_ColNames)) {stop("Input file needs to have a 'Snow' column!")}
  if (!("Total.Precip..mm." %in% CombinedWxData_ColNames)) {stop("Input file needs to have a 'Precip' column!")}
  if (!("Snow.on.Grnd..cm." %in% CombinedWxData_ColNames)) {stop("Input file needs to have a 'Snow on Ground' column!")}

  ## Selct desired variables for Snow Climate Classification
  WXData_Combined <- Year_WX_Combined[c(1,10,16,18,20,22)]
  ## I would pick columns by name and not index. This is more transparent for reading the code.
  names(WXData_Combined) <- c( "Date", "AvgTemp", "Rain", "Hn24", "Precip", "Hs")

  ## Change class of Date variable to dates to be compatiable in chron()
  WXData_Combined$Date <- dates(WXData_Combined$Date, format="y-m-d")

  ## Return
  return(WXData_Combined)
}


