#' Combines and formats BC MoTI weather obs files in preperation for Mock & Birkeland (2000) snow climate classification
#'
#' Combines and formats BC MoTI weather obs .txt file(s) downloaded from https://prdoas3.pub-apps.th.gov.bc.ca/saw-paws/weatherstation in preperation for Mock & Birkeland (2000) Snow Climate Classification Prepares a data frame in the required format for Mock & Birkeland (2000) snow climate classification.
#' @param filepath Folder containing WX Station Data i.e. "/user/you/Documents/WXstation/"
#' @param Year_First First year of data in format YYYY
#' @param Year_Last Last year of data in format YYYY
#'
#' @return Dataframe with weather oberservation variables for combined season(s) formatted for Mock & Birkeland (2000) Snow Climate Classification
#'
#' @export

prepareWxDataMoTI <- function(filepath, Year_First, Year_Last)  {

  ## Loop to bind years of data from BC MoTI Weather Observation files
  ## *****************************************************************

  for (Year_Index in Year_First:Year_Last)
  {
    FileName <- paste0(filepath,"/weatherdata", Year_Index, ".txt")

    if(!file.exists(FileName)) {stop("Filepath for WXdata does not exist")}

    Tbl_Year <- read.table(FileName, header = T, sep = "\t", as.is = T, quote = "")

    Tbl_Year <- Tbl_Year[grep("ST",Tbl_Year$OBS.TYPE),]   ## Eliminate RAW data

    if(exists("Year_WX_Combined")) {
      Year_WX_Combined <- rbind(Year_WX_Combined, Tbl_Year)
    } else {
      Year_WX_Combined <- Tbl_Year
    }

  } ## End of loop combining years
  rm(Tbl_Year)


  ## Check input .txt for correct variables
  ## **************************************

  CombinedWxData_ColNames <- names(Year_WX_Combined)

  ## Variable
  if (!("DATETIME" %in% CombinedWxData_ColNames))  {stop("Input file needs to have a 'Date' column!")}
  if (!("MAX.TEMP" %in% CombinedWxData_ColNames))  {stop("Input file needs to have a 'Max Temp' column!")}
  if (!("MIN.TEMP" %in% CombinedWxData_ColNames))  {stop("Input file needs to have a 'Min Temp' column!")}
  if (!("NEW.PRCP" %in% CombinedWxData_ColNames))  {stop("Input file needs to have a 'Precip' column!")}
  if (!("NEW.SNOW" %in% CombinedWxData_ColNames))  {stop("Input file needs to have a 'Snow' column!")}
  if (!("SNOW.PACK" %in% CombinedWxData_ColNames)) {stop("Input file needs to have a 'Snow on Ground' column!")}

  ## Keep relevant data in new table
  Wx_Select <- Year_WX_Combined[c("DATETIME","MAX.TEMP","MIN.TEMP","NEW.PRCP","NEW.SNOW","SNOW.PACK")]

  ## Change Date class() to dates for chron
  ## **************************************

  Wx_Select$Date <- dates(substr(Wx_Select$DATETIME,1,10), format = "y/m/d")

  ## For loop calculates 2 ST obs variables for each day into daily avg/max/min
  ## ***************************************************************************

  DateUnique <- unique(Wx_Select$Date)
  TempAvg    <- numeric(length = length(DateUnique))
  PrcpNew    <- numeric(length = length(DateUnique))
  SnowNew    <- numeric(length = length(DateUnique))
  SnowPak    <- numeric(length = length(DateUnique))
  TempMax    <- numeric(length = length(DateUnique))
  TempMin    <- numeric(length = length(DateUnique))

  for (Index_Date in 1:length(DateUnique)){
    # Index_Date <- 2
    temp_Wx <- Wx_Select[DateUnique[Index_Date] == Wx_Select$Date,]
    TempAvg[Index_Date] <- (max(temp_Wx$MAX.TEMP) + min(temp_Wx$MIN.TEMP))/2
    TempMax[Index_Date] <- max(temp_Wx$MAX.TEMP)
    TempMin[Index_Date] <- min(temp_Wx$MIN.TEMP)
    PrcpNew[Index_Date] <- sum(temp_Wx$NEW.PRCP)
    SnowNew[Index_Date] <- sum(temp_Wx$NEW.SNOW)
    SnowPak[Index_Date] <- sum(temp_Wx$SNOW.PACK)

  } # End of for loop for Index_Date, individual date observation
  rm(Index_Date, temp_Wx)
  ## Problems with mis match data frame size above

  Wx_Data <- data.frame(Date    = DateUnique,
                        Hn24    = SnowNew,
                        Hs      = SnowPak,
                        AvgTemp = TempAvg,
                        MaxTemp = TempMax,
                        MinTemp = TempMin,
                        Precip  = PrcpNew)


  ## Precip data has negative values, assuming this is from evapoartion.
  ## Change all negative Precip values to 0
  ## *****************************************

  Wx_Data$Precip <- ifelse(Wx_Data$Precip < 0, 0, Wx_Data$Precip)


  ## Create Rain variable, if Max Temp is above 0
  ## Assuming SWE of 100 kg/m3 of snow
  ## *****************************************

  SnowScrap <- Wx_Data$Hn24
  SnowScrap[is.na(SnowScrap)] <- 0
  Wx_Data$Rain <- ifelse(Wx_Data$MaxTemp > 0 & (Wx_Data$Precip - SnowScrap) > 0, Wx_Data$Precip - SnowScrap, 0)


  ## Calculate Density  kg/m3
  ## ************************

  Wx_Data$Density <- ifelse(Wx_Data$Hn24 > 0 & Wx_Data$Rain == 0, Wx_Data$Precip / Wx_Data$Hn24 *100, NA)

  ## Select desired variables for Snow Climate Classification
  ## ********************************************************

  Wx_Data <- as.data.frame(Wx_Data)
  #Wx_Data <- deleteDFColumnsByName(Wx_Data, "MaxTemp")
  names(Wx_Data) <- c("Date", "Hn24", "Hs", "AvgTemp", "MaxTemp", "MinTemp", "Precip", "Rain","Density")

  ## Return
  return(Wx_Data)
}
