#' Summarize and plot aspect distributions of avalanche problems.
#'
#' Summarize and plot aspect distributions of avalanche problems.
#' @param AvProblems DF with avalanche problems
#' @param GroupByCharacter Switch for whether to summarize aspect information by avalanche problem type. Default value is T.
#' @param ShowPlots Switch for whether aspect summary should be plotted. Default value is T.
#' @param Title Title of chart. Default value is 'Hazard Chart'.
#' @param WithJitter Switch to add jitter to plot to ensure that all lines can be seen. Default value is F.
#'
#' @examples
#'
#' require(SarpBulletinToolsPrivate)
#' require(SarpBulletinTools)
#'
#' Operation <- "AvCan"
#' Season <- 2015
#' Region <- "Sea-to-Sky"
#'
#' ## Entire Season - alpine elevation band
#' BulletinsSeason <- getBulletins(Agency=Operation, Seasons=Season, Regions=Region)
#' AvProblems <- BulletinsSeason$AvProblems$Alp
#'
#' AspectSummary <- summarizeAvProbAspect(AvProblems, Title="Aspect Summary - Individual Av Problem Types")
#' AspectSummary <- summarizeAvProbAspect(AvProblems, Title="Aspect Summary - Individual Av Problem Types", WhithJitter=T)
#' AspectSummary <- summarizeAvProbAspect(AvProblems, GroupByCharacter=F, Title="Aspect Summary - All Av Problem Types")
#'
#' @export

summarizeAvProbAspect <- function(AvProblems, GroupByCharacter=T, ShowPlots=T, Title="Aspect Summary", WithJitter=F) {

  ## Counting grouped by character
  if (GroupByCharacter) {

    CharArray <- unique(AvProblems$CHARACTER)

    for (Index in 1:length(CharArray)) {

      AvProblemsChar <- AvProblems[AvProblems$CHARACTER==CharArray[Index],]

      OutputChar <- data.frame(CHARACTER = CharArray[Index],
                               Num=nrow(AvProblemsChar),
                               ASP_N_Count  = sum(as.numeric(AvProblemsChar$ASP_N), na.rm = T),
                               ASP_NE_Count = sum(as.numeric(AvProblemsChar$ASP_NE), na.rm = T),
                               ASP_E_Count  = sum(as.numeric(AvProblemsChar$ASP_E), na.rm = T),
                               ASP_SE_Count = sum(as.numeric(AvProblemsChar$ASP_SE), na.rm = T),
                               ASP_S_Count  = sum(as.numeric(AvProblemsChar$ASP_S), na.rm = T),
                               ASP_SW_Count = sum(as.numeric(AvProblemsChar$ASP_SW), na.rm = T),
                               ASP_W_Count  = sum(as.numeric(AvProblemsChar$ASP_W), na.rm = T),
                               ASP_NW_Count = sum(as.numeric(AvProblemsChar$ASP_NW), na.rm = T))

      if(exists("Output")) {
        Output <- rbind(Output, OutputChar)
      } else {
        Output <- OutputChar
      }

    }

  ## Counting all avalanche problems together
  } else {

    if(length(unique(AvProblems$CHARACTER))==1) {
      MedianCharacter <- AvProblems$CHARACTER[1]
    } else {
      MedianCharacter <- "Multiple"
    }

    Output <- data.frame(CHARACTER = MedianCharacter,
                         Num=nrow(AvProblems),
                         ASP_N_Count  = sum(as.numeric(AvProblems$ASP_N), na.rm = T),
                         ASP_NE_Count = sum(as.numeric(AvProblems$ASP_NE), na.rm = T),
                         ASP_E_Count  = sum(as.numeric(AvProblems$ASP_E), na.rm = T),
                         ASP_SE_Count = sum(as.numeric(AvProblems$ASP_SE), na.rm = T),
                         ASP_S_Count  = sum(as.numeric(AvProblems$ASP_S), na.rm = T),
                         ASP_SW_Count = sum(as.numeric(AvProblems$ASP_SW), na.rm = T),
                         ASP_W_Count  = sum(as.numeric(AvProblems$ASP_W), na.rm = T),
                         ASP_NW_Count = sum(as.numeric(AvProblems$ASP_NW), na.rm = T))

  }

  ## Calculate percentages
  Output$ASP_N_Perc  <- round(Output$ASP_N_Count/Output$Num*100, 2)
  Output$ASP_NE_Perc <- round(Output$ASP_NE_Count/Output$Num*100, 2)
  Output$ASP_E_Perc  <- round(Output$ASP_E_Count/Output$Num*100, 2)
  Output$ASP_SE_Perc <- round(Output$ASP_SE_Count/Output$Num*100, 2)
  Output$ASP_S_Perc  <- round(Output$ASP_S_Count/Output$Num*100, 2)
  Output$ASP_SW_Perc <- round(Output$ASP_SW_Count/Output$Num*100, 2)
  Output$ASP_W_Perc  <- round(Output$ASP_W_Count/Output$Num*100, 2)
  Output$ASP_NW_Perc <- round(Output$ASP_NW_Count/Output$Num*100, 2)

  ## Plotting
  if (ShowPlots) {

    plot(NULL, type="b", xlim=c(1,8), ylim=c(0,100), ylab="Percentage",xaxt="n", xlab="Aspect", main=Title)
    axis(1, at=c(1:8), labels=c("N", "NE", "E", "SE", "S", "SW", "W", "NW"))

    if(WithJitter) {
      for (Index in 1:nrow(Output)) {
        lines(jitter(as.numeric(Output[Index,c(11:18)])), type="b", col=getAvCharColor(Output$CHARACTER[Index]))
      }
      mtext("WITH JITTER", side=3, adj=1, cex=0.75)
    } else {
      for (Index in 1:nrow(Output)) {
        lines(as.numeric(Output[Index,c(11:18)]), type="b", col=getAvCharColor(Output$CHARACTER[Index]))
      }
    }

    legend(0.85, 5+nrow(Output)*5, legend=paste0(Output$CHARACTER, " (", Output$Num, ")"), col=getAvCharColor(Output$CHARACTER), cex=0.75, bty = "n", pch=1)

  }

  ## Output
  return(Output)

}
