#' Test statistical differences in avalanche problem distribution parameter.
#'
#' Test statistical differences in avalanche problem distribution parameter by applying the Wilcoxon rank-sum test.
#' @param AvProblems1 DF with first set of avalanche problems
#' @param AvProblems2 DF with second set of avalanche problems
#' @param AvProblems1Label Optional label for AvProblems1 in output dataframe. Default value is 'AvProblem1'.
#' @param AvProblems2Label Optional label for AvProblems2 in output dataframe. Default value is 'AvProblem2'.
#' @param SuppressWarnings Switch for suppressing warning message about application of is.na to S4 object. Default is TRUE.
#' @param ShowPlots Switch for showing plots. Default value is T.
#'
#' @examples
#' require(SarpBulletinToolsPrivate)
#' require(SarpBulletinTools)
#'
#' ## Retrieving data
#' Bulletins_Gl2012 <- getBulletins(Agency="PkCan", Seasons=2012, Regions="Glacier")
#' AvProblems1 <- Bulletins_Gl2012$AvProblems$Alp[Bulletins_Gl2012$AvProblems$Alp$CHARACTER=="Persistent Slabs",]
#' AvProblems2 <- Bulletins_Gl2012$AvProblems$Alp[Bulletins_Gl2012$AvProblems$Alp$CHARACTER=="Storm Slabs",]
#'
#' ## Testing differences in distribution parameter
#' Results <- testAvProbDistributionDiff(AvProblems1, AvProblems2, AvProblems1Label="Persistent Slabs", AvProblems2Label="Storm Slabs")
#'
#' @export

testAvProbDistributionDiff <- function(AvProblems1, AvProblems2, AvProblems1Label="AvProblems1", AvProblems2Label="AvProblems2", SuppressWarnings=T, ShowPlots=T) {

  if (nrow(AvProblems1)>0 & nrow(AvProblems2)>0 & sum(!is.na(AvProblems1$DISTRIBUTION))>0 & sum(!is.na(AvProblems2$DISTRIBUTION))>0) {

    ## Formatting column
    AvProblems1$DISTRIBUTION <- ordered(as.character(AvProblems1$DISTRIBUTION), levels=LevelsDistribution)
    AvProblems2$DISTRIBUTION <- ordered(as.character(AvProblems2$DISTRIBUTION), levels=LevelsDistribution)

    Dist1 <- as.numeric(AvProblems1$DISTRIBUTION)
    Dist2 <- as.numeric(AvProblems2$DISTRIBUTION)

    ## Output dataframe
    Output <- data.frame(AvProblems1=AvProblems1Label,
                         AvProblems2=AvProblems2Label,
                         Num1=nrow(AvProblems1),
                         Num2=nrow(AvProblems2),
                         Dist1Med=median(Dist1),
                         Dist1Avg=round(mean(Dist1),3),
                         Dist2Med=median(Dist2),
                         Dist2Avg=round(mean(Dist2),3),
                         DistTestW=NA)

    ## Testing size parameters
    if (SuppressWarnings) {options(warn=-1)}
    Output$DistTestW  <- format(round(wilcox.test(Dist1, Dist2)$p.value, 5), scientific=F)
    if (SuppressWarnings) {options(warn=0)}


  } else if (nrow(AvProblems1)>0 & sum(!is.na(AvProblems1$DISTRIBUTION))>0 & (nrow(AvProblems2)==0 | sum(!is.na(AvProblems2$DISTRIBUTION))==0)) {

    ## Warnings
    if (nrow(AvProblems2)==0) {
      warning("No avalanche problem records in AvProblem2!", immediate. = T)
    } else if (sum(!is.na(AvProblems2$DISTRIBUTION))==0) {
      warning("No avalanche distribution values in AvProblem2!", immediate. = T)
    }

    ## Formatting column
    AvProblems1$DISTRIBUTION <- ordered(as.character(AvProblems1$DISTRIBUTION), levels=LevelsDistribution)

    Dist1 <- as.numeric(AvProblems1$DISTRIBUTION)

    ## Output dataframe
    Output <- data.frame(AvProblems1=AvProblems1Label,
                         AvProblems2=AvProblems2Label,
                         Num1=nrow(AvProblems1),
                         Num2=nrow(AvProblems2),
                         Dist1Med=median(Dist1),
                         Dist2Med=NA,
                         DistTestW=NA)

  } else if ((nrow(AvProblems1)==0 | sum(!is.na(AvProblems1$DISTRIBUTION))>0) & nrow(AvProblems2)>0 & sum(!is.na(AvProblems2$DISTRIBUTION))>0) {

    ## Warnings
    if (nrow(AvProblems1)==0) {
      warning("No avalanche problem records in AvProblem2!", immediate. = T)
    } else if (sum(!is.na(AvProblems1$DISTRIBUTION))==0) {
      warning("No avalanche distribution values in AvProblem2!", immediate. = T)
    }

    ## Formatting column
    AvProblems2$DISTRIBUTION <- ordered(as.character(AvProblems2$DISTRIBUTION), levels=LevelsDistribution)

    Dist2 <- as.numeric(AvProblems2$DISTRIBUTION)

    ## Output dataframe
    Output <- data.frame(AvProblems1=AvProblems1Label,
                         AvProblems2=AvProblems2Label,
                         Num1=nrow(AvProblems1),
                         Num2=nrow(AvProblems2),
                         Dist1Med=NA,
                         Dist2Med=median(Dist2),
                         DistTestW=NA)



  } else {

    ## Warnings
    if (nrow(AvProblems1)==0) {
      warning("No avalanche problem records in AvProblem1!", immediate. = T)
    } else if (sum(!is.na(AvProblems1$DISTRIBUTION))==0) {
      warning("No avalanche distribution values in AvProblem1!", immediate. = T)
    }
    if (nrow(AvProblems2)==0) {
      warning("No avalanche problem records in AvProblem2!", immediate. = T)
    } else if (sum(!is.na(AvProblems2$DISTRIBUTION))==0) {
      warning("No avalanche distribution values in AvProblem2!", immediate. = T)
    }

    ## Output dataframe
    Output <- data.frame(AvProblems1=AvProblems1Label,
                         AvProblems2=AvProblems2Label,
                         Num1=nrow(AvProblems1),
                         Num2=nrow(AvProblems2),
                         Dist1Med=NA,
                         Dist2Med=NA,
                         DistTestW=NA)

  }

  ## Plotting
  if(ShowPlots) {
    par(mfrow=c(1,2))
    plotAvProbDistribution(AvProblems1, AvProblems1Label, Percent=T)
    plotAvProbDistribution(AvProblems2, AvProblems2Label, Percent=T)
    par(mfrow=c(1,1))
  }

  ## Output
  return(Output)

}
