#' Test statistical differences in avalanche problem ellipse parameters.
#'
#' Test statistical differences in avalanche problem ellipse parameters by applying the Wilcoxon rank-sum test.
#' @param AvProblems1 DF with first set of avalanche problems
#' @param AvProblems2 DF with second set of avalanche problems
#' @param AvProblems1Label Optional label for AvProblems1 in output dataframe. Default value is 'AvProblem1'.
#' @param AvProblems2Label Optional label for AvProblems2 in output dataframe. Default value is 'AvProblem2'.
#' @param SuppressWarnings Switch for suppressing warning message about application of is.na to S4 object. Default is TRUE.
#' @param ShowPlots Switch for showing plots. Default value is T.
#'
#' @examples
#' require(SarpBulletinToolsPrivate)
#' require(SarpBulletinTools)
#'
#' ## Retrieving data
#' Bulletins_SC2012 <- getBulletins(Agency="AvCan", Seasons=2012, Regions="South Columbia")
#' AvProblems1 <- Bulletins_SC2012$AvProblems$Alp[Bulletins_SC2012$AvProblems$Alp$CHARACTER=="Persistent Slabs",]
#' AvProblems2 <- Bulletins_SC2012$AvProblems$Alp[Bulletins_SC2012$AvProblems$Alp$CHARACTER=="Storm Slabs",]
#'
#' ## Testing differences in ellipse parameters
#' Results <- testAvProbEllipseDiff(AvProblems1, AvProblems2, AvProblems1Label="Persistent Slabs", AvProblems2Label="Storm Slabs")
#'
#' @export

testAvProbEllipseDiff <- function(AvProblems1, AvProblems2, AvProblems1Label="AvProblems1", AvProblems2Label="AvProblems2", SuppressWarnings=T, ShowPlots=T) {

  ## Function for plotting distribution of ellipse parameters
  plotEllispeParameters <- function(AvProblems, AvProblemsLabel, ParamPrefix) {

    if(!(ParamPrefix %in% c("SIZE", "LIKELIHOOD"))) {
      stop("ParamPrefix needs to be 'SIZE' OR 'LIKE'!")
    }

    MinCount <- table(AvProblems[,paste0(ParamPrefix, "_MIN")])
    TypCount <- table(AvProblems[,paste0(ParamPrefix, "_TYP")])
    MaxCount <- table(AvProblems[,paste0(ParamPrefix, "_MAX")])

    if (ParamPrefix=="SIZE") {
      plot(TypCount, type="b", xaxt="n", ylab="Count", xlab="Destructive Size", xlim=c(1,9), ylim=c(0, max(MaxCount, TypCount, MinCount)),
           main=AvProblemsLabel)
    } else {
      plot(TypCount, type="b", xaxt="n", ylab="Count", xlab="Likelihood of Triggering", xlim=c(1,9), ylim=c(0, max(MaxCount, TypCount, MinCount)),
           main=AvProblemsLabel)
    }
    lines(MinCount, type="b", col="blue")
    lines(MaxCount, type="b", col="red")
    abline(v=median(AvProblems[,paste0(ParamPrefix, "_MIN")]), lty=2, col="blue")
    abline(v=median(AvProblems[,paste0(ParamPrefix, "_TYP")]), lty=2)
    abline(v=median(AvProblems[,paste0(ParamPrefix, "_MAX")]), lty=2, col="red")
    if (ParamPrefix=="SIZE") {
      axis(1, at=c(1,3,5,7,9), labels=c("Sz 1", "Sz 2", "Sz 3", "Sz 4", "Sz 5"))
    } else {
      axis(1, at=c(1,3,5,7,9), labels=c("Unl", "Poss", "Lik", "VLik", "ACer"))
    }
    mtext("MIN", side=3, adj=0, cex=0.75, col="blue")
    mtext("TYP", side=3, adj=0.5, cex=0.75)
    mtext("MAX", side=3, adj=1, cex=0.75, col="red")

  }

  ## Calculation of axis
  Size1Axis  <- AvProblems1$SIZE_MAX - AvProblems1$SIZE_MIN
  Size1AxisL <- AvProblems1$SIZE_TYP - AvProblems1$SIZE_MIN
  Size1AxisR <- AvProblems1$SIZE_MAX - AvProblems1$SIZE_TYP
  Size2Axis  <- AvProblems2$SIZE_MAX - AvProblems2$SIZE_MIN
  Size2AxisL <- AvProblems2$SIZE_TYP - AvProblems2$SIZE_MIN
  Size2AxisR <- AvProblems2$SIZE_MAX - AvProblems2$SIZE_TYP

  Like1Axis  <- AvProblems1$LIKELIHOOD_MAX - AvProblems1$LIKELIHOOD_MIN
  Like1AxisD <- AvProblems1$LIKELIHOOD_TYP - AvProblems1$LIKELIHOOD_MIN
  Like1AxisU <- AvProblems1$LIKELIHOOD_MAX - AvProblems1$LIKELIHOOD_TYP
  Like2Axis  <- AvProblems2$LIKELIHOOD_MAX - AvProblems2$LIKELIHOOD_MIN
  Like2AxisD <- AvProblems2$LIKELIHOOD_TYP - AvProblems2$LIKELIHOOD_MIN
  Like2AxisU <- AvProblems2$LIKELIHOOD_MAX - AvProblems2$LIKELIHOOD_TYP

  ## Output dataframe
  Output <- data.frame(AvProblems1=AvProblems1Label,
                       AvProblems2=AvProblems2Label,
                       Num1=nrow(AvProblems1),
                       Num2=nrow(AvProblems2),
                       Size1MinMed=median(AvProblems1$SIZE_MIN),
                       Size1TypMed=median(AvProblems1$SIZE_TYP),
                       Size1MaxMed=median(AvProblems1$SIZE_MAX),
                       Size1AxisMed=median(Size1Axis),
                       Size1AxisLMed=median(Size1AxisL),
                       Size1AxisRMed=median(Size1AxisR),
                       Size1Q1Typ=quantile(AvProblems1$SIZE_TYP, 0.25),
                       Size1MedTyp=quantile(AvProblems1$SIZE_TYP, 0.50),
                       Size1Q3Typ=quantile(AvProblems1$SIZE_TYP, 0.75),
                       Size1MnTyp=mean(AvProblems1$SIZE_TYP),
                       Size1MnMM=mean(Size1Axis),
                       Size2MinMed=median(AvProblems2$SIZE_MIN),
                       Size2TypMed=median(AvProblems2$SIZE_TYP),
                       Size2MaxMed=median(AvProblems2$SIZE_MAX),
                       Size2AxisMed=median(Size2Axis),
                       Size2AxisLMed=median(Size2AxisL),
                       Size2AxisRMed=median(Size2AxisR),
                       Size2Q1Typ=quantile(AvProblems2$SIZE_TYP, 0.25),
                       Size2MedTyp=quantile(AvProblems2$SIZE_TYP, 0.50),
                       Size2Q3Typ=quantile(AvProblems2$SIZE_TYP, 0.75),
                       Size2MnTyp=mean(AvProblems2$SIZE_TYP),
                       Size2MnMM=mean(Size2Axis),
                       SizeMinTestW=NA,
                       SizeTypTestW=NA,
                       SizeMaxTestW=NA,
                       SizeAxisTestW=NA,
                       SizeAxisLTestW=NA,
                       SizeAxisRTestW=NA,
                       Like1MinMed=median(AvProblems1$LIKELIHOOD_MIN),
                       Like1TypMed=median(AvProblems1$LIKELIHOOD_TYP),
                       Like1MaxMed=median(AvProblems1$LIKELIHOOD_MAX),
                       Like1AxisMed=median(Like1Axis),
                       Like1AxisDMed=median(Like1AxisD),
                       Like1AxisUMed=median(Like1AxisU),
                       Like1Q1Typ=quantile(AvProblems1$LIKELIHOOD_TYP, 0.25),
                       Like1MedTyp=quantile(AvProblems1$LIKELIHOOD_TYP, 0.50),
                       Like1Q3Typ=quantile(AvProblems1$LIKELIHOOD_TYP, 0.75),
                       Like1MnTyp=mean(AvProblems1$LIKELIHOOD_TYP),
                       Like1MnMM=mean(Like1Axis),
                       Like2MinMed=median(AvProblems2$LIKELIHOOD_MIN),
                       Like2TypMed=median(AvProblems2$LIKELIHOOD_TYP),
                       Like2MaxMed=median(AvProblems2$LIKELIHOOD_MAX),
                       Like2AxisMed=median(Like2Axis),
                       Like2AxisDMed=median(Like2AxisD),
                       Like2AxisUMed=median(Like2AxisU),
                       Like2Q1Typ=quantile(AvProblems2$LIKELIHOOD_TYP, 0.25),
                       Like2MedTyp=quantile(AvProblems2$LIKELIHOOD_TYP, 0.50),
                       Like2Q3Typ=quantile(AvProblems2$LIKELIHOOD_TYP, 0.75),
                       Like2MnTyp=mean(AvProblems2$LIKELIHOOD_TYP),
                       Like2MnMM=mean(Like2Axis),
                       LikeMinTestW=NA,
                       LikeTypTestW=NA,
                       LikeMaxTestW=NA,
                       LikeAxisTestW=NA,
                       LikeAxisDTestW=NA,
                       LikeAxisUTestW=NA)

  ## Testing size parameters

  if (SuppressWarnings) {options(warn=-1)}

  Output$SizeMinTestW  <- format(round(wilcox.test(AvProblems1$SIZE_MIN, AvProblems2$SIZE_MIN)$p.value, 5), scientific=F)
  Output$SizeTypTestW  <- format(round(wilcox.test(AvProblems1$SIZE_TYP, AvProblems2$SIZE_TYP)$p.value, 5), scientific=F)
  Output$SizeMaxTestW  <- format(round(wilcox.test(AvProblems1$SIZE_MAX, AvProblems2$SIZE_MAX)$p.value, 5), scientific=F)

  Output$SizeAxisTestW   <- format(round(wilcox.test(Size1Axis, Size2Axis)$p.value, 5), scientific=F)
  Output$SizeAxisLTestW  <- format(round(wilcox.test(Size1AxisL, Size2AxisL)$p.value, 5), scientific=F)
  Output$SizeAxisRTestW  <- format(round(wilcox.test(Size1AxisR, Size2AxisR)$p.value, 5), scientific=F)

  ## Testing likelihood parameters
  Output$LikeMinTestW  <- format(round(wilcox.test(AvProblems1$LIKELIHOOD_MIN, AvProblems2$LIKELIHOOD_MIN)$p.value, 5), scientific=F)
  Output$LikeTypTestW  <- format(round(wilcox.test(AvProblems1$LIKELIHOOD_TYP, AvProblems2$LIKELIHOOD_TYP)$p.value, 5), scientific=F)
  Output$LikeMaxTestW  <- format(round(wilcox.test(AvProblems1$LIKELIHOOD_MAX, AvProblems2$LIKELIHOOD_MAX)$p.value, 5), scientific=F)

  Output$LikeAxisTestW   <- format(round(wilcox.test(Like1Axis, Like2Axis)$p.value, 5), scientific=F)
  Output$LikeAxisDTestW  <- format(round(wilcox.test(Like1AxisD, Like2AxisD)$p.value, 5), scientific=F)
  Output$LikeAxisUTestW  <- format(round(wilcox.test(Like1AxisU, Like2AxisU)$p.value, 5), scientific=F)

  if (SuppressWarnings) {options(warn=0)}

  ## Plotting
  if(ShowPlots) {
    par(mfrow=c(2,2))
    plotEllispeParameters(AvProblems1, AvProblems1Label, "SIZE")
    plotEllispeParameters(AvProblems2, AvProblems2Label, "SIZE")
    plotEllispeParameters(AvProblems1, AvProblems1Label, "LIKELIHOOD")
    plotEllispeParameters(AvProblems2, AvProblems2Label, "LIKELIHOOD")
    par(mfrow=c(1,1))
  }

  ## Output
  return(Output)

}
