#' Test statistical differences in avalanche problem sensitivity parameter.
#'
#' Test statistical differences in avalanche problem sensitivity parameter by applying the Wilcoxon rank-sum test.
#' @param AvProblems1 DF with first set of avalanche problems
#' @param AvProblems2 DF with second set of avalanche problems
#' @param AvProblems1Label Optional label for AvProblems1 in output dataframe. Default value is 'AvProblem1'.
#' @param AvProblems2Label Optional label for AvProblems2 in output dataframe. Default value is 'AvProblem2'.
#' @param SuppressWarnings Switch for suppressing warning message about application of is.na to S4 object. Default is TRUE.
#' @param ShowPlots Switch for showing plots. Default value is T.
#'
#' @examples
#' require(SarpBulletinToolsPrivate)
#' require(SarpBulletinTools)
#'
#' ## Retrieving data
#' Bulletins_Gl2012 <- getBulletins(Agency="PkCan", Seasons=2012, Regions="Glacier")
#' AvProblems1 <- Bulletins_Gl2012$AvProblems$Alp[Bulletins_Gl2012$AvProblems$Alp$CHARACTER=="Persistent Slabs",]
#' AvProblems2 <- Bulletins_Gl2012$AvProblems$Alp[Bulletins_Gl2012$AvProblems$Alp$CHARACTER=="Storm Slabs",]
#'
#' ## Testing differences in sensitivity parameter
#' Results <- testAvProbSensitivityDiff(AvProblems1, AvProblems2, AvProblems1Label="Persistent Slabs", AvProblems2Label="Storm Slabs")
#'
#' @export

testAvProbSensitivityDiff <- function(AvProblems1, AvProblems2, AvProblems1Label="AvProblems1", AvProblems2Label="AvProblems2", SuppressWarnings=T, ShowPlots=T) {

  if (nrow(AvProblems1)>0 & nrow(AvProblems2)>0 & sum(!is.na(AvProblems1$SENSITIVITY))>0 & sum(!is.na(AvProblems2$SENSITIVITY))>0) {

    ## Formatting column
    AvProblems1$SENSITIVITY <- ordered(as.character(AvProblems1$SENSITIVITY), levels=LevelsSensitivity)
    AvProblems2$SENSITIVITY <- ordered(as.character(AvProblems2$SENSITIVITY), levels=LevelsSensitivity)

    Sens1 <- as.numeric(AvProblems1$SENSITIVITY)
    Sens2 <- as.numeric(AvProblems2$SENSITIVITY)

    ## Output dataframe
    Output <- data.frame(AvProblems1=AvProblems1Label,
                         AvProblems2=AvProblems2Label,
                         Num1=nrow(AvProblems1),
                         Num2=nrow(AvProblems2),
                         Sens1Med=median(Sens1),
                         Sens1Avg=round(mean(Sens1),3),
                         Sens2Med=median(Sens2),
                         Sens2Avg=round(mean(Sens2),3),
                         SensTestW=NA)

    ## Testing size parameters
    if (SuppressWarnings) {options(warn=-1)}
    Output$SensTestW  <- format(round(wilcox.test(Sens1, Sens2)$p.value, 5), scientific=F)
    if (SuppressWarnings) {options(warn=0)}


  } else if (nrow(AvProblems1)>0 & sum(!is.na(AvProblems1$SENSITIVITY))>0 & (nrow(AvProblems2)==0 | sum(!is.na(AvProblems2$SENSITIVITY))==0)) {

    ## Warnings
    if (nrow(AvProblems2)==0) {
      warning("No avalanche problem records in AvProblem2!", immediate. = T)
    } else if (sum(!is.na(AvProblems2$SENSITIVITY))==0) {
      warning("No avalanche sensitivity values in AvProblem2!", immediate. = T)
    }

    ## Formatting column
    AvProblems1$SENSITIVITY <- ordered(as.character(AvProblems1$SENSITIVITY), levels=LevelsSensitivity)

    Sens1 <- as.numeric(AvProblems1$SENSITIVITY)

    ## Output dataframe
    Output <- data.frame(AvProblems1=AvProblems1Label,
                         AvProblems2=AvProblems2Label,
                         Num1=nrow(AvProblems1),
                         Num2=nrow(AvProblems2),
                         Sens1Med=median(Sens1),
                         Sens2Med=NA,
                         SensTestW=NA)


  } else if ((nrow(AvProblems1)==0 | sum(!is.na(AvProblems1$SENSITIVITY))>0) & nrow(AvProblems2)>0 & sum(!is.na(AvProblems2$SENSITIVITY))>0) {

    ## Warnings
    if (nrow(AvProblems1)==0) {
      warning("No avalanche problem records in AvProblem2!", immediate. = T)
    } else if (sum(!is.na(AvProblems1$SENSITIVITY))==0) {
      warning("No avalanche sensitivity values in AvProblem2!", immediate. = T)
    }

    ## Formatting column
    AvProblems2$SENSITIVITY <- ordered(as.character(AvProblems2$SENSITIVITY), levels=LevelsSensitivity)

    Sens2 <- as.numeric(AvProblems2$SENSITIVITY)

    ## Output dataframe
    Output <- data.frame(AvProblems1=AvProblems1Label,
                         AvProblems2=AvProblems2Label,
                         Num1=nrow(AvProblems1),
                         Num2=nrow(AvProblems2),
                         Sens1Med=NA,
                         Sens2Med=median(Sens2),
                         SensTestW=NA)



  } else {

    ## Warnings
    if (nrow(AvProblems1)==0) {
      warning("No avalanche problem records in AvProblem1!", immediate. = T)
    } else if (sum(!is.na(AvProblems1$SENSITIVITY))==0) {
      warning("No avalanche sensitivity values in AvProblem1!", immediate. = T)
    }
    if (nrow(AvProblems2)==0) {
      warning("No avalanche problem records in AvProblem2!", immediate. = T)
    } else if (sum(!is.na(AvProblems2$SENSITIVITY))==0) {
      warning("No avalanche sensitivity values in AvProblem2!", immediate. = T)
    }

    ## Output dataframe
    Output <- data.frame(AvProblems1=AvProblems1Label,
                         AvProblems2=AvProblems2Label,
                         Num1=nrow(AvProblems1),
                         Num2=nrow(AvProblems2),
                         Sens1Med=NA,
                         Sens2Med=NA,
                         SensTestW=NA)

  }


  ## Plotting
  if(ShowPlots) {
    par(mfrow=c(1,2))
    plotAvProbSensitivity(AvProblems1, AvProblems1Label, Percent=T)
    plotAvProbSensitivity(AvProblems2, AvProblems2Label, Percent=T)
    par(mfrow=c(1,1))
  }

  ## Output
  return(Output)

}
