#' Performs testAvProbSensitivityDiff() for all combinations of values in a specified column.
#'
#' Performs testAvProbSensitivityDiff() for all combinations of values in a specified column.
#' @param AvProblemDF DF with avalanche problems
#' @param ColName Name of column in AvProblemDF that contains the values for the comparisons. Needs to be a factor.
#' @param ColValues Values for comparison. Default value is the levels of the ColName column.
#' @param SuppressWarnings Switch for suppressing warning message about application of is.na to S4 object. Default is TRUE.
#' @param ShowPlots Switch for showing plots. Default value is T.
#'
#' @examples
#' require(SarpBulletinToolsPrivate)
#' require(SarpBulletinTools)
#'
#' ## Retrieving data
#' StartDate <- dates("2014-10-01", format="y-m-d")
#' EndDate <- dates("2016-05-01", format="y-m-d")
#'
#' Bulletins_Gl <- getBulletins(Agency="PkCan", StartDate=StartDate, EndDate=EndDate, Regions="Glacier")
#' summary(Bulletins_Gl)
#'
#' ## Comparing all avalanche problem types in the alpine
#' Results_Alp_AllProblems <- testAvProbSensitivityDiffLoop(Bulletins_Gl$AvProblems$Alp, ColName="CHARACTER")
#'
#' ## Only comparing persistent and storm avalanche problem types in the alpine
#' Results_Alp_PersVsStorm <- testAvProbSensitivityDiffLoop(Bulletins_Gl$AvProblems$Alp, ColName="CHARACTER", ColValues=c("Persistent Slabs", "Storm Slabs"))
#'
#' ## Comparing persistent avalanche problems between different elevation bands
#' Pers_Alp <- Bulletins_Gl$AvProblems$Alp[Bulletins_Gl$AvProblems$Alp$CHARACTER=="Persistent Slabs",]
#' Pers_Alp$Elev <- "Alp"
#' Pers_Tl  <- Bulletins_Gl$AvProblems$Tl[Bulletins_Gl$AvProblems$Tl$CHARACTER=="Persistent Slabs",]
#' Pers_Tl$Elev <- "Tl"
#' Pers_Btl <- Bulletins_Gl$AvProblems$Btl[Bulletins_Gl$AvProblems$Btl$CHARACTER=="Persistent Slabs",]
#' Pers_Btl$Elev <- "Btl"
#'
#' Pers_Elev <- rbind(Pers_Alp, Pers_Tl, Pers_Btl)
#' Pers_Elev$Elev <- factor(Pers_Elev$Elev)
#' rm(Pers_Alp, Pers_Tl, Pers_Btl)
#'
#' Results_Pers_AllElev <- testAvProbSensitivityDiffLoop(Pers_Elev, ColName="Elev")
#'
#' @export

testAvProbSensitivityDiffLoop <- function(AvProblemsDF, ColName, ColValues=levels(AvProblemsDF[,ColName]), SuppressWarnings=T, ShowPlots=T) {

  ## Remove Results dataframe is already exists
  if(exists("Results")) {rm(Results)}

  ## Examine differences between avalanche problems
  for (Index_AvProb1 in 1:(length(ColValues)-1)) {

    AvProb1Lbl <- ColValues[Index_AvProb1]
    AvProb1 <- AvProblemsDF[AvProblemsDF[,ColName]==AvProb1Lbl,]


    for (Index_AvProb2 in (Index_AvProb1+1):length(ColValues)) {

      AvProb2Lbl <- ColValues[Index_AvProb2]
      AvProb2 <- AvProblemsDF[AvProblemsDF[,ColName]==AvProb2Lbl,]

      print(paste("Testing", AvProb1Lbl, "vs.", AvProb2Lbl, "..."))

      Result <- testAvProbSensitivityDiff(AvProb1, AvProb2,
                                      AvProblems1Label = AvProb1Lbl,
                                      AvProblems2Label = AvProb2Lbl,
                                      SuppressWarnings=SuppressWarnings,
                                      ShowPlots=ShowPlots)

      row.names(Result)[1] <- createUuid()

      if(exists("Results")) {
        Results <- rbind(Results, Result)
      }  else {
        Results <- Result
      }

      rm(Result)

    }

  }

  ## Output
  row.names(Results) <- c(1:nrow(Results))
  return(Results)

}
