#' Test statistical differences between two summary hazard charts.
#'
#' Test statistical differences between two summary hazard charts by applying the Wilcoxon rank-sum test and the Kolmogorov–Smirnov test in the x and y axis through the cell with the highest count overall.
#' @param AvProblems1 DF with first set of avalanche problems
#' @param AvProblems2 DF with second set of avalanche problems
#' @param AvProblems1Label Optional label for AvProblems1 in output dataframe. Default value is 'AvProblem1'.
#' @param AvProblems2Label Optional label for AvProblems2 in output dataframe. Default value is 'AvProblem2'.
#' @param Shape For specifying the shape of the blob. Values can be 'ellipse' (default) or 'square'.
#' @param NumPoints Number of point per quater section of ellipse. Default value is 50.
#' @param SuppressWarnings Switch for suppressing warning message about application of is.na to S4 object. Default is TRUE.
#' @param ShowPlots Switch for showing plots. Default value is T.
#'
#' @examples
#' require(SarpBulletinToolsPrivate)
#' require(SarpBulletinTools)
#'
#' ## Retrieving data
#' Bulletins_SC2012 <- getBulletins(Agency="AvCan", Seasons=2012, Regions="South Columbia")
#' AlpAvProb_SC2012 <- Bulletins_SC2012$AvProblems$Alp
#' AlpAvProb_SC2012 <- AlpAvProb_SC2012[AlpAvProb_SC2012$CHARACTER=="Persistent Slabs",]
#'
#' Bulletins_SC2014 <- getBulletins(Agency="AvCan", Seasons=2014, Regions="South Columbia")
#' AlpAvProb_SC2014 <- Bulletins_SC2014$AvProblems$Alp
#' AlpAvProb_SC2014 <- AlpAvProb_SC2014[AlpAvProb_SC2014$CHARACTER=="Persistent Slabs",]
#'
#' ## Percentage plots
#' par(mfrow=c(1,2))
#' plotHzdChartCount(AlpAvProb_SC2012, Shape = "Square", Percent = T, PercentInterval=10, Title="2012", cex.AdditionalText=0.5)
#' plotHzdChartCount(AlpAvProb_SC2014, Shape = "Square", Percent = T, PercentInterval=10, Title="2014", cex.AdditionalText=0.5)
#' par(mfrow=c(1,1))
#'
#' ## Testing differences
#' Results <- testHzdChartCountDiff(AlpAvProb_SC2012, AlpAvProb_SC2014,
#'                                  AvProblems1Label = "Alp_Pers_SC2012",
#'                                  AvProblems2Label = "Alp_Pers_SC2014",
#'                                  Shape = "Square",
#'                                  ShowPlots=T)
#'
#' @export

testHzdChartCountDiff <- function(AvProblems1, AvProblems2, AvProblems1Label="AvProblems1", AvProblems2Label="AvProblems2", Shape='square', NumPoints=50, SuppressWarnings=T, ShowPlots=T) {

  if (nrow(AvProblems1)==0 & nrow(AvProblems2)==0) {

    Output <- data.frame(AvProblems1=AvProblems1Label,
                         AvProblems2=AvProblems2Label,
                         Num1=nrow(AvProblems1),
                         Num2=nrow(AvProblems2),
                         Size1Mode=NA,
                         Size1Min=NA,
                         Size1Q1=NA,
                         Size1Med=NA,
                         Size1Q3=NA,
                         Size1Max=NA,
                         Size2Mode=NA,
                         Size2Min=NA,
                         Size2Q1=NA,
                         Size2Med=NA,
                         Size2Q3=NA,
                         Size2Max=NA,
                         SizeTestW = NA,
                         SizeTestKS = NA,
                         SizeTestKS_MedDiffRemoved = NA,
                         SizeTestFK = NA,
                         Like1Mode=NA,
                         Like1Min=NA,
                         Like1Q1=NA,
                         Like1Med=NA,
                         Like1Q3=NA,
                         Like1Max=NA,
                         Like2Mode=NA,
                         Like2Min=NA,
                         Like2Q1=NA,
                         Like2Med=NA,
                         Like2Q3=NA,
                         Like2Max=NA,
                         LikeTestW = NA,
                         LikeTestKS = NA,
                         LikeTestKS_MedDiffRemoved = NA,
                         LikeTestFK = NA)

  } else if (nrow(AvProblems1)==0) {

    ## Calculate RasterPoints
    RasterPoints2 <- calculateHzdChartCountRasterPoints(AvProblems2, Shape=Shape, NumPoints=NumPoints, SuppressWarnings=SuppressWarnings)$RasterPoints

    ## Identifiying raster points with highst overall count
    XMax2 <- RasterPoints2$x[which.max(RasterPoints2$count)]
    YMax2 <- RasterPoints2$y[which.max(RasterPoints2$count)]

    ## Extracting array for size
    Size2 <- RasterPoints2[RasterPoints2$y==YMax2, c("x", "y", "count")]
    SizeArray2 <- rep(Size2$x, Size2$count)

    ## Extracting array for likelihood
    Like2 <- RasterPoints2[RasterPoints2$x==XMax2, c("x", "y", "count")]
    LikeArray2 <- rep(Like2$y, Like2$count)

    ## Output
    Output <- data.frame(AvProblems1=AvProblems1Label,
                         AvProblems2=AvProblems2Label,
                         Num1=nrow(AvProblems1),
                         Num2=nrow(AvProblems2),
                         Size1Mode=NA,
                         Size1Min=NA,
                         Size1Q1=NA,
                         Size1Med=NA,
                         Size1Q3=NA,
                         Size1Max=NA,
                         Size2Mode=XMax2,
                         Size2Min=min(SizeArray2),
                         Size2Q1=quantile(SizeArray2, 0.25),
                         Size2Med=quantile(SizeArray2, 0.50),
                         Size2Q3=quantile(SizeArray2, 0.75),
                         Size2Max=max(SizeArray2),
                         SizeTestW = NA,
                         SizeTestKS = NA,
                         SizeTestKS_MedDiffRemoved = NA,
                         SizeTestFK = NA,
                         Like1Mode=NA,
                         Like1Min=NA,
                         Like1Q1=NA,
                         Like1Med=NA,
                         Like1Q3=NA,
                         Like1Max=NA,
                         Like2Mode=YMax2,
                         Like2Min=min(LikeArray2),
                         Like2Q1=quantile(LikeArray2, 0.25),
                         Like2Med=quantile(LikeArray2, 0.50),
                         Like2Q3=quantile(LikeArray2, 0.75),
                         Like2Max=max(LikeArray2),
                         LikeTestW = NA,
                         LikeTestKS = NA,
                         LikeTestKS_MedDiffRemoved = NA,
                         LikeTestFK = NA)


    ## Plots
    if(ShowPlots==T) {

      par(mfrow=c(1,2))

      ## Size Plot
      plot(Size2$x, Size2$count, type="b", col="red", ylim=c(0,max(Size2$count)), xaxt="n", main="Destructive Size", ylab="Count", xlab="Category")
      axis(1, at=c(1,3,5,7,9), labels=c("Sz 1", "Sz 2", "Sz 3", "Sz 4", "Sz 5"))

      abline(v=Output$Size2Q1, lty=3, col="red")
      abline(v=Output$Size2Med, lty=1, col="red")
      abline(v=Output$Size2Q3, lty=3, col="red")
      mtext(AvProblems1Label, side=3, adj=0, cex=0.75)
      mtext(AvProblems2Label, side=3, adj=1, cex=0.75, col="red")

      ## Likelihood Plot
      plot(Like2$y, Like2$count, type="b", col="red", ylim=c(0,max(Like2$count)), xaxt="n", main="Likelihood", ylab="Count", xlab="Category")
      axis(1, at=c(1,3,5,7,9), labels=c("Unl", "Poss", "Lik", "VLik", "ACer"))
      abline(v=Output$Like2Q1, lty=3, col="red")
      abline(v=Output$Like2Med, lty=1, col="red")
      abline(v=Output$Like2Q3, lty=3, col="red")
      mtext(AvProblems1Label, side=3, adj=0, cex=0.75)
      mtext(AvProblems2Label, side=3, adj=1, cex=0.75, col="red")

      par(mfrow=c(1,1))

    }


  } else if (nrow(AvProblems2)==0) {

    ## Calculate RasterPoints
    RasterPoints1 <- calculateHzdChartCountRasterPoints(AvProblems1, Shape=Shape, NumPoints=NumPoints, SuppressWarnings=SuppressWarnings)$RasterPoints

    ## Identifiying raster points with highst overall count
    XMax1 <- RasterPoints1$x[which.max(RasterPoints1$count)]
    YMax1 <- RasterPoints1$y[which.max(RasterPoints1$count)]

    ## Extracting array for size
    Size1 <- RasterPoints1[RasterPoints1$y==YMax1, c("x", "y", "count")]
    SizeArray1 <- rep(Size1$x, Size1$count)

    ## Extracting array for likelihood
    Like1 <- RasterPoints1[RasterPoints1$x==XMax1, c("x", "y", "count")]
    LikeArray1 <- rep(Like1$y, Like1$count)

    ## Output
    Output <- data.frame(AvProblems1=AvProblems1Label,
                         AvProblems2=AvProblems2Label,
                         Num1=nrow(AvProblems1),
                         Num2=nrow(AvProblems2),
                         Size1Mode=XMax1,
                         Size1Min=min(SizeArray1),
                         Size1Q1=quantile(SizeArray1, 0.25),
                         Size1Med=quantile(SizeArray1, 0.50),
                         Size1Q3=quantile(SizeArray1, 0.75),
                         Size1Max=max(SizeArray1),
                         Size2Mode=NA,
                         Size2Min=NA,
                         Size2Q1=NA,
                         Size2Med=NA,
                         Size2Q3=NA,
                         Size2Max=NA,
                         SizeTestW = NA,
                         SizeTestKS = NA,
                         SizeTestKS_MedDiffRemoved = NA,
                         SizeTestFK = NA,
                         Like1Mode=YMax1,
                         Like1Min=min(LikeArray1),
                         Like1Q1=quantile(LikeArray1, 0.25),
                         Like1Med=quantile(LikeArray1, 0.50),
                         Like1Q3=quantile(LikeArray1, 0.75),
                         Like1Max=max(LikeArray1),
                         Like2Mode=NA,
                         Like2Min=NA,
                         Like2Q1=NA,
                         Like2Med=NA,
                         Like2Q3=NA,
                         Like2Max=NA,
                         LikeTestW = NA,
                         LikeTestKS = NA,
                         LikeTestKS_MedDiffRemoved = NA,
                         LikeTestFK = NA)

    ## Plots
    if(ShowPlots==T) {

      par(mfrow=c(1,2))

      ## Size Plot
      plot(Size1$x, Size1$count, type="b", ylim=c(0,max(Size1$count)), xaxt="n", main="Destructive Size", ylab="Count", xlab="Category")
      axis(1, at=c(1,3,5,7,9), labels=c("Sz 1", "Sz 2", "Sz 3", "Sz 4", "Sz 5"))
      abline(v=Output$Size1Q1, lty=3)
      abline(v=Output$Size1Med, lty=1)
      abline(v=Output$Size1Q3, lty=3)
      mtext(AvProblems1Label, side=3, adj=0, cex=0.75)
      mtext(AvProblems2Label, side=3, adj=1, cex=0.75, col="red")

      ## Likelihood Plot
      plot(Like1$y, Like1$count, type="b", ylim=c(0,max(Like1$count)), xaxt="n", main="Likelihood", ylab="Count", xlab="Category")
      axis(1, at=c(1,3,5,7,9), labels=c("Unl", "Poss", "Lik", "VLik", "ACer"))
      abline(v=Output$Like1Q1, lty=3)
      abline(v=Output$Like1Med, lty=1)
      abline(v=Output$Like1Q3, lty=3)
      mtext(AvProblems1Label, side=3, adj=0, cex=0.75)
      mtext(AvProblems2Label, side=3, adj=1, cex=0.75, col="red")

      par(mfrow=c(1,1))

    }

  } else {

    ## Calculate RasterPoints
    RasterPoints1 <- calculateHzdChartCountRasterPoints(AvProblems1, Shape=Shape, NumPoints=NumPoints, SuppressWarnings=SuppressWarnings)$RasterPoints
    RasterPoints2 <- calculateHzdChartCountRasterPoints(AvProblems2, Shape=Shape, NumPoints=NumPoints, SuppressWarnings=SuppressWarnings)$RasterPoints

    ## Identifiying raster points with highst overall count
    XMax1 <- RasterPoints1$x[which.max(RasterPoints1$count)]
    YMax1 <- RasterPoints1$y[which.max(RasterPoints1$count)]

    XMax2 <- RasterPoints2$x[which.max(RasterPoints2$count)]
    YMax2 <- RasterPoints2$y[which.max(RasterPoints2$count)]

    ## Extracting array for size
    Size1 <- RasterPoints1[RasterPoints1$y==YMax1, c("x", "y", "count")]
    Size2 <- RasterPoints2[RasterPoints2$y==YMax2, c("x", "y", "count")]
    SizeArray1 <- rep(Size1$x, Size1$count)
    SizeArray2 <- rep(Size2$x, Size2$count)

    ## Extracting array for likelihood
    Like1 <- RasterPoints1[RasterPoints1$x==XMax1, c("x", "y", "count")]
    Like2 <- RasterPoints2[RasterPoints2$x==XMax2, c("x", "y", "count")]
    LikeArray1 <- rep(Like1$y, Like1$count)
    LikeArray2 <- rep(Like2$y, Like2$count)

    ## Ouput with summary
    Output <- data.frame(AvProblems1=AvProblems1Label,
                         AvProblems2=AvProblems2Label,
                         Num1=nrow(AvProblems1),
                         Num2=nrow(AvProblems2),
                         Size1Mode=XMax1,
                         Size1Min=min(SizeArray1),
                         Size1Q1=quantile(SizeArray1, 0.25),
                         Size1Med=quantile(SizeArray1, 0.50),
                         Size1Q3=quantile(SizeArray1, 0.75),
                         Size1Max=max(SizeArray1),
                         Size2Mode=XMax2,
                         Size2Min=min(SizeArray2),
                         Size2Q1=quantile(SizeArray2, 0.25),
                         Size2Med=quantile(SizeArray2, 0.50),
                         Size2Q3=quantile(SizeArray2, 0.75),
                         Size2Max=max(SizeArray2),
                         SizeTestW = NA,
                         SizeTestKS = NA,
                         SizeTestKS_MedDiffRemoved = NA,
                         SizeTestFK = NA,
                         Like1Mode=YMax1,
                         Like1Min=min(LikeArray1),
                         Like1Q1=quantile(LikeArray1, 0.25),
                         Like1Med=quantile(LikeArray1, 0.50),
                         Like1Q3=quantile(LikeArray1, 0.75),
                         Like1Max=max(LikeArray1),
                         Like2Mode=YMax2,
                         Like2Min=min(LikeArray2),
                         Like2Q1=quantile(LikeArray2, 0.25),
                         Like2Med=quantile(LikeArray2, 0.50),
                         Like2Q3=quantile(LikeArray2, 0.75),
                         Like2Max=max(LikeArray2),
                         LikeTestW = NA,
                         LikeTestKS = NA,
                         LikeTestKS_MedDiffRemoved = NA,
                         LikeTestFK = NA)


    ## Test for size
    Output$SizeTestW  <- format(round(wilcox.test(SizeArray1, SizeArray2)$p.value, 5), scientific=F)
    Output$SizeTestKS <- format(round(ks.test(SizeArray1, SizeArray2)$p.value, 5), scientific=F)

    SizeMedDiff <- Output$Size2Med-Output$Size2Med
    Output$SizeTestKS_MedDiffRemoved <- format(round(ks.test(SizeArray1, SizeArray2-SizeMedDiff)$p.value, 5), scientific=F)

    Output$SizeTestFK <- format(round(fligner.test(list(x = SizeArray1,
                                                        y = SizeArray2))$p.value, 5), scientific=F)  ##Fligner-Killeen test, testing homogeneity of variance 
    ## add average
    Output$Size1Ave <- format(round(mean(SizeArray1, na.rm = TRUE),3), scientific=F)
    Output$Size2Ave <- format(round(mean(SizeArray2, na.rm = TRUE),3), scientific=F)

    ## Test for like
    Output$LikeTestW  <- format(round(wilcox.test(LikeArray1, LikeArray2)$p.value, 5), scientific=F)
    Output$LikeTestKS <- format(round(ks.test(LikeArray1, LikeArray2)$p.value, 5), scientific=F)

    LikeMedDiff <- Output$Like2Med-Output$Like2Med
    Output$LikeTestKS_MedDiffRemoved <- format(round(ks.test(LikeArray1, LikeArray2-LikeMedDiff)$p.value, 5), scientific=F)

    Output$LikeTestFK <- format(round(fligner.test(list(x = LikeArray1,
                                                        y = LikeArray2))$p.value, 5), scientific=F) ##Fligner-Killeen test, testing homogeneity of variance 
    ## add average
    Output$Like1Ave <- format(round(mean(LikeArray1, na.rm = TRUE),3), scientific=F)
    Output$Like2Ave <- format(round(mean(LikeArray2, na.rm = TRUE),3), scientific=F)

    ## Plots
    if(ShowPlots==T) {

      par(mfrow=c(1,2))

      ## Size Plot
      plot(Size1$x, Size1$count, type="b", ylim=c(0,max(c(Size1$count, Size2$count))), xaxt="n", main="Destructive Size", ylab="Count", xlab="Category")
      lines(Size2$x, Size2$count, type="b", col="red")
      axis(1, at=c(1,3,5,7,9), labels=c("Sz 1", "Sz 2", "Sz 3", "Sz 4", "Sz 5"))

      abline(v=Output$Size1Q1, lty=3)
      abline(v=Output$Size1Med, lty=1)
      abline(v=Output$Size1Q3, lty=3)
      abline(v=Output$Size2Q1, lty=3, col="red")
      abline(v=Output$Size2Med, lty=1, col="red")
      abline(v=Output$Size2Q3, lty=3, col="red")
      mtext(AvProblems1Label, side=3, adj=0, cex=0.75)
      mtext(AvProblems2Label, side=3, adj=1, cex=0.75, col="red")

      ## Likelihood Plot
      plot(Like1$y, Like1$count, type="b", ylim=c(0,max(c(Like1$count, Like2$count))), xaxt="n", main="Likelihood", ylab="Count", xlab="Category")
      lines(Like2$y, Like2$count, type="b", col="red")
      axis(1, at=c(1,3,5,7,9), labels=c("Unl", "Poss", "Lik", "VLik", "ACer"))
      abline(v=Output$Like1Q1, lty=3)
      abline(v=Output$Like1Med, lty=1)
      abline(v=Output$Like1Q3, lty=3)
      abline(v=Output$Like2Q1, lty=3, col="red")
      abline(v=Output$Like2Med, lty=1, col="red")
      abline(v=Output$Like2Q3, lty=3, col="red")
      mtext(AvProblems1Label, side=3, adj=0, cex=0.75)
      mtext(AvProblems2Label, side=3, adj=1, cex=0.75, col="red")

      par(mfrow=c(1,1))

    }

  }

  ## Output
  return(Output)

}
