#' Write geometries of bulletin region to bulletin geodatabase.
#'
#' Write geometries of bulletin region to bulletin geodatabase.
#' @param SPDF Spatial dataframe with bulletin geometries.
#' @param GeomCol Name of geometry column in database table
#' @param SRID Numeric reference to coordinate system for geospatial information
#' @param Overwrite Boolean flag for whether the entire table should be overwritten (i.e., all records deleted first)
#' @param Polygon2Multipolygon Boolean flag for indicating whether a polygon layer needs to be converted into a multipolygon layer
#' @param Verbose Switch for printing of query
#' @param SuppressPostgreSQLWarnings Switch for supressing warning messages from postgresqlExecStatement. Default value is TRUE. Turn on for debugging!
#' @param ShowCounter Numeric value that specifies whether counter for queries should be shown (>0) and at what interval
#' @param ObjLabel Label for making process counter more meaningful.
#' @param MaxNumTries Number of maximum tries that are attempted. Default value is 10.
#' @export

writeBulletinRegionGeomToDB <- function(SPDF, GeomCol="geom", SRID=4617, Overwrite=F, Polygon2Multipolygon=T, Verbose=F, SuppressPostgreSQLWarnings=T, ShowCounter=1, ObjLabel="row", MaxNumTries=10) {

  ## Extract pure DF
  DF <- SPDF@data
  ColNames <- colnames(DF)

  ## Create geometry column with specified name (requires rgeos)
  DF$geom <- rgeos::writeWKT(SPDF, byid = T)
  # Does not work for multi* geometries --> creates array instead of single string

  ## Convert polygon to multipolygon if necessary
  if (Polygon2Multipolygon) {
    if (substr(DF$geom[1], 1, 7) == "POLYGON") {
      DF$geom <- paste0("MULTIPOLYGON (", substring(DF$geom, 9), ")")
    } else {
      stop("Cannot apply Polygon2Multipolygon since original geometry is not polygon!")
    }
  }

  ## Update name of geometry column
  colnames(DF) <- c(ColNames, GeomCol)

  ## Delete all records in destination table if requested
  if (Overwrite) {
    Query <- paste0("DELETE FROM bulletin.regions")

    if (Verbose) {
      print("Deleting existing records in GPS DB ...")
      cat(paste0(Query, "\n\n"))
    }

    ## Send query to DB
    sendQueryToBulletinRegionDB(Query, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings, MaxNumTries=MaxNumTries)

  }

  ## Establish DB connection
  DBCon <- tryMultipleTimes(connectToBulletinRegionDB(ReadOnly=F), MaxNumTries=MaxNumTries)

  ## Write entire DF to DB
  for (Index_Rows in 1:nrow(DF)) {

    ## User feedback
    if (ShowCounter>0) {
      if (Index_Rows==nrow(DF)) {
        print(paste0("Writing ", ObjLabel, " ", Index_Rows, " of ", nrow(DF), " to GPS DB."))
      } else if ((Index_Rows==1) | (Index_Rows %% ShowCounter == 0)) {
        print(paste0("Writing ", ObjLabel, " ", Index_Rows, " of ", nrow(DF), " to GPS DB ..."))
      }
    }

    ## Build query - start
    Query <- paste0("INSERT INTO bulletin.regions (\"")
    Query <- paste0(Query, paste(colnames(DF), collapse ="\", \""), "\") VALUES (")

    ## Build query - add values and finish query
    for (Index_Col in 1:ncol(DF)) {

      Value <- DF[Index_Rows, Index_Col]

      if (colnames(DF)[Index_Col]==GeomCol) {
        Query <- paste0(Query, "ST_GeomFromText('", Value, "', ", SRID, ")")
      } else if(class(Value)[1]=="numeric") {
        Query <- paste0(Query, convertNumericToSQL(Value))
      } else if(class(Value)[1]=="chron") {
        Query <- paste0(Query, convertDateTimeToSQL(Value))
      } else if(class(Value)[1]=="dates") {
        Query <- paste0(Query, convertDateToSQL(Value))
      } else {
        Query <- paste0(Query, convertStringToSQL(Value))
      }

      if (Index_Col<ncol(DF)) {
        Query <- paste0(Query, ", ")
      } else {
        Query <- paste0(Query, ")")
      }

    }

    ## Send query to DB
    if(Verbose) {cat(Query, "\n")}
    if (SuppressPostgreSQLWarnings) {options(warn=-1)}

    Result <- RPostgreSQL::dbSendQuery(DBCon, Query)

    if (SuppressPostgreSQLWarnings) {options(warn=0)}

  }

  ## Close DB connection
  dbDisconnect(DBCon)
  rm(DBCon)

}
