#' Write dataframe to prexisting table in PostgreSQL/PostGIS Wx database
#'
#' Writes entire content of spatial dataframe to prexisting table in PostgreSQL/PostGIS Wx database. All of the columns of the spatial dataframe need to exist in the database table.
#' @param DF Dataframe
#' @param Schema Name of database schema that contains database table
#' @param DBTbl Name of database table with geometry
#' @param Overwrite Boolean flag for whether the entire table should be overwritten (i.e., all recrods deleted first)
#' @param Verbose Switch for printing of query
#' @param SuppressPostgreSQLWarnings Switch for supressing warning messages from postgresqlExecStatement. Default value is TRUE. Turn on for debugging!
#' @param ShowCounter Numeric value that specifies whether counter for queries should be shown (>0) and at what interval
#' @param ObjLabel Label for making process counter more meaningful.
#' @param MaxNumTries Number of maximum tries that are attempted. Default value is 10.
#' @export

writeDFToWxDB <- function(DF, Schema, DBTbl, Overwrite=F, Verbose=F, SuppressPostgreSQLWarnings=T, ShowCounter=1, ObjLabel="row", MaxNumTries=10) {

  ## Delete all records in destination table is requested
  if (Overwrite) {
    if (Schema=="") {
      Query <- paste0("DELETE FROM \"", DBTbl, "\"")
    } else {
      Query <- paste0("DELETE FROM \"", Schema, "\".\"", DBTbl, "\"")
    }
    if (Verbose) {
      print("Deleting existing records in Wx DB ...")
      cat(paste0(Query, "\n\n"))
    }

    ## Send query to DB
    sendQueryToWxDB(Query, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)

  }

  ## Establish DB connection
  DBCon <- tryMultipleTimes(connectToWeatherDB(ReadOnly=F), MaxNumTries=MaxNumTries)

  ## Write entire DF to DB
  for (Index_Rows in 1:nrow(DF)) {

    ## User feedback
    if (ShowCounter>0) {
      if (Index_Rows==nrow(DF)) {
        print(paste0("Writing ", ObjLabel, " ", Index_Rows, " of ", nrow(DF), " to Wx DB."))
      } else if ((Index_Rows==1) | (Index_Rows %% ShowCounter == 0)) {
        print(paste0("Writing ", ObjLabel, " ", Index_Rows, " of ", nrow(DF), " to Wx DB ..."))
      }
    }

    ## Build query - start
    if (is.na(Schema)) {
      Query <- paste0("INSERT INTO \"", DBTbl, "\" (")
    } else {
      Query <- paste0("INSERT INTO \"", Schema, "\".\"", DBTbl, "\" (\"")
    }
    Query <- paste0(Query, paste(colnames(DF), collapse ="\", \""), "\") VALUES (")

    ## Build query - add values and finish query
    for (Index_Col in 1:ncol(DF)) {

      Value <- DF[Index_Rows, Index_Col]

      if(class(Value)[1]=="numeric") {
        Query <- paste0(Query, convertNumericToSQL(Value))
      } else if(class(Value)[1]=="chron") {
        Query <- paste0(Query, convertDateTimeToSQL(Value))
      } else if(class(Value)[1]=="dates") {
        Query <- paste0(Query, convertDateToSQL(Value))
      } else {
        Query <- paste0(Query, convertStringToSQL(Value))
      }

      if (Index_Col<ncol(DF)) {
        Query <- paste0(Query, ", ")
      } else {
        Query <- paste0(Query, ")")
      }

    }

    if(Verbose) {cat(Query, "\n")}
    if (SuppressPostgreSQLWarnings) {options(warn=-1)}

    Result <- RPostgreSQL::dbSendQuery(DBCon, Query)

    if (SuppressPostgreSQLWarnings) {options(warn=0)}

  }

  ## Close DB connection
  dbDisconnect(DBCon)
  rm(DBCon)

}
