#' Adds flying condition column to raster value dataframe.
#'
#' Adds flying condition column to raster value dataframe created with the getRasterValueDfFromRunUUID function.
#' @param Operation Name of Operation. Used for accessing PostgreSQL database.
#' @param RasterValueDf Raster value dataframe created with the getRasterValueDfFromRunUUID function. Dataframe needs to have column 'elevband'.
#' @param DBType Parameter for specifying database type. Value can be 'Main' (default), 'NodeJS' or 'old'.
#' @param ColLabel Name of flying condition column . Default is 'flying'.
#' 
#' @return Original RastervalueDf data frame with flying condition column attached
#' 
#' @seealso \code{\link{getRasterValueDfFromRunUUID}}
#' 
#' @export

addFlyingCondToRasterValueDf <- function (Operation, RasterValueDf, DBType="Main", ColLabel="flying") {

  ## Check for required type
  if(!("RasterValueDf" %in% class(RasterValueDf))) {
    stop("RasterValueDf not of class 'RasterValueDf'! Create dataframe with function 'getRasterValueDfFromRunUUID'.")
  }
  
  ## Check required column
  if (!("elevband" %in% names(RasterValueDf))) {
    stop("Input dataframe is missing required 'elevband' column! Set AddElevBand=T in getRasterValueDfFromRunUUID() to add column.")
  }

  ## Get flying condition table
  if (Operation=="MWHS" | Operation=="MWHS_old") {
    Query <- "SELECT date, alp, tl, btl FROM obs.cond_flying"
    RatingLevels <- c("good", "limited", "inaccessible")
  } else {
    stop (paste0("Operation'", Operation, "' not supported!"))
  }
  CondDF <- getRecordsFromQuery(Operation, Query, DBType=DBType)

  ## Split DF into three elevation bands
  RasterValueDf_Alp <- RasterValueDf[RasterValueDf$elevband=="ALP",]
  RasterValueDf_Tl <- RasterValueDf[RasterValueDf$elevband=="TL",]
  RasterValueDf_Btl <- RasterValueDf[RasterValueDf$elevband=="BTL",]

  ## Create elevation band specific rating DF
  CondDF_Alp <- CondDF[,c(1,2)]
  names(CondDF_Alp) <- c("date", "flying")

  CondDF_Tl <- CondDF[,c(1,3)]
  names(CondDF_Tl) <- c("date", "flying")

  CondDF_Btl <- CondDF[,c(1,4)]
  names(CondDF_Btl) <- c("date", "flying")

  ## Merge DF
  RasterValueDf_Alp <- merge(RasterValueDf_Alp, CondDF_Alp, all.x=T)
  RasterValueDf_Tl  <- merge(RasterValueDf_Tl, CondDF_Tl, all.x=T)
  RasterValueDf_Btl <- merge(RasterValueDf_Btl, CondDF_Btl, all.x=T)

  RasterValueDf <- rbind(RasterValueDf_Alp, RasterValueDf_Tl, RasterValueDf_Btl)

  ## Turn rating into ordered factor
  RasterValueDf$flying <- ordered(RasterValueDf$flying, levels=RatingLevels)
  
  ## Rename rating column
  names(RasterValueDf$rating) <- ColLabel
  
  ## Add class label
  class(RasterValueDf) <- append(class(RasterValueDf), "RasterValueDf")
  
  ## Return dataframe
  return(RasterValueDf)

}
