#' Adds PM rating column to raster value dataframe.
#'
#' Adds PM rating column to raster value dataframe created with the getRasterValueDfFromRunUUID function.
#' @param Operation Name of Operation. Used for accessing PostgreSQL database.
#' @param RasterValueDf Raster value dataframe created with the getRasterValueDfFromRunUUID function. Dataframe needs to have column 'elevband'.
#' @param DBType Parameter for specifying database type. Value can be 'Main' (default), 'NodeJS' or 'old'.
#' @param Type Type of rating. Needs to be either 'Hzd' (default) or 'Stb'. Not all operations support all ratings.
#' @param Time Time of rating. Needs to be either 'PM' (default) or 'AM'.
#' @param ColLabel label fo rating column.
#' 
#' @return Original RastervalueDf data frame with PM rating attached
#' 
#' @examples
#' require(SarpGPSTools)
#' require(SarpGPSToolsPrivate)
#' 
#' Oper <- "CMHGL"
#' 
#' ## Retrieve UUIDs of GPS runs for a time period
#' GPSRunsUUID <- getRunUUIDsFromTrackerUnitAndDate(Operation = Oper, DateStart="2018-02-01", DateEnd = "2018-02-10")
#' 
#' ## Getting raster values for these GPS runs
#' RasterValues <- getRasterValueDfFromRunUUID(Operation = Oper, RunUUIDList = GPSRunsUUID) 
#' 
#' ## Adding the PM hazard ratings to raster values
#' RasterValues <- addRatingToRasterValueDf(Operation = Oper, RasterValueDf = RasterValues)
#' 
#' @seealso \code{\link{getRasterValueDfFromRunUUID}}
#' 
#' @export

addRatingToRasterValueDf <- function (Operation, RasterValueDf, DBType="Main", Type="Hzd", Time="PM", ColLabel="rating") {

  ## Check for required type
  if(!("RasterValueDf" %in% class(RasterValueDf))) {
    stop("RasterValueDf not of class 'RasterValueDf'! Create dataframe with function 'getRasterValueDfFromRunUUID'.")
  }
  
  ## Check required column
  DFNames <- names(RasterValueDf)
  if (!("elevband" %in% names(RasterValueDf))) {
    stop("Input dataframe is missing required 'elevband' column! Set AddElevBand=T in getRasterValueDfFromRunUUID() to add column.")
  }
  
  ## Check Time parameter
  Time <- toupper(Time)
  if(Time!="PM" & Time!="AM") {
    stop("The Time parameter needs to be either 'PM' or 'AM'!")
  }
  
  ## Check Type parameter
  Type <- toupper(Type)
  if(Type!="HZD" & Type!="STB") {
    stop("The Type parameter needs to be either 'Hzd' or 'Stb'!")
  }

  ## Get ratings
  if (Operation=="MWHS" | Operation=="MWHS_old") {
    if(Time=="PM") {
      if(Type=="STB") {
        Query <- "SELECT date, alp_pm AS alp, tl_pm AS tl, btl_pm AS btl FROM obs.cond_stability"
      } else {
        stop("No hazard rating available for MWHS!")
      }  
    } else {
      if(Type=="STB") {
        Query <- "SELECT date, alp_am AS alp, tl_am AS tl, btl_am AS btl FROM obs.cond_stability"
      } else {
        stop("No hazard rating available for MWHS!")
      }
    }
    RatingLevels <- c("7", "6", "5", "4", "3", "2", "1")
  
  } else if (Operation=="NEH") {
    if(Time=="PM") {
      if(Type=="STB") {
        stop("No pm stability rating available for NEH!")
      } else {
        Query <- "SELECT date, alp, tl, btl FROM obs.cond_hazard"
      }  
    } else {
      if(Type=="STB") {
        stop("No am stability rating available for NEH!")
      } else {
        stop("No am hazard rating available for NEH!")
      }
    }
    RatingLevels <- c("L", "M", "C", "H", "X")
  
  } else if (Operation=="CMHGL") {
    if(Time=="PM") {
      if(Type=="STB") {
        Query <- "SELECT date_assess AS date, alp_stb AS alp, tl_stb AS tl, btl_stb AS btl FROM infoex.hzdassess WHERE datetime_assess::time > '12:00:00' ORDER BY date_assess"
        RatingLevels <- c("VG", "G", "F", "P", "VP")
      } else {
        Query <- "SELECT date_assess AS date, alp_hzd AS alp, tl_hzd AS tl, btl_hzd AS btl FROM infoex.hzdassess WHERE datetime_assess::time > '12:00:00' ORDER BY date_assess"
        RatingLevels <- c(1, 2, 3, 4, 5)
      }
    } else {
      if(Type=="STB") {
        Query <- "SELECT date_assess AS date, alp_stb AS alp, tl_stb AS tl, btl_stb AS btl FROM infoex.hzdassess WHERE datetime_assess::time < '12:00:00' ORDER BY date_assess"
        RatingLevels <- c("VG", "G", "F", "P", "VP")
      } else {
        Query <- "SELECT date_assess AS date, alp_hzd AS alp, tl_hzd AS tl, btl_hzd AS btl FROM infoex.hzdassess WHERE datetime_assess::time < '12:00:00' ORDER BY date_assess"
        RatingLevels <- c(1, 2, 3, 4, 5)
      }
    }
    
  } else {
    stop (paste0("Operation'", Operation, "' not supported!"))
  }
  CondDF <- getRecordsFromQuery(Operation, Query, DBType=DBType)
  
  ## Split DF into three elevation bands
  RasterValueDf_Alp <- RasterValueDf[RasterValueDf$elevband=="ALP",]
  RasterValueDf_Tl <- RasterValueDf[RasterValueDf$elevband=="TL",]
  RasterValueDf_Btl <- RasterValueDf[RasterValueDf$elevband=="BTL",]

  ## Create elevation band specific rating DF
  CondDF_Alp <- CondDF[,c(1,2)]
  names(CondDF_Alp) <- c("date", "rating")

  CondDF_Tl <- CondDF[,c(1,3)]
  names(CondDF_Tl) <- c("date", "rating")

  CondDF_Btl <- CondDF[,c(1,4)]
  names(CondDF_Btl) <- c("date", "rating")

  ## Merge DF
  RasterValueDf_Alp <- merge(RasterValueDf_Alp, CondDF_Alp, all.x=T)
  RasterValueDf_Tl  <- merge(RasterValueDf_Tl, CondDF_Tl, all.x=T)
  RasterValueDf_Btl <- merge(RasterValueDf_Btl, CondDF_Btl, all.x=T)

  RasterValueDf <- rbind(RasterValueDf_Alp, RasterValueDf_Tl, RasterValueDf_Btl)

  ## Clean up
  RasterValueDf$rating[RasterValueDf$rating==""] <- NA
  RasterValueDf$rating[RasterValueDf$rating=="n/a"] <- NA
  RasterValueDf$rating[RasterValueDf$rating=="N/A"] <- NA
  RasterValueDf$rating[RasterValueDf$rating=="NA"] <- NA
  
  ## Turn rating into ordered factor
  RasterValueDf$rating <- ordered(RasterValueDf$rating, levels=RatingLevels)
  
  ## Rename rating column
  names(RasterValueDf$rating) <- ColLabel

  ## Add class label
  class(RasterValueDf) <- append(class(RasterValueDf), "RasterValueDf")
  
  ## Return value
  return(RasterValueDf)

}
