#' Adds skiing condition column to raster value dataframe.
#'
#' Adds skiing condition column to raster value dataframe created with the getRasterValueDfFromRunUUID function.
#' @param Operation Name of Operation. Used for accessing PostgreSQL database.
#' @param RasterValueDf Raster value dataframe created with the getRasterValueDfFromRunUUID function. Dataframe needs to have column 'elevband'.
#' @param DBType Parameter for specifying database type. Value can be 'Main' (default), 'NodeJS' or 'old'.
#' @param ColLabel Name of skiing condition column . Default is 'skiing'.
#' @param AddAspectCat Switch whether to add an aspect_cat column with N, E, S, W to dataframe. Default is TRUE.
#' 
#' @return Original RastervalueDf data frame with skiing condition column attached
#' 
#' @seealso \code{\link{getRasterValueDfFromRunUUID}}
#' 
#' @export

addSkiingCondToRasterValueDf <- function (Operation, RasterValueDf, DBType="Main", ColLabel="skiing", AddAspectCat=T) {

  ## Check for required type
  if(!("RasterValueDf" %in% class(RasterValueDf))) {
    stop("RasterValueDf not of class 'RasterValueDf'! Create dataframe with function 'getRasterValueDfFromRunUUID'.")
  }
  
  ## Check required columns
  if (!("elevband" %in% names(RasterValueDf))) {
    stop("Input dataframe is missing required 'elevband' column! Set AddElevBand=T in getRasterValueDfFromRunUUID() to add column.")
  }
  
  if (!("aspect" %in% names(RasterValueDf))) {
    stop("Input dataframe is missing required 'aspect' column!")
  }

  ## Get skiing condition table
  Aspects <- c("n", "e", "s", "w")
  ElevBands <- c("alp", "tl", "btl")
  
  Query_SkiingCond <- character(0)
  for (ElevBand in ElevBands) {
    for(Aspect in Aspects) {
      Query_SkiingCond <- paste0(Query_SkiingCond , " SELECT date, '", Aspect, "' as aspect_cat, '", ElevBand, "' as elevband, ", ElevBand, "_", Aspect, " as skiing from obs.cond_skiing UNION")    
    }
  }
  Query_SkiingCond <- substr(Query_SkiingCond, 2, nchar(Query_SkiingCond)-6)
  
  CondDF <- getRecordsFromQuery(Operation, Query_SkiingCond, DBType=DBType)
  
  CondDF$elevband <- toupper(CondDF$elevband)
  CondDF$elevband <- ordered(CondDF$elevband, levels=c("BTL", "TL", "ALP"))
  
  CondDF$aspect_cat <- toupper(CondDF$aspect_cat)
  CondDF$aspect_cat <- factor(CondDF$aspect_cat)
  
  CondDF$skiing <- ordered(CondDF$skiing, levels=c("avoided", "average", "good"))
  
  rm(Query_SkiingCond, Aspects, ElevBands, Aspect, ElevBand)
  
  ## Create aspect_cat column
  RasterValueDf$aspect_cat <- convertAspectToAspectCat(RasterValueDf$aspect)

  ## Merge DF
  RasterValueDf <- merge(RasterValueDf, CondDF, by=c("date", "aspect_cat", "elevband"), all.x=T)
  
  ## Rename rating column
  names(RasterValueDf$skiing) <- ColLabel
  
  ## Delete 'aspect_cat' column if necessary
  if (!AddAspectCat) {
    RasterValueDf <- deleteDFColumnsByName(RasterValueDf, "aspect_cat")
  }
  
  ## Add class label
  class(RasterValueDf) <- append(class(RasterValueDf), "RasterValueDf")
  
  ## Return dataframe
  return(RasterValueDf)

}
