#' Lists and counts the used enumerations in the multiple choice questions for the run characterization.
#'
#' Lists and counts the used enumerations in the multiple choice questions for the run characterization. This is useful to spot typos and new enumerations since the enumerations cannot be enforced in the multiple choice questions. 
#' @param Operation Name of Operation. Used for accessing PostgreSQL database.
#' @param QuestionCode Code for question from the runchar.questions table
#' @return DF with enumerations found in relevation fields in the runchar.assessments table. 
#' 
#' @examples 
#' require(SarpGPSTools)
#' require(SarpGPSToolsPrivate)
#' 
#' Operation <- "CMHGL"
#' QuestionCode <- "ski_spec"
#' checkRunCharMCEnumerations(Operation, QuestionCode)
#' 
#' @export

checkRunCharMCEnumerations <- function(Operation, QuestionCode) {
  
  ## Get Question information
  Query <- paste0("SELECT * FROM runchar.questions where code='", QuestionCode, "'")
  QuestionInfo <- getRecordsFromQuery(Operation, Query)
  
  if (nrow(QuestionInfo)!=1) {
    
    warning(paste0("'", QuestionCode, "' is not a correct question code!"), immediate. = T)
    
    ## Output
    return(NULL)
    
  } else if (QuestionInfo$responsetype[1]!="MC") {
    
    warning(paste0("'", QuestionCode, "' is not a multiple choice question!"), immediate. = T)
    
  } else {
    
    ## Extracting line and elevation band information
    Lines <- convertArrayFromSQL(QuestionInfo$linelevels[1], Numeric=F)
    Elevs <- convertArrayFromSQL(QuestionInfo$elevlevels[1], Numeric=F)
    
    ## Extract DF
    DF <- data.frame(uuid=character(0),
                     value=character(0),
                     column=character(0))
    
    if (is.na(Lines[1]) & is.na(Elevs[1])) {
      
      ColName <- QuestionCode
      Query <- paste0("SELECT uuid, ", ColName, " as value, '", ColName, "' as column FROM runchar.assessments")
      Temp <- getRecordsFromQuery(Operation, Query)
      DF <- rbind(DF, Temp) 
      
    } else if (is.na(Lines[1])) {
      
      stop("This should not happen!")
      
    } else if (is.na(Elevs[1])) {
      
      for (IndexLines in 1:length(Lines)) {
        ColName <- paste0(QuestionCode, "_", Lines[IndexLines])
        Query <- paste0("SELECT uuid, ", ColName, " as value, '", ColName, "' as column FROM runchar.assessments")
        Temp <- getRecordsFromQuery(Operation, Query)
        DF <- rbind(DF, Temp) 
      }
      
    } else {
      
      for (IndexLines in 1:length(Lines)) {
        for (IndexElevs in 1:length(Elevs)) {
          ColName <- paste0(QuestionCode, "_", Lines[IndexLines], "_", Elevs[IndexElevs])
          Query <- paste0("SELECT uuid, ", ColName, " as value, '", ColName, "' as column FROM runchar.assessments")
          Temp <- getRecordsFromQuery(Operation, Query)
          DF <- rbind(DF, Temp) 
        }
      }
      
    } 
    
    ## Parse arrays
    DF$arrays <- sapply(DF$value, function(x) convertArrayFromSQL(Array = x, Numeric = F))
    
    ## Expand arrays for overall count
    DF2 <- data.frame(value=character(0), column=character(0))
    for (IndexRec in 1:nrow(DF)) {
      Temp <- data.frame(value=DF$arrays[IndexRec], column=DF$column[IndexRec])
      names(Temp) <- c("value", "column")
      DF2 <- rbind(DF2, Temp)
    }
    
    DF2$value <- factor(DF2$value)
    DF2$column <- factor(DF2$column)
    
    ## Count and format output dataframe
    Output <- as.data.frame(table(DF2$value))
    names(Output)[1] <- QuestionCode
    
    ## Output
    return(Output)
    
  }
  
}
