#' Converts original run characterization assessments into three analysis table.
#'
#' Converts original run characterization assessments into three analysis table, one for each aggregation level: run, line and elevation band.
#' @param Survey Survey object created with getRunCharAssessmentsSurvey() function
#' @param addUserInfo Switch whether user info from the UserInfo table should be added. Default is TRUE.
#' @return List of three dataframes. One for each aggregation level (run, line and elevation band).
#'
#' @examples
#' require(SarpGPSTools)
#' require(SarpGPSToolsPrivate)
#'
#' Survey <- getRunCharAssessmentsSurvey("NEH")
#' AnalysisTbls <- convertRunCharRaw2Long(Survey)
#'
#'
#' @export

convertRunCharRaw2Long <- function(Survey, addUserInfo = TRUE) {
  
  ## Passing tables
  Tbl <- Survey$Assess
  UserInfo <- Survey$UserInfo[,-1]
  
  ## Stacking lines
  ## **************
  # Create temporary tables (TblL1) to islate all columns from Survey$Assess
  # that operate on the LINE or ELEV level (names contain "_1", "_2", or "_3")
  TblL1 <- Tbl[ ,c(1:7, grep("_1", names(Tbl), perl = TRUE))]
  names(TblL1) <- gsub("_1", "", names(TblL1))
  TblL1$Line <- "Line 1"
  
  TblL2 <- Tbl[ ,c(1:7, grep("_2", names(Tbl), perl = TRUE))]
  names(TblL2) <- gsub("_2", "", names(TblL2))
  TblL2$Line <- "Line 2"
  
  TblL3 <- Tbl[ ,c(1:7, grep("_3", names(Tbl), perl = TRUE))]
  names(TblL3) <- gsub("_3", "", names(TblL3))
  TblL3$Line <- "Line 3"
  
  # Combine all these temporary tables into TblL (contains both LINE and ELEV variabless)
  TblL <- rbind(TblL1, TblL2, TblL3)
  TblL <- TblL[order(TblL$id), ]
  rm(TblL1, TblL2, TblL3)
  
  ## Eliminate lines that do not exist (were not asked or assessed in survey)
  TblL <- TblL[!(is.na(TblL$access_gen_feel) & is.na(TblL$ski_overall) & is.na(TblL$mitig_guideabil)), ]
  
  ## Stacking elevation bands
  ## ************************
  # From TblL, extract all columns that operate ONLY on the ELEV level (names
  # contain "_alp", "_tl", or "_btl") and store them in temporary tables (TblTl)
  TblAlp <- TblL[, c(1:7, grep("Line", names(TblL)), grep("_alp", names(TblL), perl = TRUE))]
  names(TblAlp) <- gsub("_alp", "", names(TblAlp))
  TblAlp$Elev <- "Alp"
  
  TblTl <- TblL[, c(1:7, grep("Line", names(TblL)), grep("_tl", names(TblL), perl = TRUE))]
  names(TblTl) <- gsub("_tl", "", names(TblTl))
  TblTl$Elev <- "Tl"
  
  TblBtl <- TblL[, c(1:7, grep("Line", names(TblL)), grep("_btl", names(TblL), perl = TRUE))]
  names(TblBtl) <- gsub("_btl", "", names(TblBtl))
  TblBtl$Elev <- "Btl"
  
  TblE <- gtools::smartbind(TblAlp, TblTl, TblBtl)
  TblE <- TblE[order(TblE$id, TblE$Line),]
  rm(TblAlp, TblTl, TblBtl)
  
  ## Eliminate elevation bands that do not exist
  TblE <- TblE[!(is.na(TblE$hzd_friendly) &
                   is.na(TblE$hzd_steep) &
                   is.na(TblE$hzd_terrainhzd) &
                   is.na(TblE$hzd_snowcond) &
                   is.na(TblE$hzd_otherhzd)), ]
  
  ## Isolate non-elevation band specific column from lines
  ## *****************************************************
  # Now, extract ONLY columns for variables that operate on the LINE level
  TblL <- TblL[,-grep("_alp", names(TblL), perl = TRUE)]
  TblL <- TblL[,-grep("_tl", names(TblL), perl = TRUE)]
  TblL <- TblL[,-grep("_btl", names(TblL), perl = TRUE)]
  
  ## Elimonate lines that do not exist
  TblL <- TblL[, colSums(is.na(TblL)) < nrow(TblL)]
  
  
  ## Merging elevation band table with line information
  ## **************************************************
  TblE <- merge(TblE, TblL[,-c(2:7)], by.x = c("id", "Line"), by.y = c("id", "Line"))
  
  
  ## Isolate RUN level variables from original table
  ## ************************************************
  # TblR contains only RUN level variables
  TblR <- Tbl
  TblR <- TblR[,-grep("_1", names(TblR), perl = TRUE)]
  TblR <- TblR[,-grep("_2", names(TblR), perl = TRUE)]
  TblR <- TblR[,-grep("_3", names(TblR), perl = TRUE)]
  
  TblR <- TblR[, colSums(is.na(TblR)) < nrow(TblR)]
  
  ## Merging elevation band and line table with run information
  ## ***********************************************************
  # Bring it all together now!!
  TblE <- merge(TblE, TblR[,-c(2:7)], by.x = "id", by.y = "id")
  TblL <- merge(TblL, TblR[,-c(2:7)], by.x = "id", by.y = "id")
  
  
  ## AGGREGATION from elevation to line ####
  ## *********************************
  # TblL needs to have information at the LINE level, so we must aggregate
  # elevation data for each variable down to the line level. This is done by
  # taking either the MINIMUM or the MAXIMUM observation of the three ELEV bands
  # (alp, tl, btl) depending on how the variable influences decision-making.
  # Ex: Steepness is aggregegated FUN = max because the MAXIMUM steepness is
  # the crux observation that ultimately determines a guide's assessment of the
  # LINE
  # TODO: Aggregate "comment_1-3"
  
  ## Special aggregation function
  aggElev2Line <- function(TblE, TblL, Attribute, FUN = max) {
    
    if (sum(is.na(TblE[,Attribute])) < nrow(TblE)) {
      Formula <- paste0(Attribute, " ~ id + Line")
      Temp <- aggregate(as.formula(Formula), data = TblE, FUN = FUN, na.rm = T)
      TblL <- merge(TblL, Temp, by.x = c("id", "Line"), by.y = c("id", "Line"), all.x = T)
    } else {
      TblL[, Attribute] <- NA
    }
    
    return(TblL)
    
  }
  
  ## --------------------- AGGREGATE VARIABLES -------------------------
  ## Aggregation function depends on how the factor limits Guide decisions
  
  # Skiing Experience: "ski_terrain"
  TblL <- aggElev2Line(TblE, TblL, "ski_terrain_glac")
  TblL <- aggElev2Line(TblE, TblL, "ski_terrain_nonglac")
  TblL <- aggElev2Line(TblE, TblL, "ski_terrain_extremealp")
  TblL <- aggElev2Line(TblE, TblL, "ski_terrain_planar")
  TblL <- aggElev2Line(TblE, TblL, "ski_terrain_featured")
  TblL <- aggElev2Line(TblE, TblL, "ski_terrain_avfan")
  TblL <- aggElev2Line(TblE, TblL, "ski_terrain_ridgetop")
  TblL <- aggElev2Line(TblE, TblL, "ski_terrain_moraine")
  TblL <- aggElev2Line(TblE, TblL, "ski_terrain_other")
  TblL <- aggElev2Line(TblE, TblL, "ski_terrain_likealp")
  TblL <- aggElev2Line(TblE, TblL, "ski_terrain_open")
  TblL <- aggElev2Line(TblE, TblL, "ski_terrain_glades")
  TblL <- aggElev2Line(TblE, TblL, "ski_terrain_burn")
  TblL <- aggElev2Line(TblE, TblL, "ski_terrain_rollsattl")
  TblL <- aggElev2Line(TblE, TblL, "ski_terrain_snowforest")
  TblL <- aggElev2Line(TblE, TblL, "ski_terrain_cut")
  TblL <- aggElev2Line(TblE, TblL, "ski_terrain_avpath")
  TblL <- aggElev2Line(TblE, TblL, "ski_terrain_regen")
  TblL <- aggElev2Line(TblE, TblL, "ski_terrain_dense")
  
  # Skiing Experience: "ski_spec"
  TblL <- aggElev2Line(TblE, TblL, "ski_spec_corn")
  TblL <- aggElev2Line(TblE, TblL, "ski_spec_pillows")
  TblL <- aggElev2Line(TblE, TblL, "ski_spec_fallline")
  TblL <- aggElev2Line(TblE, TblL, "ski_spec_deep")
  TblL <- aggElev2Line(TblE, TblL, "ski_spec_protect")
  TblL <- aggElev2Line(TblE, TblL, "ski_spec_other")
  
  # Hazard Potential
  TblL <- aggElev2Line(TblE, TblL, "hzd_steep")
  TblL <- aggElev2Line(TblE, TblL, "hzd_expo")
  TblL <- aggElev2Line(TblE, TblL, "hzd_friendly", FUN = min)
  
  # Hazard Potential: "hzd_terrainhzd"
  TblL <- aggElev2Line(TblE, TblL, "hzd_terrainhzd")
  TblL <- aggElev2Line(TblE, TblL, "hzd_terrainhzd_regularoh")
  TblL <- aggElev2Line(TblE, TblL, "hzd_terrainhzd_largeoh")
  TblL <- aggElev2Line(TblE, TblL, "hzd_terrainhzd_triggeroh")
  TblL <- aggElev2Line(TblE, TblL, "hzd_terrainhzd_unsupport")
  TblL <- aggElev2Line(TblE, TblL, "hzd_terrainhzd_norough")
  TblL <- aggElev2Line(TblE, TblL, "hzd_terrainhzd_freqperf")
  TblL <- aggElev2Line(TblE, TblL, "hzd_terrainhzd_highcons")
  TblL <- aggElev2Line(TblE, TblL, "hzd_terrainhzd_other")
  
  # Hazard Potential: "hzd_snowcond"
  TblL <- aggElev2Line(TblE, TblL, "hzd_snowcond")
  TblL <- aggElev2Line(TblE, TblL, "hzd_snowcond_windsl")
  TblL <- aggElev2Line(TblE, TblL, "hzd_snowcond_sh")
  TblL <- aggElev2Line(TblE, TblL, "hzd_snowcond_thin")
  TblL <- aggElev2Line(TblE, TblL, "hzd_snowcond_variable")
  TblL <- aggElev2Line(TblE, TblL, "hzd_snowcond_other")

  # Hazard Potential: "hzd_otherhzd"
  TblL <- aggElev2Line(TblE, TblL, "hzd_otherhzd")
  TblL <- aggElev2Line(TblE, TblL, "hzd_otherhzd_isolcrev")
  TblL <- aggElev2Line(TblE, TblL, "hzd_otherhzd_widecrev")
  TblL <- aggElev2Line(TblE, TblL, "hzd_otherhzd_cornice")
  TblL <- aggElev2Line(TblE, TblL, "hzd_otherhzd_other")
  TblL <- aggElev2Line(TblE, TblL, "hzd_otherhzd_treewell")
  TblL <- aggElev2Line(TblE, TblL, "hzd_otherhzd_creeks")
  TblL <- aggElev2Line(TblE, TblL, "hzd_otherhzd_rescue")
  TblL <- aggElev2Line(TblE, TblL, "hzd_otherhzd_bombs")
  
  ## Adding labels
  ## *************
  TblR$Label <- factor(TblR$loccatskirunslines_name)
  TblL$Label <- factor(paste(TblL$loccatskirunslines_name, "-", TblL$Line))
  TblE$Label <- factor(paste(TblE$loccatskirunslines_name, "-", TblE$Line, "-", TblE$Elev))
  
  ## Adding user info
  if (addUserInfo) {
    TblR <- merge(TblR, UserInfo, by.x = "user_id", by.y = "user_id")
    TblL <- merge(TblL, UserInfo, by.x = "user_id", by.y = "user_id")
    TblE <- merge(TblE, UserInfo, by.x = "user_id", by.y = "user_id")
  }
  
  ## Return
  ## ******
  return(list(TblRun = TblR,
              TblLine = TblL,
              TblElev = TblE))
  
}
