#' Counts all records in all tables in gps schema in PostgreSQL/PostGIS GPS database for multiple operations
#'
#' Counts all records in all tables in gps schema in PostgreSQL/PostGIS GPS database for multiple operations. Returns dataframe with number of records in each relevant table as well as date of first and last entry.
#' @param Operations List of operations to be included. Uses Const_operationList from SarpGPSToolsPrivate package as default.
#' @param DBType Type of database to count records. Can be 'Main' (default), 'Old' or 'NodeJS' for database that feeds online viewer
#' @param WithTotal Switch for adding totals. Default is TRUE.
#' @param Verbose Switch for printing of query. Default is FALSE.
#' @param SuppressPostgreSQLWarnings Switch for turning warnings off. Default value is TRUE
#' @return Dataframe with counting results.
#' @export

countGPSRecordsMultiple <- function(Operations=Const_OperationList, DBType="Main", WithTotal=T, Verbose=F, SuppressPostgreSQLWarnings=T) {

  ## Preparing DF_Output
  DF_Output <- data.frame(operation=Const_OperationList,
                          db_name=NA,
                          db_size=NA,
                          gps.obspoints=NA,
                          gps.runs_pausepoints=NA,
                          gps.runs=NA,
                          gps.potruns_skipped=NA,
                          gps.tracks_days=NA,
                          gps.tracks=NA,
                          tracks_day_first=NA,
                          tracks_day_last=NA,
                          stringsAsFactors = F)
  
  
  ## Retrieve list of avaiable databases
  DBCon <- connectToGPSDB(Operations[1], DBType=DBType, ConnectToDefaultPostgresDB=T)
  Result <- dbSendQuery(DBCon, "SELECT datname FROM pg_database")
  ResultDF <- dbFetch(Result,-1)
  dbClearResult(Result)
  dbDisconnect(DBCon)
  
  ListAvailDB <- ResultDF$datname
  rm(DBCon, Result, ResultDF)
  
  
  ## Looping through operations
  for (i in 1:nrow(DF_Output)){
    
    print(paste0("Processing ", DF_Output$operation[i], " ..."))
    
    if (getGPSDBName(DF_Output$operation[i], DBType=DBType) %in% ListAvailDB) {
    
      OpCounts <- countGPSRecords(DF_Output$operation[i], DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)$NumRec
      OpFirstTrackDay <- getRecordsFromQuery(DF_Output$operation[i], "SELECT MIN(date_local) AS first from gps.tracks_days", DBType=DBType)$first[1]
      OpLastTrackDay  <- getRecordsFromQuery(DF_Output$operation[i], "SELECT MAX(date_local) AS last from gps.tracks_days", DBType=DBType)$last[1]
  
      DF_Output[i,2] <- getGPSDBName(DF_Output$operation[i], DBType=DBType)
      DF_Output[i,3] <- getGPSDBSize(DF_Output$operation[i], DBType=DBType)
      DF_Output[i, c(4:9)] <- OpCounts
      if(!is.na(OpFirstTrackDay)) {DF_Output[i,10] <- as.character(OpFirstTrackDay)}
      if(!is.na(OpLastTrackDay)) {DF_Output[i,11] <- as.character(OpLastTrackDay)}
      
    } else {
      
      print(paste0("Database '", getGPSDBName(DF_Output$operation[i], DBType=DBType), "' not available."))
      
    }

  }
  
  ## Eliminate NA databases
  DF_Output <- DF_Output[!is.na(DF_Output$db_name),]
  
  ## Adding totals
  if(WithTotal) {
    DF_Output <- rbind(DF_Output,
                       c("Total",
                         NA,
                         NA,
                         sum(DF_Output$gps.obspoints, na.rm=T), 
                         sum(DF_Output$gps.runs_pausepoints, na.rm=T), 
                         sum(DF_Output$gps.runs, na.rm=T), 
                         sum(DF_Output$gps.potruns_skipped, na.rm=T), 
                         sum(DF_Output$gps.tracks_days, na.rm=T), 
                         sum(DF_Output$gps.tracks, na.rm=T), 
                         NA, NA))
  }
  
  ## Return DF
  return(DF_Output)

}