#' Counts number of runs per unit and day in PostgreSQL/PostGIS GPS database
#'
#' Counts number of runs per unit and day in PostgreSQL/PostGIS GPS database and returns sorted data frame
#' @param Operation Name of operation
#' @param DBType Parameter for specifying database type. Values can be 'Main' (default), 'NodeJS' or 'Old'.
#' @param Verbose Switch for printing of query.
#' @param SuppressPostgreSQLWarnings Switch for supressing warning messages from postgresqlExecStatement. Turn on for debugging!
#' @return Dataframe with count results.
#' @export

countRunNumbers <- function(Operation, DBType="Main", Verbose=F, SuppressPostgreSQLWarnings=T) {

  ## Get list of unit labels
  Query <- paste0("SELECT gps.tracks.unit, Count(gps.runs.uuid) AS all ",
                  "FROM gps.tracks ",
                  "INNER JOIN gps.tracks_days ON gps.tracks_days.gpstracks_uuid = gps.tracks.uuid ",
                  "INNER JOIN gps.runs ON gps.runs.gpstracksdays_uuid = gps.tracks_days.uuid ",
                  "GROUP BY gps.tracks.unit ",
                  "ORDER BY gps.tracks.unit ASC")

  ListUnits <- getRecordsFromQuery(Operation, Query, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)


  ## Get list of dates with and GPS observations (i.e., track days)

  Query <- paste0("SELECT gps.tracks_days.date_local, Count(gps.tracks_days.uuid) AS count_trackdays ",
                  "FROM gps.tracks_days ",
                  "GROUP BY gps.tracks_days.date_local ",
                  "ORDER BY gps.tracks_days.date_local ASC")

  ListDates <- getRecordsFromQuery(Operation, Query, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)


  ## Empty output dataframe
  OutputDF <- cbind(ListDates$date, as.data.frame(matrix(nrow=nrow(ListDates), ncol=nrow(ListUnits)+1)))
  colnames(OutputDF) <- c("date", "all", ListUnits$unit)

  ## Loop for filling output data frame
  for (Index_Date in 1:nrow(ListDates)) {

    Query <- paste0("SELECT Count(gps.runs.uuid) AS count ",
                    "FROM gps.runs ",
                    "INNER JOIN gps.tracks_days ON gps.runs.gpstracksdays_uuid = gps.tracks_days.uuid ",
                    "INNER JOIN gps.tracks ON gps.tracks_days.gpstracks_uuid = gps.tracks.uuid ",
                    "WHERE gps.runs.date_local = '", ListDates$date_local[Index_Date], "'")

    OutputDF[Index_Date, 2] <- getRecordsFromQuery(Operation, Query, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)$count[1]

    for (Index_Unit in 1:nrow(ListUnits)) {

      ## Checking whther track day is available
      Query <- paste0("SELECT gps.tracks_days.date_local ",
                      "FROM gps.tracks_days ",
                      "INNER JOIN gps.tracks ON gps.tracks_days.gpstracks_uuid = gps.tracks.uuid ",
                      "WHERE gps.tracks_days.date_local = '", ListDates$date_local[Index_Date], "' AND ",
                      "gps.tracks.unit = '", ListUnits$unit[Index_Unit], "'")

      ## Determine number of runs if track day is available
      if (nrow(getRecordsFromQuery(Operation, Query, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings))>0) {

        Query <- paste0("SELECT Count(gps.runs.uuid) AS count ",
                        "FROM gps.runs ",
                        "INNER JOIN gps.tracks_days ON gps.runs.gpstracksdays_uuid = gps.tracks_days.uuid ",
                        "INNER JOIN gps.tracks ON gps.tracks_days.gpstracks_uuid = gps.tracks.uuid ",
                        "WHERE gps.runs.date_local = '", ListDates$date_local[Index_Date], "' AND ",
                        "gps.tracks.unit = '", ListUnits$unit[Index_Unit], "'")

        OutputDF[Index_Date, Index_Unit+2] <- getRecordsFromQuery(Operation, Query, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)$count[1]

      }

    }

  }

  ## Calcuate and add total counts
  OutputDF$date <- as.character(OutputDF$date)
  TotalCounts <- c("total", colSums(OutputDF[,-1], na.rm=T))
  OutputDF <- rbind(OutputDF, TotalCounts)

  ## Return DF
  return(OutputDF)

}
