#' Counts records associated with a track in PostgreSQL/PostGIS GPS database
#'
#' Counts all records associated with a track in PostgreSQL/PostGIS GPS database. Returns dataframe with number of records in each relevant table.
#' @param Operation Name of operation
#' @param Track_UUID UUID of track to be investigated.
#' @param DBType Parameter for specifying database type. Values can be 'Main' (default), 'NodeJS' or 'Old'.
#' @param Verbose Switch for printing of query
#' @param SuppressPostgreSQLWarnings Switch for turning warnings off. Default value is TRUE
#' @return Dataframe with counting results.
#' @export

countTrackRecords <- function(Operation, Track_UUID, DBType="Main", Verbose=F, SuppressPostgreSQLWarnings=T) {

  DBType <- toupper(DBType)
  
  ## Create Output DF
  DF_Output <- data.frame(Table = character(0),
                         NumRec = numeric(0))

  ## 1) gps.obspoints
  if(DBType!="NODEJS") {
    Query  <- paste0("SELECT Count(gps.obspoints.uuid) AS \"Count\" FROM gps.tracks ",
                               "INNER JOIN gps.tracks_days ON gps.tracks_days.gpstracks_uuid = gps.tracks.uuid ",
                               "INNER JOIN gps.obspoints ON gps.obspoints.gpstracksdays_uuid = gps.tracks_days.uuid ",
                               "WHERE gps.tracks.uuid = '", Track_UUID, "'")
    DF_Output <- rbind(DF_Output, data.frame(Table="gps.obspoints",
                                             NumRec=getRecordsFromQuery(Operation, Query, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)$Count[1]))
  }

  ## 2) gps.runs_pausepoints
  Query <- paste0("SELECT Count(gps.runs_pausepoints.uuid) AS \"Count\" FROM gps.tracks " ,
                              "INNER JOIN gps.tracks_days ON gps.tracks_days.gpstracks_uuid = gps.tracks.uuid ",
                              "INNER JOIN gps.runs ON gps.runs.gpstracksdays_uuid = gps.tracks_days.uuid ",
                              "INNER JOIN gps.runs_pausepoints ON gps.runs_pausepoints.gpsruns_uuid = gps.runs.uuid ",
                              "WHERE gps.tracks.uuid = '", Track_UUID, "'")
  DF_Output <- rbind(DF_Output, data.frame(Table="gps.runs_pausepoints",
                                           NumRec=getRecordsFromQuery(Operation, Query, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)$Count[1]))

  ## 3) gps.runs
  Query <- paste0("SELECT Count(gps.runs.uuid) AS \"Count\" FROM gps.tracks ",
                       "INNER JOIN gps.tracks_days ON gps.tracks_days.gpstracks_uuid = gps.tracks.uuid ",
                       "INNER JOIN gps.runs ON gps.runs.gpstracksdays_uuid = gps.tracks_days.uuid ",
                       "WHERE gps.tracks.uuid = '", Track_UUID, "'")
  DF_Output <- rbind(DF_Output, data.frame(Table="gps.runs",
                                             NumRec=getRecordsFromQuery(Operation, Query, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)$Count[1]))
  
  ## 4) gps.potruns_skipped
  if(DBType!="NODEJS") {
    Query <- paste0("SELECT Count(gps.potruns_skipped.uuid) AS \"Count\" FROM gps.tracks ",
                  "INNER JOIN gps.tracks_days ON gps.tracks_days.gpstracks_uuid = gps.tracks.uuid ",
                  "INNER JOIN gps.potruns_skipped ON gps.potruns_skipped.gpstracksdays_uuid = gps.tracks_days.uuid ",
                  "WHERE gps.tracks.uuid = '", Track_UUID, "'")
    DF_Output <- rbind(DF_Output, data.frame(Table="gps.potruns_skipped",
                                           NumRec=getRecordsFromQuery(Operation, Query, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)$Count[1]))
  }
    
  ## 5) gps.tracks_days
  Query <- paste0("SELECT Count(gps.tracks_days.uuid) AS \"Count\" FROM gps.tracks ",
                             "INNER JOIN gps.tracks_days ON gps.tracks_days.gpstracks_uuid = gps.tracks.uuid ",
                             "WHERE gps.tracks.uuid = '", Track_UUID, "'")
  DF_Output <- rbind(DF_Output, data.frame(Table="gps.tracks_days",
                                           NumRec=getRecordsFromQuery(Operation, Query, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)$Count[1]))

  ## 6) gps.tracks
  Query <- paste0("SELECT Count(uuid) AS \"Count\" FROM gps.tracks WHERE uuid='", Track_UUID, "'")
  DF_Output <- rbind(DF_Output, data.frame(Table="gps.tracks",
                                           NumRec=getRecordsFromQuery(Operation, Query, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)$Count[1]))

  ## Return DF
  return(DF_Output)

}
