#' Create missing Snowbase locations in PostgreSQL/PostGIS GPS database.
#'
#' Creates missing Sbnowbase locations in PostgreSQL/PostGIS GPS database by retrieving location information from Snowbase and adding the default location specified in the PostGIS database.
#' @param Operation Name of operation
#' @param MissingLocExternalID Array of external id of missing locations
#' @param TargetDBTbl Target table in Database. Default is skiruns.
#' @param DBType Optional variable for specifying which database type the location catalog should be imported. Values can be 'Main' (default) or 'NodeJS

#' @export


createMissingSnowbaseLoc <- function(Operation, MissingLocExternalID, TargetDBTbl='skiruns', DBType="Main", Verbose=F) {

  ## Marking external ids are unique
  MissingLocExternalID <- unique(MissingLocExternalID)
  
  ## Retrieve information from Snowbase
  require(SarpCMHSnowbase)
  MissingLocSnowbase <- SarpCMHSnowbase::getSnowbaseRecordsFromUUID("Geo_Global", UUID = MissingLocExternalID, UUIDCol = "Geo_Id", SuppressOrderWarning=T)
  detach("package:SarpCMHSnowbase", unload=TRUE)
  
  ## Retrieve default location
  DefaultLocGeom <- getSpatialObjectsDataFrameFromUUID(Operation, Schema = "loccat", paste0(TargetDBTbl, "_placeholder"))
  
  for (Index in 1:nrow(MissingLocSnowbase)) {
    
    LocSPDF <- DefaultLocGeom
    
    # Updating information
    LocSPDF$name <- MissingLocSnowbase$Geo_Name[Index]
    if(is.na(LocSPDF$name[1])) {LocSPDF$name <- "UNKNOWN NAME"}
    LocSPDF$external_id <- MissingLocSnowbase$Geo_Id[Index]
    LocSPDF$comment <- "Placeholder geometry"
    
    ## Assigning new UUID
    uuid <- createUuid()
    rownames(LocSPDF@data) <- uuid
    LocSPDF$uuid <- uuid
    LocSPDF@polygons[[1]]@ID <- uuid
    
    ## Creating output SPDF
    if(Index>1) {
      OutputSPDF <- spRbind(OutputSPDF, LocSPDF)
    } else {
      OutputSPDF <- LocSPDF
    }
    
  }
  
  Test <<- OutputSPDF
  
  ## Writing to DB
  writeSPDFToGPSDB(OutputSPDF, Operation = Operation, Schema = "loccat", DBTbl = TargetDBTbl, DBType=DBType, Verbose=Verbose)

}