#' Creates operation and user account in terrain survey.
#'
#' Creates operation and user account in terrain survey.
#' @param OpName Name of operation
#' @param OpAbbrev Abbreviation of operation (default is empty string)
#' @param UserName Name of user
#' @param UserEmail Email address of user
#' @param UserPassword Password for user
#' @param Line1Label General label for Line 1 (default is Line 1)
#' @param Line2Label General label for Line 2 (default is Line 2)
#' @param Line3Label General label for Line 3 (default is Line 3)
#' @param SunCrustDate Default date for suncrust question (default is April 1)
#' @export
#' 
#' @examples 
#' require(SarpGPSTools)
#' require(SarpGPSToolsPrivate)
#' 
#' ## Creating operation first first user
#' createOperUserInTerrainSurveyDB(OpName="Test", UserName = "Test", UserEmail = "test@test.com", UserPassword = "test")
#' 
#' ## Adding additional user to operation
#' createOperUserInTerrainSurveyDB(OpName="Test", UserName = "Test", UserEmail = "test2@test.com", UserPassword = "test")

createOperUserInTerrainSurveyDB <- function(OpName, OpAbbrev="", UserName, UserEmail, UserPassword, Line1Label="Line 1", Line2Label="Line 2", Line3Label="Line 3", SunCrustDate="April 1") {
  

  ## OPERATION
  ## *********
  
  ## Check whether operation already exists
  Query <- paste0("SELECT id FROM operations WHERE name='", OpName, "'")
  OpId <- getRecordsFromQueryTerrainSurvey(Query)$id
  
  if (length(OpId)>0) {
    
    print(paste0("Operation ", OpName, " already exists (id=", OpId, ")."))
    NewOperation <- FALSE
    
  } else {
  
    ## Add operation and 
    Query <- paste0("INSERT INTO operations (name, abbrev, date_suncrust, label_line1, label_line2, label_line3, dtcre) ",
                    "VALUES ('", OpName, "', '", OpAbbrev, "', '", SunCrustDate, "', '", 
                                 Line1Label, "', '", Line2Label, "', '", Line3Label, "', '", as.character(Sys.time()),"');")
    
    sendQueryToTerrainSurveyDB(Query)
    Query <- paste0("SELECT id FROM operations WHERE name='", OpName, "'")
    OpId <- getRecordsFromQueryTerrainSurvey(Query)$id
    NewOperation <- TRUE
    print(paste0("Operation ", OpName, "has been added (id=", OpId, ")."))
    
  }
  
  ## USER
  ## ****
  
  ## Check whether user already exists
  Query <- paste0("SELECT id FROM users WHERE username='", UserEmail, "'")
  UserId <- getRecordsFromQueryTerrainSurvey(Query)$id
  
  if (length(UserId)>0) {
    
    print(paste0("User with username ", UserEmail, " already exists (id=", UserId, ")."))
    NewUser <- FALSE
    
  } else {
   
    ## Add operation and 
    Query <- paste0("INSERT INTO users (operation_id, name, username, password, dtcre) ",
                    "VALUES (", OpId, ", '", UserName, "', '", UserEmail, "', '", 
                    UserPassword, "', '", as.character(Sys.time()),"');")
    
    sendQueryToTerrainSurveyDB(Query)
    Query <- paste0("SELECT id FROM users WHERE username='", UserEmail, "'")
    UserId <- getRecordsFromQueryTerrainSurvey(Query)$id
    print(paste0("User with username ", UserEmail, " has been added (id=", UserId, ")."))
    NewUser <- TRUE
     
  }
  
  
  ## ADD TEST RUN CLUSTER AND TEST RUNS
  ## **********************************
  
  if (NewOperation) {
    
    print("Adding Test Run Cluster and two Test Runs ...")
    
    ## Cluster
    ## *******
    Query <- paste0("INSERT INTO runclusters (operation_id, `name`, `order`, `show`, dtcre) ",
                   "VALUES (", OpId, ", 'Test Run Cluster', 1, 1, '", as.character(Sys.time()),"');")
    
    sendQueryToTerrainSurveyDB(Query)
    
    Query <- paste0("SELECT id FROM runclusters WHERE operation_id=", OpId, " AND `name`='Test Run Cluster'")
    ClusterId <- getRecordsFromQueryTerrainSurvey(Query)$id
    
    print(paste0("New cluster called 'Test Run Cluster' created (id=", ClusterId, ")"))
    
    ## Test run 1 
    ## **********
    Query <- paste0("INSERT INTO runs (runcluster_id, `name`, `show`, photo, line1_incl, line1_name, line1_alp, line1_tl, line1_btl, dtcre) ",
                    "VALUES (", ClusterId, ", 'Test Run 1', 1, 'RunPhotos_Test/Test_Run01_WithLine.png', 1, '", Line1Label, "', 1, 1, 1, '", as.character(Sys.time()),"');")
    
    sendQueryToTerrainSurveyDB(Query)
    
    Query <- paste0("SELECT id FROM runs WHERE runcluster_id=", ClusterId, " AND `name`='Test Run 1'")
    Run1Id <- getRecordsFromQueryTerrainSurvey(Query)$id
    
    print(paste0("New run called 'Test Run 1' created (id=", Run1Id, ")"))
    
    ## Test run 2 
    ## **********
    Query <- paste0("INSERT INTO runs (runcluster_id, `name`, `show`, photo, line1_incl, line1_name, line1_alp, line1_tl, line1_btl, dtcre) ",
                    "VALUES (", ClusterId, ", 'Test Run 2', 1, 'RunPhotos_Test/Test_Run02_WithLine.png', 1, '", Line1Label, "', 0, 1, 1, '", as.character(Sys.time()),"');")
    
    sendQueryToTerrainSurveyDB(Query)
    
    Query <- paste0("SELECT id FROM runs WHERE runcluster_id=", ClusterId, " AND `name`='Test Run 2'")
    Run2Id <- getRecordsFromQueryTerrainSurvey(Query)$id
    
    print(paste0("New run called 'Test Run 2' created (id=", Run2Id, ")"))
    
  }
  
  
  ## ADD COMPLETED ASSIGNMENTS FOR TEST RUNS
  ## ***************************************
  
  if (NewUser) {
    
    ## Get cluster ID
    Query <- paste0("SELECT id FROM runclusters WHERE operation_id=", OpId, " AND `name`='Test Run Cluster'")
    ClusterId <- getRecordsFromQueryTerrainSurvey(Query)$id
    
    ## Test run 1
    ## **********
    
    print("Transferring test assessment for Test Run 1 (Alexis) ...")
    
    ## Get run id
    Query <- paste0("SELECT id FROM runs WHERE runcluster_id=", ClusterId, " AND `name`='Test Run 1'")
    Run1Id <- getRecordsFromQueryTerrainSurvey(Query)$id
    
    ## Get Roger's assessment of Alexis
    Query <- "SELECT * FROM assessments WHERE run_id=5 AND user_id=3"
    RunAssess <- getRecordsFromQueryTerrainSurvey(Query)
    
    ## Update assessment
    RunAssess$dtmod <- NA
    RunAssess$dtcre <- as.character(Sys.time())
    RunAssess$run_id <- Run1Id
    RunAssess$user_id <- UserId
    
    ## Create new run assessment id
    Query <- paste0("INSERT INTO assessments (run_id, user_id, dtcre) VALUES (", Run1Id, ", ", UserId, ", '", as.character(Sys.time()),"');")
    sendQueryToTerrainSurveyDB(Query)
    
    Query <- paste0("SELECT id FROM assessments WHERE run_id=", Run1Id, " AND user_id=", UserId, ";")
    Assess1Id <- getRecordsFromQueryTerrainSurvey(Query)$id
    
    ## Transferring test assessment
    for (ColIndex in 4:length(names(RunAssess))) {
      if(!is.na(RunAssess[1,ColIndex])) {
        if (class(RunAssess[1,ColIndex])=="integer") {
          Query <- paste0("UPDATE assessments SET `", names(RunAssess)[ColIndex], "`=", RunAssess[1,ColIndex], " WHERE id=", Assess1Id, ";")
        } else {
          Query <- paste0("UPDATE assessments SET `", names(RunAssess)[ColIndex], "`='", RunAssess[1,ColIndex], "' WHERE id=", Assess1Id, ";")
        }
        sendQueryToTerrainSurveyDB(Query)
      }
    }
    
    ## Test run 2
    ## **********
    
    print("Transferring test assessment for Test Run 2 (Bunker Hill) ...")
    
    ## Get run id
    Query <- paste0("SELECT id FROM runs WHERE runcluster_id=", ClusterId, " AND `name`='Test Run 2'")
    Run2Id <- getRecordsFromQueryTerrainSurvey(Query)$id
    
    ## Get Roger's assessment of Bunker Hill
    Query <- "SELECT * FROM assessments WHERE run_id=101 AND user_id=3"
    RunAssess <- getRecordsFromQueryTerrainSurvey(Query)
    
    ## Update assessment
    RunAssess$dtmod <- NA
    RunAssess$dtcre <- as.character(Sys.time())
    RunAssess$run_id <- Run2Id
    RunAssess$user_id <- UserId
    
    ## Create new run assessment id
    Query <- paste0("INSERT INTO assessments (run_id, user_id, dtcre) VALUES (", Run2Id, ", ", UserId, ", '", as.character(Sys.time()),"');")
    sendQueryToTerrainSurveyDB(Query)
    
    Query <- paste0("SELECT id FROM assessments WHERE run_id=", Run2Id, " AND user_id=", UserId, ";")
    Assess2Id <- getRecordsFromQueryTerrainSurvey(Query)$id
    
    ## Transferring test assessment
    for (ColIndex in 4:length(names(RunAssess))) {
      if(!is.na(RunAssess[1,ColIndex])) {
        if (class(RunAssess[1,ColIndex])=="integer") {
          Query <- paste0("UPDATE assessments SET `", names(RunAssess)[ColIndex], "`=", RunAssess[1,ColIndex], " WHERE id=", Assess2Id, ";")
        } else {
          Query <- paste0("UPDATE assessments SET `", names(RunAssess)[ColIndex], "`='", RunAssess[1,ColIndex], "' WHERE id=", Assess2Id, ";")
        }
        sendQueryToTerrainSurveyDB(Query)
      }
    }
    
  }
    
  
}