#' Creates new reference for raster and array table for extracted values in DB
#'
#' Creates new reference for raster and array table for extracted values in DB.
#' @param Operation Name of Operation. Used for accessing PostgreSQL database.
#' @param RasterName Name of raster.
#' @param FileName File name of raster file. Typically a tif file.
#' @param Subfolder Name of the local subfolder there the tif file is located. If there are nested subfolder, please not that the slashes need to be FOUR (4!) backslashes.
#' @param GisArrayTblNameExt Name extension for table in gis schema. Default value is RasterName in all lowercase.
#' @param ValueType Numeric type for value array. Needs to be 'integer' or 'real'. Default value is 'integer'.
#' @param DBType Parameter for specifying which database type to create the raster reference and array table into. Values can be 'Main' (default), 'NodeJS' and 'Old'.
#' @param TimeDiffToServer Time difference to database server in hours. Default value is 0 hrs.
#' @param Verbose Switch for printing SQL query. Default value is FALSE.
#' @param SuppressPostgreSQLWarnings Switch for turning warnings off. Default value is TRUE.
#'
#' @export

createRasterRefAndArrayTbl <- function(Operation, RasterName, FileName, Subfolder, GisArrayTblNameExt=tolower(RasterName), ValueType="Integer", DBType="Main", TimeDiffToServer=0, Verbose=F, SuppressPostgreSQLWarnings=T) {

  if (tolower(ValueType)!="integer" & tolower(ValueType)!="real") {
    stop("The value of the parameter 'ValueType needs to be integer or real!")
  }

  GISTblName <- paste0("gpsruns_array_", GisArrayTblNameExt)

  ## Check whether reference entry and/or table already exist
  ExistingRef <- getRecordsFromQuery(Operation, paste0("SELECT * FROM misc.rasters WHERE Name='", RasterName, "';"), DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)
  ExistingTbl <- getRecordsFromQuery(Operation, paste0("SELECT table_name FROM information_schema.tables WHERE table_schema = 'gis' and table_name='", GISTblName, "';"), DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)

  if (nrow(ExistingRef)>0 & nrow(ExistingTbl)>0) {
    stop(paste0("Reference to raster '", RasterName, "' and table '", GISTblName, "' already exists in database!"))
  } else if (nrow(ExistingRef)>0) {
    stop(paste0("Reference to raster '", RasterName, "' already exists in misc.rasters!"))
  } else if (nrow(ExistingTbl)>0) {
    stop(paste0("Table '", GISTblName, "' already exists in gis schema!"))
  } else {

    ## Query for adding new record to misc.rasters
    Query_Ref <- paste0("INSERT INTO misc.rasters (name, file_name, subfolder, gis_array_name_ext, dtcre) VALUES (",
                        convertStringToSQL(RasterName), ", ",
                        convertStringToSQL(FileName), ", ",
                        convertStringToSQL(Subfolder), ", ",
                        convertStringToSQL(GisArrayTblNameExt), ", ",
                        convertDateTimeToSQL((Sys.time()+TimeDiffToServer*60*60)), ");")

    ## Queries for adding new table in gis schema
    Query_CreateTbl <- paste0("CREATE TABLE gis.", GISTblName, " (",
                              "gpsruns_uuid VARCHAR(36) PRIMARY KEY, ",
                              "buffer INTEGER, ",
                              "valuearray ", ValueType, "[], ",
                              "dtcre TIMESTAMP (0) WITH TIME ZONE, ",
                              "dtmod TIMESTAMP (0) WITH TIME ZONE DEFAULT CURRENT_TIMESTAMP);")

    Query_AddComment <- paste0("COMMENT ON TABLE gis.", GISTblName, " IS '", RasterName, " score extracted from raster along run tracks';")

    Query_ForeignKey <- paste0("ALTER TABLE gis.", GISTblName, " ADD CONSTRAINT gpsruns_array_overheadhzd_fkey_gpsruns FOREIGN KEY (gpsruns_uuid) REFERENCES gps.runs(uuid) ON UPDATE CASCADE ON DELETE RESTRICT;")

    ## Submitting queries to database
    sendQueryToGPSDB(Operation, Query_Ref, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)
    sendQueryToGPSDB(Operation, Query_CreateTbl, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)
    sendQueryToGPSDB(Operation, Query_AddComment, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)
    sendQueryToGPSDB(Operation, Query_ForeignKey, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)

  }

}
