#' Creates view of GPS runs from list of UUID.
#'
#' Creates view of GPS runs from list of UUID.
#' @param Operation Name of operation.
#' @param User Name of user.
#' @param Schema Name of schema where view should be created.
#' @param ViewName Name of view to be created.
#' @param ViewDescription Description for view to be created.
#' @param UUID Single UUID or array of UUIDs.
#' @param DBType Parameter for specifying which database type tp create the view in. Values can be 'Main' (default), 'NodeJS' or 'Old'.
#' @param OnlyQCRuns Switch to specify whether to process only runs that are quality controlled (default) or all runs. Default value is TRUE.
#' @param Overwrite Switch for specifying whether an existing view of the same name should be overwritten.
#' @param Verbose Switch for printing of query
#' @param SuppressPostgreSQLWarnings Switch for suppressing warning messages from postgresqlExecStatement. Turn on for debugging!
#' @export

createRunViewFromUUID <- function(Operation, User, Schema, ViewName, ViewDescription, UUID, DBType="Main", OnlyQCRuns=T, Overwrite=F, Verbose=F, SuppressPostgreSQLWarnings=T) {

  ## Turn schema and view name to all lower case
  ## *******************************************
  Schema <- tolower(Schema)
  ViewName <- tolower(ViewName)


  ## Check list of existing views
  ## ****************************
  Query <- paste0("SELECT pg_class.relname as name ",
                  "FROM pg_class INNER JOIN information_schema.tables ON pg_class.relname=information_schema.tables.table_name ",
                  "WHERE information_schema.tables.table_schema = '", Schema, "' and information_schema.tables.table_type='VIEW' ",
                  "ORDER BY information_schema.tables.table_name")

  ListViews <- getRecordsFromQuery(Operation, Query, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)$name


  ## If it is a view that cannot be deleted
  ## **************************************
  if ((ViewName=="qc_runs" & Schema=="gps") | (ViewName=="qc_runs_pausepoints" & Schema=="gps")) {
    stop(paste0("The view '", ViewName, "' is a system view in the schema '", Schema, "' that cannot be deleted or overwritten!"))
  }


  ## If view already exists
  ## **********************
  if (ViewName %in% ListViews) {
    if (Overwrite) {
      deleteViews(Operation, Schema, ViewName, ShowList=F, DBType=DBType)
    } else {
      stop(paste0("The view '", ViewName, "' already exists in the schema '", Schema, "'!"))
    }
  }


  ## Creating new view
  ## *****************

  ## Build query - beginning
  if(OnlyQCRuns) {
    Query <- paste0("CREATE VIEW ", Schema, ".", ViewName, " AS SELECT * FROM gps.qc_runs")
  } else {
    Query <- paste0("CREATE VIEW ", Schema, ".", ViewName, " AS SELECT * FROM gps.runs")
  }

  ## Build query - UUID selection
  if(length(UUID)==1) {
    if(!is.na(UUID)) {
      Query <- paste0(Query, " WHERE (uuid=", convertStringToSQL(UUID), ")")
    }
  } else {
    Query <- paste0(Query, " WHERE (")
    for (i in 1:length(UUID)) {
      Query <- paste0(Query, "uuid=", convertStringToSQL(UUID[i]), " OR ")
    }
    Query <- paste0(substr(Query, 1, nchar(Query)-4), ")")
  }

  ## Submit create query to DB
  sendQueryToGPSDB(Operation, Query, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)

  ## Add description to view
  Query <- paste0("COMMENT ON VIEW ", Schema, ".", ViewName, " IS '", ViewDescription, " (created by ", User, " on ", Sys.Date(), ")';")
  sendQueryToGPSDB(Operation, Query, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)

  ## Feedback
  print(paste0("The view '", ViewName, "' successfully created in the schema '", Schema, "'."))

}
