#' Cut GPSRuns into elevation band segments
#'
#' Cuts GPS ski runs into elevation band segments and stores them in PostgreSQL/PostGIS GPS database. Only processes ski run tracks that have not been processed yet.
#' @param Operation Name of operation
#' @param User Name of user. Required for local root folder for location of raster.
#' @param DBType Parameter to specify which database type the elevant band segements the runs should be written to. The value can be 'Main' (default), 'NodeJS' or 'Old'.
#' @param Method Method for extraction of raster values. Must be 'simple' or 'bilinear'. See help for raster::extract for more detail.
#' @param Buffer size of buffer around GPS ski run in metres. Default value is 20 m. See help for raster::extract for more detail.
#' @param OnlyQCRuns Switch to specify whether to process only runs that are quality controlled (default) or all runs. Default value is TRUE
#' @param DateStart Start date if time period is specified in format 'YYYY-MM-DD'. Default value is NA.
#' @param DateEnd End date if time period is specified in format 'YYYY-MM-DD'. Default value is DateStart.
#' @param Overwrite Switch to specify whether to erase all records from the table before (re-) processing the runs. Default value id FALSE.
#' @param Test Only the first 10 runs are processed in Test mode. Default value is FALSE.
#' @param TimeDiffToDBServer Time difference to database server. Default value is 7 hrs (PST -> GMT)
#' @param UserConfirm Flag whether the user needs to actively confirm deletion. True by default.
#' @param Verbose Switch for printing of query. Default value is FALSE.
#' @param SuppressPostgreSQLWarnings Switch for turning warnings off. Default value is TRUE.
#'
#' @export

cutGPSRunsIntoElevBandSegments <- function(Operation, User, DBType="Main", Method="simple", Buffer=20, OnlyQCRuns=T, DateStart=NA, DateEnd=DateStart, Overwrite=F, Test=F, TimeDiffToDBServer=7, UserConfirm=T, Verbose=F, SuppressPostgreSQLWarnings=T) {

  ## Validating input parameters
  ## ***************************

  ## Methods
  if (Method!="simple" & Method!="bilinear") {
    stop("Value of Method parameter must be 'simple' or 'bilinear'! See help for raster::extract for more detail.")
  }


  ## Getting GPS runs
  ## ****************

  ## Delete existing records if requested
  if (Overwrite) {
    NumSegm <- getRecordsFromQuery(Operation, "Select count(uuid) FROM gis.gpsruns_segments_elevband", DBType=DBType)
    if (NumSegm>0) {
      if (UserConfirm==F) {
        UserResponse <- "Yes"
      } else {
        UserResponse <- readline(paste0("Type 'Yes' to confirm deletion of ", NumSegm, " existing records:"))
      }

      ## Deletion in response to user confirmation
      if (UserResponse=="Yes") {
        sendQueryToGPSDB(Operation, "DELETE FROM gis.gpsruns_segments_elevband", DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)
      }
    }
  }

  ## Retrieve UUIDs of runs to be processed
  if (OnlyQCRuns) {
    Query <- "SELECT gps.qc_runs.uuid FROM gps.qc_runs LEFT JOIN gis.gpsruns_segments_elevband ON gis.gpsruns_segments_elevband.gpsruns_uuid = gps.qc_runs.uuid WHERE (gis.gpsruns_segments_elevband.uuid IS NULL)"
  } else {
    Query <- "SELECT gps.runs.uuid FROM gps.runs LEFT JOIN gis.gpsruns_segments_elevband ON gis.gpsruns_segments_elevband.gpsruns_uuid = gps.runs.uuid WHERE (gis.gpsruns_segments_elevband.uuid IS NULL)"
  }
  
  if (!is.na(DateStart)) {
    Query <- paste0(Query, " AND (date_local >= '", DateStart, "' AND date_local <= '", DateEnd, "')") 
  }

  GPSRuns <- getRecordsFromQuery(Operation, Query, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)
  NumRuns <- nrow(GPSRuns)
  
  if(NumRuns==0) {
    
    warning("No runs need to be processed!")
    
  } else {
    
    ## Determining number of GPS runs to be retrieved
    RunIndexMax <- ifelse(Test, min(10, NumRuns), NumRuns)
    print(paste0("Getting ", RunIndexMax, " GPS run tracks ..."))
  
    ## Retrieving GPS runs
    SPDF_GPSRuns <- getSpatialObjectsDataFrameFromUUID(Operation, "gps", "runs", UUID = GPSRuns$uuid[1:RunIndexMax], DBType=DBType)
    SPDF_GPSRuns@proj4string <- CRS(Const_GlobalCrsString)
  
  
    ## Getting elevation band boundaries
    ## *********************************
    print("Getting elevation band boundaries ....")
    SPDF_ElevB <- getSpatialObjectsDataFrameFromUUID(Operation, "dem", "elevbands_boundaries", DBType=DBType)
    SPDF_ElevB@proj4string <- CRS(Const_GlobalCrsString)
    
    
    ## Getting elevation information
    ## *****************************
    print("Getting elevation raster information ....")
    Raster_Elev <- getRaster(Operation, User, "Elevation")
  
    ElevBoundary_TlAlp <- getElevBandBoundary(Operation, "TL")
    ElevBoundary_BtlTl <- getElevBandBoundary(Operation, "BTL")
  
  
    ## Reduce SPDF_GPSRuns for testing
    ## *******************************
    for (Index_Run in 1:nrow(SPDF_GPSRuns)) {
  
      # Index_Run <- 10
  
      print("******")
      print(paste(Sys.time(), ": Processing run", Index_Run, "of", nrow(SPDF_GPSRuns), "..."))
  
      ## Extract Run UUID
      RunUUID <- SPDF_GPSRuns$uuid[Index_Run]
  
      ## Extract Run Line
      RunLine <- as.SpatialLines.SLDF(SPDF_GPSRuns[Index_Run,])
      RunLine@proj4string <- CRS(Const_GlobalCrsString)
  
      ## Cut Run Line
      Segments <- gDifference(RunLine, SPDF_ElevB)
  
      ## Determine number of segments
      SegNum <- length(Segments@lines[[1]]@Lines)
  
      ## Extract segments and add to SPDF
      for (Index_Seg in 1:SegNum) {
  
        SegUUID  <- createUuid()
        SegLine  <- Segments@lines[[1]]@Lines[[Index_Seg]]
        SegLines <- Lines(list(SegLine), ID=SegUUID)
  
        SegSPLines <- SpatialLines(list(SegLines), proj4string = CRS(Const_GlobalCrsString))
        SegSPDF <- SpatialLinesDataFrame(SegSPLines, data.frame(uuid=SegUUID, gpsruns_uuid=RunUUID, row.names=SegUUID))
  
        if(exists("SPDF_ElevSegments")) {
          SPDF_ElevSegments <- rbind(SPDF_ElevSegments, SegSPDF)
        } else {
          SPDF_ElevSegments <- SegSPDF
        }
  
      }
      rm(Index_Seg)
  
      ## Clean up
      rm(SegLine, SegLines, Segments, SegNum, SegSPDF, SegSPLines, SegUUID)
  
  
      ## Determining elevation band
      ## **************************
  
      print("Determining elevation for segments ....")
  
      ## Calculate elevation statistics
      for (Index_Seg in 1:nrow(SPDF_ElevSegments)) {
  
        SegLines <- as.SpatialLines.SLDF(SPDF_ElevSegments[Index_Seg,])
  
        ElevArray <- round(unlist(extract(Raster_Elev, SegLines, buffer=Buffer)), 0)
  
        SPDF_ElevSegments$elev_avg[Index_Seg] <- round(mean(ElevArray),0)
        SPDF_ElevSegments$elev_min[Index_Seg] <- min(ElevArray)
        SPDF_ElevSegments$elev_max[Index_Seg] <- max(ElevArray)
  
      }
      rm(Index_Seg, SegLines)
  
      ## Assign elevation bands
      SPDF_ElevSegments$elev_band[SPDF_ElevSegments$elev_avg <= ElevBoundary_BtlTl] <- "BTL"
      SPDF_ElevSegments$elev_band[SPDF_ElevSegments$elev_avg >  ElevBoundary_BtlTl & SPDF_ElevSegments$elev_avg <= ElevBoundary_TlAlp] <- "TL"
      SPDF_ElevSegments$elev_band[SPDF_ElevSegments$elev_avg >  ElevBoundary_TlAlp] <- "ALP"
  
      ## Calculating length and eliminate segments shorter than the minimal length
      SPDF_ElevSegments$length_m <- round(SpatialLinesLengths(SPDF_ElevSegments, longlat=T)*1000, 0)
  
      ## Writing to DB
      ## *************
  
      SegNum <- nrow(SPDF_ElevSegments)
      print(paste("Writing", SegNum, "segments to DB..."))
  
      if (SegNum>0) {
        writeSPDFToGPSDB(SPDF_ElevSegments, Operation, "gis", "gpsruns_segments_elevband", ShowCounter=0, DBType=DBType)
      }
  
      rm(SPDF_ElevSegments)
  
    } ## End of run loop

  } ## End of if-condition NumRuns==0
    
}
