#' Deletes all GPS data in PostgreSQL/PostGIS GPS database
#'
#' Deletes all GPS data in PostgreSQL/PostGIS GPS database. Returns dataframe with number of records deleted.
#' @param Operation Name of operation.
#' @param DBType Parameter for specifying for which database type to delete the gps data from. Values can be 'Main' (default), 'NodeJS' or 'Old'.
#' @param UserConfirm Flag whether the user needs to actively confirm deletion. True by default.
#' @param Verbose Switch for printing of query. Default value is FALSE.
#' @param SuppressPostgreSQLWarnings Switch for turning warnings off Default value is TRUE.
#' @export

deleteAllGPSData <- function(Operation, DBType="Main", UserConfirm=T, Verbose=F, SuppressPostgreSQLWarnings=T) {

  DBType <- toupper(DBType)
  
  ## Count records before delete
  print(paste0("Database name: ", getGPSDBName(Operation, DBType=DBType)))
  
  DF_CountBefore <- countGPSRecords(Operation, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)
  names(DF_CountBefore) <- c("Table", "Before")

  ## User confirmation
  if (UserConfirm==F) {
      UserResponse <- "Yes"
  } else {
      print(DF_CountBefore)
      UserResponse <- readline("Type 'Yes' to confirm deletion: ")
  }

  ## Deletion in response to user confirmation
  if (UserResponse=="Yes") {

    ## 1) Delete obspoints
    if (DBType!="NODEJS") {
      Query <- paste0("DELETE FROM gps.obspoints")
      sendQueryToGPSDB(Operation, Query, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)
    }

    ## 2) Delete runs_pausepoints
    Query <- paste0("DELETE FROM gps.runs_pausepoints")
    sendQueryToGPSDB(Operation, Query, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)

    ## 3) Delete runs
    Query <- paste0("DELETE FROM gps.runs")
    sendQueryToGPSDB(Operation, Query, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)

    ## 4) Delete potruns_skipped
    if (DBType!="NODEJS") {
      Query <- paste0("DELETE FROM gps.potruns_skipped")
      sendQueryToGPSDB(Operation, Query, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)
    }
    
    ## 5) Delete tracks_days
    Query <- paste0("DELETE FROM gps.tracks_days")
    sendQueryToGPSDB(Operation, Query, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)

    ## 6) Delete tracks
    Query <- paste0("DELETE FROM gps.tracks")
    sendQueryToGPSDB(Operation, Query, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)
    
    ## Count records after delete
    DF_CountAfter <- countGPSRecords(Operation, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)
    names(DF_CountAfter) <- c("Table", "After")

    ## Merge before and after table
    DF_Count <- merge(DF_CountBefore, DF_CountAfter)
    return(DF_Count)

  } else {

    warning("No records were deleted.", immediate. = T)
    return(NA)

  }

}
